const express = require('express');
const fs = require('fs');
const path = require('path');
const bodyParser = require('body-parser');
const { isDeviceConnected, sendSmsViaIntent, checkSmsSent, checkInboxForReply, adb } = require('./adbRunner');

const app = express();
app.use(bodyParser.json());

// Handle malformed JSON bodies gracefully (body-parser throws a SyntaxError)
app.use((err, req, res, next) => {
  if (err && err instanceof SyntaxError && err.status === 400 && 'body' in err) {
    // Return a helpful 400 response instead of letting the process crash
    console.error('Invalid JSON received:', err.message);
    return res.status(400).json({ error: 'Invalid JSON', message: err.message });
  }
  next(err);
});
// Allow CORS from local frontend during development
app.use((req, res, next) => {
  res.setHeader('Access-Control-Allow-Origin', '*');
  res.setHeader('Access-Control-Allow-Methods', 'GET,POST,OPTIONS');
  res.setHeader('Access-Control-Allow-Headers', 'Content-Type');
  if (req.method === 'OPTIONS') return res.sendStatus(204);
  next();
});

const QUEUE_FILE = path.join(__dirname, 'queue.json');
const LOG_FILE = path.join(__dirname, 'logs.txt');

let queue = [];
let logs = [];
let sseClients = [];

function broadcast(eventName, data) {
  const payload = typeof data === 'string' ? data : JSON.stringify(data);
  sseClients.forEach((res) => {
    try {
      res.write(`event: ${eventName}\n`);
      res.write(`data: ${payload}\n\n`);
    } catch (e) {
      // ignore
    }
  });
}

function log(line) {
  const t = `${new Date().toISOString()} - ${line}`;
  logs.push(t);
  fs.appendFileSync(LOG_FILE, t + '\n');
  console.log(t);
}

function loadQueue() {
  try {
    if (fs.existsSync(QUEUE_FILE)) {
      queue = JSON.parse(fs.readFileSync(QUEUE_FILE, 'utf8')) || [];
    }
  } catch (e) {
    queue = [];
  }
}

function saveQueue() {
  fs.writeFileSync(QUEUE_FILE, JSON.stringify(queue, null, 2));
}

loadQueue();

// Simple worker
let running = false;
async function workerLoop() {
  if (running) return;
  running = true;
  try {
    while (queue.length) {
      const item = queue[0];
      // Remove items that are already in a terminal state
      if (['sent','failed','confirmed','no_response'].includes(item.status)) {
        queue.shift();
        saveQueue();
        continue;
      }
      log(`Processing ${item.id} -> ${item.to}`);
      const connected = await isDeviceConnected();
      if (!connected) {
        log('No device connected. Waiting...');
        break; // wait for next tick
      }
      try {
        const sendResult = await sendSmsViaIntent(item.to, item.body);
        // If sendSmsViaIntent reports already_sent, skip active send and continue to confirmation
        if (sendResult && sendResult.status === 'already_sent') {
          log(`Item ${item.id} already appears in Sent; skipping send.`);
        }
        // give device time to record
        await new Promise(r => setTimeout(r, 2000));
        const ok = await checkSmsSent(item.to, item.body);
        if (ok) {
          item.status = 'sent';
          item.sentAt = new Date().toISOString();
          log(`Sent ${item.id} OK`);
          broadcast('queue:update', item);

          // Wait up to 40s for an inbox reply that confirms the bet
          const conf = await waitForConfirmation(item.to, item.body, 40000, 4000);
          if (conf && conf.confirmed) {
            item.status = 'confirmed';
            item.betId = conf.betId;
            item.confirmedAt = new Date().toISOString();
            log(`Confirmed ${item.id} with BetID ${item.betId}`);
            broadcast('queue:update', item);
          } else if (conf && conf.error) {
            // If we received a rejection or low funds response, mark as failed and include reason
            item.status = 'failed';
            item.errorCode = conf.error;
            item.errorReason = conf.reason;
            item.checkedAt = new Date().toISOString();
            log(`Failed ${item.id}: ${conf.error} ${conf.reason || ''}`);
            broadcast('queue:update', item);
          } else {
            item.status = 'no_response';
            item.checkedAt = new Date().toISOString();
            log(`No confirmation reply for ${item.id} after 40s`);
            broadcast('queue:update', item);
          }
        } else {
          item.retries = (item.retries || 0) + 1;
          log(`Not confirmed for ${item.id}; retries=${item.retries}`);
          broadcast('queue:update', item);
          if (item.retries >= 3) {
            item.status = 'failed';
            log(`Marked failed ${item.id}`);
            broadcast('queue:update', item);
          }
        }
      } catch (err) {
        item.retries = (item.retries || 0) + 1;
        log(`Error sending ${item.id}: ${err && err.message ? err.message : err}`);
        broadcast('queue:update', item);
        if (item.retries >= 3) {
          item.status = 'failed';
        }
      }
      saveQueue();
      // small delay between messages
      await new Promise(r => setTimeout(r, 1000));
    }
  } finally {
    running = false;
  }
}

// Schedule worker every 3s
setInterval(() => {
  workerLoop().catch(err => log('Worker error: ' + err));
}, 3000);

app.post('/send', async (req, res) => {
  const { to, body, meta } = req.body || {};
  if (!to || !body) return res.status(400).json({ error: 'to and body required' });
  try {
    if (await checkSmsSent(to, body)) {
      return res.json({ ok: false, message: 'already_sent' });
    }
  } catch (e) {
    // ignore check failures and continue to enqueue
  }
  const id = `sms_${Date.now()}`;
  const item = { id, to, body, meta: meta || {}, status: 'queued', retries: 0, createdAt: new Date().toISOString() };
  queue.push(item);
  saveQueue();
  log(`Enqueued ${id} -> ${to}`);
  // attempt immediate work
  workerLoop().catch(err => log('Worker start error: ' + err));
  // broadcast enqueue
  broadcast('queue:enqueue', item);
  res.json({ ok: true, id });
});

// SSE endpoint for live events
app.get('/events', (req, res) => {
  res.setHeader('Content-Type', 'text/event-stream');
  res.setHeader('Cache-Control', 'no-cache');
  res.setHeader('Connection', 'keep-alive');
  res.write('\n');
  sseClients.push(res);
  req.on('close', () => {
    sseClients = sseClients.filter(r => r !== res);
  });
});

app.get('/queue', (req, res) => {
  res.json({ queue });
});

app.get('/logs', (req, res) => {
  res.setHeader('Content-Type', 'text/plain');
  res.send(logs.join('\n'));
});

app.post('/clear', (req, res) => {
  queue = [];
  saveQueue();
  res.json({ ok: true });
});

app.get('/status', async (req, res) => {
  const connected = await isDeviceConnected();
  res.json({ connected, queueLength: queue.length });
});

app.post('/uninstall-helper', async (req, res) => {
  try {
    await adb('shell pm uninstall com.example.smshelper');
    res.json({ ok: true });
  } catch (e) {
    res.status(500).json({ error: String(e) });
  }
});

// One-shot auto-send: try to send immediately using adbRunner's sendSmsViaIntent
app.post('/auto-send', async (req, res) => {
  const { to, body } = req.body || {};
  if (!to || !body) return res.status(400).json({ error: 'to and body required' });
  try {
    log(`auto-send: attempting ${to}`);
    const sendResult = await sendSmsViaIntent(to, body);
    // give device a moment to update storage
    await new Promise(r => setTimeout(r, 1400));
    const confirmedSent = await checkSmsSent(to, body);
    let confirmation = { confirmed: false };
    if (confirmedSent) {
      confirmation = await waitForConfirmation(to, body, 40000, 4000);
    }
    res.json({ ok: true, to, bodySnippet: body.slice(0, 120), sendResult, sent: confirmedSent, confirmation });
  } catch (e) {
    // adb() may reject with an object { err, stdout, stderr } - stringify useful fields
    try {
      const details = {};
      if (e && typeof e === 'object') {
        if (e.message) details.message = e.message;
        if (e.err) details.err = (e.err && e.err.message) ? e.err.message : String(e.err);
        if (e.stdout) details.stdout = e.stdout;
        if (e.stderr) details.stderr = e.stderr;
      } else {
        details.message = String(e);
      }
      log('auto-send error: ' + JSON.stringify(details));
      res.status(500).json({ error: details });
    } catch (serr) {
      log('auto-send error (stringify failed): ' + String(e));
      res.status(500).json({ error: String(e) });
    }
  }
});

// Discover send button coordinates: open composer, dump UI, and return candidate nodes
app.get('/discover-send', async (req, res) => {
  const to = req.query.to;
  const body = req.query.body || '';
  if (!to) return res.status(400).json({ error: 'query param to is required' });
  try {
    log(`discover-send: opening composer to=${to}`);
    // Start composer intent (do not attempt to send) so UI is visible
    const escapedBody = body.replace(/"/g, '"');
    await adb(`shell am start -a android.intent.action.SENDTO -d sms:${to} --es sms_body "${escapedBody}"`);
    // give UI a moment
    await new Promise(r => setTimeout(r, 900));

    // get screen size
    let screenSize = null;
    try {
      const { stdout } = await adb('shell wm size');
      const overrideMatch = stdout.match(/Override size:\s*(\d+)x(\d+)/i);
      if (overrideMatch) screenSize = { w: parseInt(overrideMatch[1], 10), h: parseInt(overrideMatch[2], 10) };
      else {
        const physMatch = stdout.match(/Physical size:\s*(\d+)x(\d+)/i) || stdout.match(/(\d+)\s*[xX]\s*(\d+)/);
        if (physMatch) screenSize = { w: parseInt(physMatch[1], 10), h: parseInt(physMatch[2], 10) };
      }
    } catch (e) { /* ignore */ }

    const dumpName = `/sdcard/window_dump_${Date.now()}.xml`;
    await adb(`shell uiautomator dump ${dumpName}`);
    const localName = `./window_dump_${Date.now()}.xml`;
    // pull the most recent dump (the device path contains the timestamp we used)
    await adb(`pull ${dumpName} ${localName}`);
    const xml = require('fs').readFileSync(localName, 'utf8');

    // Parse node entries and collect candidates
    const nodeRegex = /<node ([^>]*?)\/>/ig;
    let match;
    const candidates = [];
    while ((match = nodeRegex.exec(xml))) {
      const attrs = match[1];
      const lower = attrs.toLowerCase();
      // Only consider nodes that mention send in resource-id/text/content-desc or have typical send-related labels
      if (/resource-id="[^"]*send[^"]*"/i.test(attrs) || /\b(text|content-desc)="[^"]*send[^"]*"/i.test(attrs) || /\bcontent-desc="[^"]*send[^"]*"/i.test(attrs) || /\bsend\b/i.test(lower)) {
        const resIdMatch = attrs.match(/resource-id="([^"]*)"/);
        const textMatch = attrs.match(/text="([^"]*)"/);
        const descMatch = attrs.match(/content-desc="([^"]*)"/);
        const boundsMatch = attrs.match(/bounds="\[?(\d+),(\d+)\]\[?(\d+),(\d+)\]"/);
        const item = { resourceId: resIdMatch ? resIdMatch[1] : null, text: textMatch ? textMatch[1] : null, contentDesc: descMatch ? descMatch[1] : null };
        if (boundsMatch) {
          const left = parseInt(boundsMatch[1], 10);
          const top = parseInt(boundsMatch[2], 10);
          const right = parseInt(boundsMatch[3], 10);
          const bottom = parseInt(boundsMatch[4], 10);
          item.bounds = { left, top, right, bottom };
          item.center = { x: Math.floor((left + right) / 2), y: Math.floor((top + bottom) / 2) };
        }
        candidates.push(item);
      }
    }

    res.json({ ok: true, to, body, screenSize, candidates, dumpFile: localName });
  } catch (err) {
    log('discover-send error: ' + (err && err.message ? err.message : String(err)));
    res.status(500).json({ error: String(err) });
  }
});

// One-shot: open composer, optionally discover send button, tap it (by coords or candidate index), and verify
app.post('/discover-and-tap', async (req, res) => {
  const to = req.body && req.body.to;
  const body = (req.body && req.body.body) || '';
  const candidateIndex = typeof (req.body && req.body.candidateIndex) === 'number' ? req.body.candidateIndex : 0;
  const explicitX = req.body && (typeof req.body.x === 'number' ? Math.floor(req.body.x) : null);
  const explicitY = req.body && (typeof req.body.y === 'number' ? Math.floor(req.body.y) : null);
  const waitMs = parseInt((req.body && req.body.waitMs) || '2000', 10) || 2000;
  if (!to) return res.status(400).json({ error: 'to is required' });
  try {
    log(`discover-and-tap: opening composer to=${to}`);
    const escapedBody = body.replace(/"/g, '\\"');
    await adb(`shell am start -a android.intent.action.SENDTO -d sms:${to} --es sms_body "${escapedBody}"`);
    await new Promise(r => setTimeout(r, 900));

    // Focus input to ensure keyboard is visible (helps send button placement)
    try {
      await adb('shell uiautomator dump /sdcard/window_dump_input_focus.xml');
      await adb('pull /sdcard/window_dump_input_focus.xml ./window_dump_input_focus.xml');
      const xmlIf = require('fs').readFileSync('./window_dump_input_focus.xml', 'utf8');
      const nodeRegexIf = /<node ([^>]*?)\/>/ig;
      let mIf;
      let inputFoundIf = null;
      while ((mIf = nodeRegexIf.exec(xmlIf))) {
        const attrsIf = mIf[1];
        const lowerIf = attrsIf.toLowerCase();
        if (/class="[^"]*(edit|edittext|text|input)[^"]*"/i.test(attrsIf) || /resource-id="[^"]*(edit|composer|message|input|text)[^"]*"/i.test(attrsIf) || /content-desc="[^"]*(composer|message|input|type)[^"]*"/i.test(attrsIf) || /\binputtype\b/i.test(lowerIf)) {
          const bIf = attrsIf.match(/bounds="\[?(\d+),(\d+)\]\[?(\d+),(\d+)\]"/);
          if (bIf) { inputFoundIf = { left: parseInt(bIf[1],10), top: parseInt(bIf[2],10), right: parseInt(bIf[3],10), bottom: parseInt(bIf[4],10) }; break; }
        }
      }
      if (inputFoundIf) {
        const cxIf = Math.floor((inputFoundIf.left + inputFoundIf.right)/2);
        const cyIf = Math.floor((inputFoundIf.top + inputFoundIf.bottom)/2);
        try { await adb(`shell input tap ${cxIf} ${cyIf}`); } catch (e) {}
        await new Promise(r => setTimeout(r, 700));
      }
    } catch (e) {
      // ignore
    }

    // If explicit coords provided, use them directly
    let tapped = null;
    let dumpLocal = null;
    let candidates = [];

    if (explicitX !== null && explicitY !== null) {
      await adb(`shell input tap ${explicitX} ${explicitY}`);
      tapped = { x: explicitX, y: explicitY, source: 'explicit' };
    } else {
      // Attempt to dump UI and find send-related nodes (reuse discovery heuristics)
      const dumpName = `/sdcard/window_dump_${Date.now()}.xml`;
      await adb(`shell uiautomator dump ${dumpName}`);
      const localName = `./window_dump_${Date.now()}.xml`;
      await adb(`pull ${dumpName} ${localName}`);
      dumpLocal = localName;
      const xml = require('fs').readFileSync(localName, 'utf8');
      const nodeRegex = /<node ([^>]*?)\/>/ig;
      let match;
      while ((match = nodeRegex.exec(xml))) {
        const attrs = match[1];
        if (/resource-id="[^"]*send[^"]*"/i.test(attrs) || /\b(text|content-desc)="[^"]*send[^"]*"/i.test(attrs) || /\bsend\b/i.test(attrs.toLowerCase())) {
          const resIdMatch = attrs.match(/resource-id="([^"]*)"/);
          const textMatch = attrs.match(/text="([^"]*)"/);
          const descMatch = attrs.match(/content-desc="([^"]*)"/);
          const boundsMatch = attrs.match(/bounds="\[?(\d+),(\d+)\]\[?(\d+),(\d+)\]"/);
          const item = { resourceId: resIdMatch ? resIdMatch[1] : null, text: textMatch ? textMatch[1] : null, contentDesc: descMatch ? descMatch[1] : null };
          if (boundsMatch) {
            const left = parseInt(boundsMatch[1], 10);
            const top = parseInt(boundsMatch[2], 10);
            const right = parseInt(boundsMatch[3], 10);
            const bottom = parseInt(boundsMatch[4], 10);
            item.bounds = { left, top, right, bottom };
            item.center = { x: Math.floor((left + right) / 2), y: Math.floor((top + bottom) / 2) };
          }
          candidates.push(item);
        }
      }

      if (candidates.length && candidates[candidateIndex]) {
        const c = candidates[candidateIndex];
        if (c.center) {
          await adb(`shell input tap ${c.center.x} ${c.center.y}`);
          tapped = { x: c.center.x, y: c.center.y, source: 'candidate', resourceId: c.resourceId };
        }
      }

      // If we still didn't tap, try conservative right-edge taps using wm size
      if (!tapped) {
        try {
          const { stdout } = await adb('shell wm size');
          let size = null;
          const overrideMatch = stdout.match(/Override size:\s*(\d+)x(\d+)/i);
          if (overrideMatch) size = { w: parseInt(overrideMatch[1], 10), h: parseInt(overrideMatch[2], 10) };
          else {
            const physMatch = stdout.match(/Physical size:\s*(\d+)x(\d+)/i) || stdout.match(/(\d+)\s*[xX]\s*(\d+)/);
            if (physMatch) size = { w: parseInt(physMatch[1], 10), h: parseInt(physMatch[2], 10) };
          }
          let w = 1080, h = 2400;
          if (size) { w = size.w; h = size.h; }
          const xs = [0.92, 0.88, 0.84].map(p => Math.max(40, Math.floor(w * p)));
          const y = Math.max(120, Math.floor(h * 0.92));
          for (const x of xs) {
            await adb(`shell input tap ${x} ${y}`);
            await new Promise(r => setTimeout(r, 800));
            // quick attempt to detect if tapped produced a send in sent table (best-effort)
            const sentOk = await checkSmsSent(to, body);
            if (sentOk) { tapped = { x, y, source: 'fallback' }; break; }
          }
        } catch (e) {
          // ignore
        }
      }
    }

    // wait and verify
    await new Promise(r => setTimeout(r, waitMs));
    const confirmed = await checkSmsSent(to, body);

    res.json({ ok: true, to, bodySnippet: body.slice(0,40), tapped, candidates, dumpFile: dumpLocal, confirmed });
  } catch (err) {
    log('discover-and-tap error: ' + (err && err.message ? err.message : String(err)));
    res.status(500).json({ error: String(err) });
  }
});

// Remote tap endpoint: POST { x, y }
app.post('/tap', async (req, res) => {
  const { x, y } = req.body || {};
  if (typeof x !== 'number' || typeof y !== 'number') return res.status(400).json({ error: 'x and y required as numbers' });
  try {
    await adb(`shell input tap ${Math.floor(x)} ${Math.floor(y)}`);
    res.json({ ok: true, x: Math.floor(x), y: Math.floor(y) });
  } catch (e) {
    res.status(500).json({ error: String(e) });
  }
});

// Query sent messages via the controller (useful when adb isn't available on the client shell)
app.get('/sent-query', async (req, res) => {
  const to = req.query.to || '';
  const body = req.query.body || '';
  const limit = parseInt(req.query.limit || '20', 10) || 20;
  try {
    let where = '';
    if (to) where += `address='${to}'`;
    if (body) {
      const snippet = body.slice(0, 40).replace(/'/g, "\\'");
      if (where) where += ' AND ';
      where += `body LIKE '%${snippet}%'`;
    }
    const wherePart = where ? ` --where "${where}" ` : ' ';
    const cmd = `shell content query --uri content://sms/sent --projection _id,address,body,date,creator --sort 'date DESC' --limit ${limit}${wherePart}`;
    const { stdout } = await adb(cmd);
    res.json({ ok: true, to, bodySnippet: body.slice(0, 60), limit, stdout });
  } catch (e) {
    res.status(500).json({ error: String(e) });
  }
});

// Periodic inbox scanner for simple low-funds detection (best-effort)
async function scanInbox() {
  try {
    // Query recent inbox messages (may be restricted on some devices)
    const { stdout } = await adb("shell content query --uri content://sms/inbox --projection 'address,body,date' -t 20");
    if (!stdout) return;
    const text = stdout.toLowerCase();
    const keywords = ['low fund', 'insufficient', 'low balance', 'insufficient funds', 'not enough'];
    for (const kw of keywords) {
      if (text.includes(kw)) {
        log(`Inbox keyword match: ${kw}`);
        broadcast('inbox:match', { keyword: kw, excerpt: text.substring(Math.max(0, text.indexOf(kw) - 40), text.indexOf(kw) + 140) });
        break;
      }
    }
  } catch (e) {
    // ignore permission errors
  }
}

setInterval(() => { scanInbox().catch(() => {}); }, 10000);

// Helper: poll inbox for confirmation pattern like 'Multi BetID 4304 confirmed'
async function waitForConfirmation(to, body, maxWaitMs = 40000, pollInterval = 4000) {
  const start = Date.now();
  while (Date.now() - start < maxWaitMs) {
    await new Promise(r => setTimeout(r, pollInterval));
    try {
      const inboxLines = await checkInboxForReply(to, body, 30);
      if (inboxLines && inboxLines.length) {
        const joined = inboxLines.join('\n');
        // Confirmed pattern
        const m = joined.match(/Multi\s*BetID\s*(\d+)\s*confirmed/i);
        if (m && m[1]) {
          return { confirmed: true, betId: m[1], raw: joined };
        }
        // Rejection / invalid patterns
        const rej = joined.match(/rejected(?: as)?[^\n]*\b(INVALID|invalid)\b.*|bet is rejected|your bet is rejected|market prediction [^\n]{0,40} INVALID/i);
        if (rej) {
          // pick a short reason excerpt
          const reason = (rej[0] || '').trim().slice(0, 180);
          return { confirmed: false, error: 'rejected', reason, raw: joined };
        }
        // Low funds / insufficient balance
        const low = joined.match(/low fund|insufficient|low balance|insufficient funds|not enough/i);
        if (low) {
          const reason = (low[0] || '').trim().slice(0, 180);
          return { confirmed: false, error: 'low_funds', reason, raw: joined };
        }
      }
    } catch (e) {
      // ignore transient errors
    }
  }
  return { confirmed: false };
}

const PORT = process.env.PORT || 5010;
app.listen(PORT, async () => {
  log(`SMS ADB controller listening on port ${PORT}`);
  console.log(`SMS ADB controller listening on port ${PORT}`);

  // Sanity-check adb availability and surface a helpful message if missing
  try {
    const { stdout } = await adb('version');
    const first = (stdout || '').split(/\r?\n/)[0] || '';
    log('adb available: ' + first);
  } catch (e) {
    try {
      const details = {};
      if (e && typeof e === 'object') {
        if (e.err && e.err.message) details.err = e.err.message;
        if (e.stdout) details.stdout = e.stdout;
        if (e.stderr) details.stderr = e.stderr;
      } else {
        details.message = String(e);
      }
      log('ADB check failed: ' + JSON.stringify(details));
    } catch (_) {
      log('ADB check failed: ' + String(e));
    }
    log("Hint: set the ADB_PATH environment variable to the full adb.exe path or add the platform-tools folder to your PATH.\nExample (PowerShell session): $env:ADB_PATH = 'C:\\android\\platform-tools\\adb.exe'\nOr permanently (PowerShell): setx ADB_PATH \"C:\\android\\platform-tools\\adb.exe\"\nThen restart this server.");
  }
});
