const { exec } = require('child_process');
const fs = require('fs');

function getAdbCommand() {
  // Allow overriding adb executable via environment variable ADB_PATH
  // Example: ADB_PATH="C:\\android\\platform-tools\\adb.exe"
  if (process.env.ADB_PATH && process.env.ADB_PATH.length) {
    // return raw path; quoting handled when building the full command
    return process.env.ADB_PATH;
  }
  return 'adb';
}

function adb(cmd) {
  const adbCmd = getAdbCommand();
  return new Promise((resolve, reject) => {
    // Build command safely: quote the adb path if it contains spaces
    const full = (adbCmd.indexOf(' ') >= 0) ? `"${adbCmd}" ${cmd}` : `${adbCmd} ${cmd}`;
    exec(full, { maxBuffer: 1024 * 1024 }, (err, stdout, stderr) => {
      if (err) return reject({ err, stdout, stderr });
      resolve({ stdout: stdout.trim(), stderr: stderr.trim() });
    });
  });
}

async function isDeviceConnected() {
  try {
    const { stdout } = await adb('devices -l');
    if (!stdout) return false;
    // Split into non-empty lines and parse lines that look like device entries.
    const lines = stdout.split(/\r?\n/).map(l => l.trim()).filter(Boolean);
    // Device entry lines typically look like: <serial>\t<state> ...
    const deviceLines = lines.filter(l => /^([^\s]+)\s+(device|unauthorized|offline|unknown)\b/i.test(l));
    if (!deviceLines.length) {
      // Debug: print adb devices output so server logs show why parsing failed
      try {
        console.log('adb devices output (no device lines matched):');
        console.log(stdout);
      } catch (e) {
        // ignore logging errors
      }
      return false;
    }
    // Consider connected only when at least one entry's state is exactly 'device'
    const connected = deviceLines.some(l => /\bdevice\b/i.test(l));
    return connected;
  } catch (e) {
    // Log error for debugging in server output
    try {
      console.error('isDeviceConnected error:', e && e.err ? e.err : e);
      if (e && e.stdout) console.error('adb stdout:', e.stdout);
      if (e && e.stderr) console.error('adb stderr:', e.stderr);
    } catch (ignore) {}
    return false;
  }
}

async function sendSmsViaIntent(to, body) {
  // Pre-send check: if this exact (or similar) message already exists in Sent, avoid re-sending
  try {
    if (await checkSmsSent(to, body)) {
      return { status: 'already_sent' };
    }
  } catch (e) {
    // ignore pre-check failures and continue to attempt send
  }
  // Prefer helper APK if installed, fallback to composer intent if not.
  // Helper: com.example.smshelper/.SendActivity
  async function isHelperInstalled() {
    try {
      const { stdout } = await adb('shell pm list packages com.example.smshelper');
      return stdout && stdout.includes('com.example.smshelper');
    } catch (e) { return false; }
  }

  if (await isHelperInstalled()) {
    // Use helper APK
    const escapedBody = body.replace(/"/g, '"');
    const cmd = `shell am start -n com.example.smshelper/.SendActivity --es to "${to}" --es body "${escapedBody}"`;
    await adb(cmd);
    // Quick logcat scan for helper confirmation
    try {
      const { stdout: logs } = await adb('logcat -d -t 40 SMS_HELPER:I');
      if (logs && logs.includes('SENT to=' + to)) {
        return { status: 'sent', confirmed: true };
      }
    } catch (e) {}
    // After attempting helper, poll Sent for a short period to confirm
    const maxPolls = 6; // total ~6 seconds
    for (let i = 0; i < maxPolls; i++) {
      await new Promise(r => setTimeout(r, 1000));
      try {
        if (await checkSmsSent(to, body)) {
          return { status: 'sent', confirmed: true };
        }
      } catch (e) {
        // ignore and continue polling
      }
    }
    // not confirmed after retries
    return { status: 'unknown', confirmed: false };
  }

  // Fallback: original composer intent + tap/keyevent sequence
  // Try inserting a draft directly (less flaky than clipboard/paste on some OEM apps)
  async function tryInsertDraft(toAddr, bodyText) {
    try {
      // Bindings: address and body
      const b = bodyText.replace(/"/g, '\\"');
      const cmd = `shell content insert --uri content://sms/draft --bind address:s:'${toAddr}' --bind body:s:'${b}'`;
  await adb(cmd);
      // Some devices return a result, but even empty stdout may be fine; assume success if no error thrown
      return true;
    } catch (e) {
      return false;
    }
  }

  // Helper: check if a messaging composer-like window is currently focused
  async function isComposerFocused() {
    try {
      const { stdout } = await adb('shell dumpsys window windows');
      if (!stdout) return false;
      const focusBlock = stdout.split(/\r?\n/).slice(0, 300).join('\n');
      // Look for signs of SMS/Messaging composer in window titles or focus lines
      if (/sms|mms|compose|conversation|messag/i.test(focusBlock)) return true;
    } catch (e) {}
    return false;
  }

  // Insert draft where supported
  await tryInsertDraft(to, body).catch(() => {});

  const escapedBody = body.replace(/"/g, '"');
  const cmd = `shell am start -a android.intent.action.SENDTO -d sms:${to} --es sms_body "${escapedBody}"`;
  await adb(cmd);
  await new Promise(r => setTimeout(r, 1600));

  // Try to focus the message input to force the soft keyboard to appear.
  try {
    await adb('shell uiautomator dump /sdcard/window_dump_input.xml');
    await adb('pull /sdcard/window_dump_input.xml ./window_dump_input.xml');
    const xmlInput = fs.readFileSync('./window_dump_input.xml', 'utf8');
    const nodeRegexInput = /<node ([^>]*?)\/>/ig;
    let m;
    let inputFound = null;
    while ((m = nodeRegexInput.exec(xmlInput))) {
      const attrs = m[1];
      const lower = attrs.toLowerCase();
      // match typical input/edit text nodes
      if (/class="[^"]*(edit|edittext|text|input)[^"]*"/i.test(attrs) || /resource-id="[^"]*(edit|composer|message|input|text)[^"]*"/i.test(attrs) || /content-desc="[^"]*(composer|message|input|type)[^"]*"/i.test(attrs) || /\binputtype\b/i.test(lower)) {
        const b = attrs.match(/bounds="\[?(\d+),(\d+)\]\[?(\d+),(\d+)\]"/);
        if (b) {
          inputFound = { left: parseInt(b[1], 10), top: parseInt(b[2], 10), right: parseInt(b[3], 10), bottom: parseInt(b[4], 10) };
          break;
        }
      }
    }
    if (inputFound) {
      const cx = Math.floor((inputFound.left + inputFound.right) / 2);
      const cy = Math.floor((inputFound.top + inputFound.bottom) / 2);
      try { await adb(`shell input tap ${cx} ${cy}`); } catch (e) {}
      await new Promise(r => setTimeout(r, 700));
    }
  } catch (e) {
    // ignore failures in focusing input
  }
  // Attempt typing the text directly into the focused input as a fallback.
  try {
    // Escape characters for 'input text' (adb interprets % specially)
    let forInput = body.replace(/%/g, '%25').replace(/ /g, '%s').replace(/#/g, '%23').replace(/"/g, '\\"');
    // Surround with double quotes so shell passes it as one arg
    await adb(`shell input text "${forInput}"`);
    await new Promise(r => setTimeout(r, 700));
  } catch (e) {
    // ignore typing failures
  }
  // If the composer did not stay focused (some apps immediately navigate back), retry open+focus sequence
  if (!(await isComposerFocused())) {
    // retry up to 3 times: reopen composer and focus
    let reopened = false;
    for (let retry = 0; retry < 3; retry++) {
      try {
        await adb(cmd);
        await new Promise(r => setTimeout(r, 900 + retry * 300));
        // dump and attempt to focus again
        await adb('shell uiautomator dump /sdcard/window_dump_input.xml');
        await adb('pull /sdcard/window_dump_input.xml ./window_dump_input.xml');
        const xmlInput2 = fs.readFileSync('./window_dump_input.xml', 'utf8');
        const nodeRegexInput2 = /<node ([^>]*?)\/>/ig;
        let m2;
        let inputFound2 = null;
        while ((m2 = nodeRegexInput2.exec(xmlInput2))) {
          const attrs = m2[1];
          if (/class="(.*(edit|edittext|text|input).*?)"/i.test(attrs) || /resource-id="(.*(edit|composer|message|input|text).*?)"/i.test(attrs)) {
            const b = attrs.match(/bounds="\[?(\d+),(\d+)\]\[?(\d+),(\d+)\]"/);
            if (b) { inputFound2 = { left: parseInt(b[1], 10), top: parseInt(b[2], 10), right: parseInt(b[3], 10), bottom: parseInt(b[4], 10) }; break; }
          }
        }
        if (inputFound2) {
          const cx2 = Math.floor((inputFound2.left + inputFound2.right) / 2);
          const cy2 = Math.floor((inputFound2.top + inputFound2.bottom) / 2);
          try { await adb(`shell input tap ${cx2} ${cy2}`); } catch (e) {}
          await new Promise(r => setTimeout(r, 700));
        }
        if (await isComposerFocused()) { reopened = true; break; }
      } catch (e) {}
    }
    if (!reopened) {
      // continue — we'll try Send discovery and taps anyway
    }
  }
  if (!(await isComposerFocused())) {
    // Give up early: composer isn't staying foreground; returning explicit status so higher layers can surface the error
    return { status: 'composer_not_focused', error: 'Composer did not remain in foreground' };
  }
  // Helper: read the composer body from the latest uiautomator dump if present
  async function readComposerBody() {
    try {
      await adb('shell uiautomator dump /sdcard/window_dump_composer.xml');
      await adb('pull /sdcard/window_dump_composer.xml ./window_dump_composer.xml');
      const xml = fs.readFileSync('./window_dump_composer.xml', 'utf8');
      const nodeRegex = /<node ([^>]*?)\/>/ig;
      let match;
      while ((match = nodeRegex.exec(xml))) {
        const attrs = match[1];
        // look for nodes that have a text attribute (composer fields often expose 'text')
        const textMatch = attrs.match(/text="([^"]*)"/);
        const resId = attrs.match(/resource-id="([^"]*)"/);
        if (textMatch) {
          const txt = textMatch[1];
          // Heuristic: composer fields often have resource-id or hint containing 'composer','message','input'
          if (resId && /composer|message|input|edit|body|sms/i.test(resId[1])) return txt;
        }
      }
    } catch (e) {
      // ignore
    }
    return null;
  }
  async function getScreenSize() {
    try {
      const { stdout } = await adb('shell wm size');
      // `wm size` often returns two lines: "Physical size: 1440x3200" and
      // optionally "Override size: 1080x2400" when display override is in
      // effect (we prefer the Override size which maps to what apps see).
      // Try to pick the Override size first, then Physical size.
      const overrideMatch = stdout.match(/Override size:\s*(\d+)x(\d+)/i);
      if (overrideMatch) return { w: parseInt(overrideMatch[1], 10), h: parseInt(overrideMatch[2], 10) };
      const physMatch = stdout.match(/Physical size:\s*(\d+)x(\d+)/i) || stdout.match(/(\d+)\s*[xX]\s*(\d+)/);
      if (physMatch) return { w: parseInt(physMatch[1], 10), h: parseInt(physMatch[2], 10) };
      const n = stdout.replace(/[^0-9xX]/g, '');
      const parts = n.split(/[xX]/);
      if (parts.length === 2) return { w: parseInt(parts[0], 10), h: parseInt(parts[1], 10) };
    } catch (e) {}
    return null;
  }
  const size = await getScreenSize();
  const taps = [];
  if (size) {
    const { w, h } = size;
    taps.push([Math.max(60, w - 150), Math.max(120, h - 200)]);
    taps.push([Math.max(60, w - 100), Math.max(120, h - 160)]);
    taps.push([Math.max(60, w - 60), Math.max(120, h - 120)]);
  } else {
    taps.push([900, 1700]);
  }
  for (const [x, y] of taps) {
    try { await adb(`shell input tap ${x} ${y}`); } catch (e) {}
    await new Promise(r => setTimeout(r, 800));
  }
  await new Promise(r => setTimeout(r, 1200));

  // Try more robust UIAutomator approach: dump view hierarchy and look for a Send button
  try {
    // Dump UI hierarchy to device and pull it
    await adb('shell uiautomator dump /sdcard/window_dump.xml');
    await adb('pull /sdcard/window_dump.xml ./window_dump.xml');
    const fs = require('fs');
    const xml = fs.readFileSync('./window_dump.xml', 'utf8');
    // Parse XML and find a node whose text/content-desc contains "send"
    // We'll try to find the closest matching node and extract its bounds.
    const nodeRegex = /<node ([^>]*?)\/>/ig;
    let match;
    let found = null;
    while ((match = nodeRegex.exec(xml))) {
      const attrs = match[1];
      // Prefer explicit resource-id matches (common on Samsung Messages)
      if (/resource-id="[^"]*send_button[^"]*"/i.test(attrs)) {
        const b = attrs.match(/bounds="\[?(\d+),(\d+)\]\[?(\d+),(\d+)\]"/);
        if (b) {
          found = { left: parseInt(b[1], 10), top: parseInt(b[2], 10), right: parseInt(b[3], 10), bottom: parseInt(b[4], 10) };
          break;
        }
      }
      // Fallback: match visible text or content-desc containing "send"
      if (/\b(text|content-desc)="[^"]*send[^"]*"/i.test(attrs)) {
        const b = attrs.match(/bounds="\[?(\d+),(\d+)\]\[?(\d+),(\d+)\]"/);
        if (b) {
          found = { left: parseInt(b[1], 10), top: parseInt(b[2], 10), right: parseInt(b[3], 10), bottom: parseInt(b[4], 10) };
          break;
        }
      }
    }
    if (found) {
      const cx = Math.floor((found.left + found.right) / 2);
      const cy = Math.floor((found.top + found.bottom) / 2);
      // Try up to 3 taps on the discovered Send button to handle flaky UIs
      for (let attempt = 0; attempt < 3; attempt++) {
        try { await adb(`shell input tap ${cx} ${cy}`); } catch (e) { }
        await new Promise(r => setTimeout(r, 1200));
        // After attempting tap, check if message is in Sent
        const sent = await checkSmsSent(to, body);
        if (sent) return { status: 'sent', confirmed: true };
        // If not sent, check if composer still contains the body; if not, break and treat as sent/cleared
        const composer = await readComposerBody();
        if (!composer || !composer.includes(body.slice(0, 8))) break;
      }
    }
    // If we didn't find a send node, fall through to conservative coordinate taps below
  } catch (e) {
    try { console.error('uiautomator dump/send attempt failed', e); } catch (_) {}
  }

  // Last-resort: try a direct telephony service call. This is vendor/platform dependent and may require root
  // We try a few common service call signatures. These will likely fail on unrooted phones but are worth attempting.
  try {
    // Android 4.4+ used 'service call isms' on some devices. We try to send via interface 7 (may vary).
    // Note: these commands are best-effort and will often be rejected by permission checks.
    const cmds = [
      // Example using service call isms (old AOSP)
      `shell service call isms 7 i32 0 s16 "${to}" s16 "${body}"`,
      // Another pattern (may not work): call it directly via 'service call isms'
      `shell service call isms 5 s16 "${to}" s16 "${body}"`,
    ];
    for (const c of cmds) {
      try {
        const { stdout } = await adb(c);
        if (stdout && /Result: Parcel\(0x0\)|Result: Parcel\(0x1\)/i.test(stdout)) {
          // Wait briefly and check Sent
          await new Promise(r => setTimeout(r, 1200));
          if (await checkSmsSent(to, body)) return true;
        }
      } catch (e) {
        // ignore individual failures and continue
      }
    }
  } catch (e) {}

  // Give one final delay for system to update; then return a best-effort true so higher layers can log status
  // If we reached here without returning, try a last conservative coordinated tap
  try {
    const size2 = await getScreenSize();
    let w = 1080, h = 2400;
    if (size2 && size2.w && size2.h) { w = size2.w; h = size2.h; }
    // Try a few X positions near the right edge (92%, 88%, 84%) and a Y near the bottom (92%)
    const xs = [0.92, 0.88, 0.84].map(p => Math.max(40, Math.floor(w * p)));
    const y = Math.max(120, Math.floor(h * 0.92));
    for (const x of xs) {
      try { await adb(`shell input tap ${x} ${y}`); } catch (e) {}
      await new Promise(r => setTimeout(r, 900));
      if (await checkSmsSent(to, body)) return true;
    }
  } catch (e) {
    // ignore
  }

  await new Promise(r => setTimeout(r, 900));
  return true;
}

async function checkSmsSent(to, body) {
  // Best-effort check: query content://sms/sent for an item matching address and body
  try {
    const where = `address='${to}' AND body LIKE '%${body.slice(0, 20).replace(/'/g, "\\'")}%'`;
    const cmd = `shell content query --uri content://sms/sent --where "${where}"`;
    const { stdout } = await adb(cmd);
    if (!stdout) return false;
    // If there's any output, assume it's been placed in Sent.
    return stdout.includes(to) || stdout.includes(body.slice(0, 10));
  } catch (e) {
    // If content provider access fails or returns nothing, as a fallback try a small logcat scan
    try {
      const { stdout: logs } = await adb('logcat -d -t 40');
      if (logs && (logs.includes(to) || logs.includes(body.slice(0, 10)) || /ComposerPresenterImpl|SmsProvider|sendDonation|RcsSender/i.test(logs))) {
        return true;
      }
    } catch (le) {
      // ignore
    }
    return false;
  }
}

async function checkInboxForReply(to, bodySnippet, limit = 20) {
  // Query inbox for messages that include bodySnippet or come from the 'to' address
  try {
    const snippet = (bodySnippet || '').slice(0, 30).replace(/'/g, "\\'");
    const where = `address='${to}' OR body LIKE '%${snippet}%'`;
    const cmd = `shell content query --uri content://sms/inbox --projection _id,address,body,date --where "${where}" --sort 'date DESC' --limit ${limit}`;
    const { stdout } = await adb(cmd);
    if (!stdout) return [];
    // return raw stdout lines for caller parsing
    return stdout.split(/\r?\n/).map(l => l.trim()).filter(Boolean);
  } catch (e) {
    try {
      const { stdout: logs } = await adb('logcat -d -t 40');
      if (logs) return [logs];
    } catch (le) {}
    return [];
  }
}

module.exports = { adb, isDeviceConnected, sendSmsViaIntent, checkSmsSent, checkInboxForReply };
