import axios from 'axios';
const express = require('express');
const router = express.Router();

// Get the backend URL from environment variables, fallback to localhost:5000
const BACKEND_URL = process.env.REACT_APP_BACKEND_URL || 'http://localhost:5000';

class PaymentService {
    constructor() {
        this.api = axios.create({
            baseURL: BACKEND_URL,
            headers: {
                'Content-Type': 'application/json'
            }
        });
    }

    // Initialize M-Pesa payment
    async initiateMpesaPayment({ phoneNumber, amount, userId }) {
        try {
            const response = await this.api.post('/api/mpesa/stk-push', {
                phoneNumber,
                amount,
                userId
            });

            return {
                success: response.data.success,
                checkoutRequestId: response.data.CheckoutRequestID,
                message: response.data.message
            };
        } catch (error) {
            console.error('M-Pesa Payment Error:', error.response || error);
            throw new Error(error.response?.data?.message || 'Failed to initiate M-Pesa payment');
        }
    }

    // Initialize card payment
    async initiateCardPayment({ amount, userId }) {
        try {
            const response = await this.api.post('/api/payments/create-card-payment', {
                amount,
                userId
            });

            return {
                success: true,
                clientSecret: response.data.clientSecret
            };
        } catch (error) {
            console.error('Card Payment Error:', error.response || error);
            throw new Error(error.response?.data?.message || 'Failed to initiate card payment');
        }
    }

    // Check payment status
    async checkPaymentStatus(checkoutRequestId) {
        try {
            const response = await this.api.get(`/api/mpesa/status/${checkoutRequestId}`);
            return {
                status: response.data.status,
                message: response.data.message
            };
        } catch (error) {
            console.error('Payment Status Error:', error.response || error);
            throw new Error(error.response?.data?.message || 'Failed to check payment status');
        }
    }

    // Verify transaction
    async verifyTransaction(transactionId) {
        try {
            const response = await this.api.get(`/api/payments/verify/${transactionId}`);
            return {
                success: response.data.success,
                verified: response.data.verified,
                message: response.data.message
            };
        } catch (error) {
            throw new Error(error.response?.data?.message || 'Failed to verify transaction');
        }
    }

    // Get payment history for a user
    async getPaymentHistory(userId) {
        try {
            const response = await this.api.get(`/api/payments/history/${userId}`);
            return {
                success: true,
                payments: response.data.payments
            };
        } catch (error) {
            throw new Error(error.response?.data?.message || 'Failed to fetch payment history');
        }
    }

    // Cancel a pending payment
    async cancelPayment(paymentId) {
        try {
            const response = await this.api.post(`/api/payments/cancel/${paymentId}`);
            return {
                success: response.data.success,
                message: response.data.message
            };
        } catch (error) {
            throw new Error(error.response?.data?.message || 'Failed to cancel payment');
        }
    }

    // Request payment refund
    async requestRefund(paymentId, reason) {
        try {
            const response = await this.api.post(`/api/payments/refund/${paymentId}`, { reason });
            return {
                success: response.data.success,
                refundId: response.data.refundId,
                message: response.data.message
            };
        } catch (error) {
            throw new Error(error.response?.data?.message || 'Failed to request refund');
        }
    }

    // Get supported payment methods
    async getSupportedPaymentMethods() {
        try {
            const response = await this.api.get('/api/payments/methods');
            return {
                success: true,
                methods: response.data.methods
            };
        } catch (error) {
            throw new Error(error.response?.data?.message || 'Failed to fetch payment methods');
        }
    }
}

// Create and export a singleton instance
const paymentService = new PaymentService();
export default paymentService;
const fs = require('fs');
const path = require('path');
const moment = require('moment');

// --- Configuration & Simple storage ------------------------------------------------
const MPESA_ENV = (process.env.MPESA_ENV || 'sandbox').toLowerCase();
const BASE_URL = MPESA_ENV === 'production'
  ? 'https://api.safaricom.co.ke'
  : 'https://sandbox.safaricom.co.ke';

const SHORTCODE = process.env.MPESA_BUSINESS_SHORTCODE;
const PASSKEY = process.env.MPESA_PASSKEY;
const CALLBACK_URL = process.env.MPESA_CALLBACK_URL; // must be public (ngrok) for sandbox callbacks
const TRANSACTION_TYPE = process.env.MPESA_TRANSACTION_TYPE || 'CustomerPayBillOnline';

// Where to persist callback JSONs (for debugging)
const CALLBACK_DIR = process.env.MPESA_CALLBACK_DIR || path.join(process.cwd(), 'mpesa_callbacks');
if (!fs.existsSync(CALLBACK_DIR)) fs.mkdirSync(CALLBACK_DIR, { recursive: true });

// In-memory tracker for pending STK requests (CheckoutRequestID -> meta)
const pendingStkRequests = new Map();

// Simple token cache
let tokenCache = { token: null, expiresAt: 0 };

// --- Helpers ----------------------------------------------------------------------
function sanitizePhone(phone) {
  if (!phone) return null;
  let s = phone.toString().replace(/[^0-9]/g, '');
  if (s.length === 9 && !s.startsWith('0')) {
    // handle short numbers like 712345678 (unlikely) - do not modify
  }
  if (s.startsWith('0')) s = '254' + s.slice(1);
  if (s.length === 12 && s.startsWith('254')) return s;
  if (s.length === 13 && s.startsWith('+254')) return s.slice(1);
  return s; // best-effort; caller should validate length 12 and prefix 254
}

async function getAccessToken() {
  // Return cached token if still valid (with 30s safety margin)
  if (tokenCache.token && Date.now() < tokenCache.expiresAt - 30000) {
    return tokenCache.token;
  }

  const consumer_key = process.env.MPESA_CONSUMER_KEY;
  const consumer_secret = process.env.MPESA_CONSUMER_SECRET;
  if (!consumer_key || !consumer_secret) {
    throw new Error('Missing MPESA_CONSUMER_KEY or MPESA_CONSUMER_SECRET in env');
  }

  const url = `${BASE_URL}/oauth/v1/generate?grant_type=client_credentials`;
  const auth = "Basic " + Buffer.from(consumer_key + ":" + consumer_secret).toString("base64");

  const resp = await axios.get(url, { headers: { Authorization: auth } });
  const access_token = resp.data.access_token;
  const expires_in = resp.data.expires_in || 3600;
  tokenCache = { token: access_token, expiresAt: Date.now() + expires_in * 1000 };
  return access_token;
}

// Persist callback data for debugging
function saveCallbackJson(checkoutRequestId, body) {
  try {
    const filename = path.join(CALLBACK_DIR, `callback_${checkoutRequestId || moment().format('YYYYMMDDHHmmss')}.json`);
    fs.writeFileSync(filename, JSON.stringify(body, null, 2));
  } catch (err) {
    console.warn('Failed to save callback json:', err.message);
  }
}

// --- Routes -----------------------------------------------------------------------
router.get('/api/home', (req, res) => {
  res.json({ message: 'MPesa router alive.' });
});

router.get('/api/mpesa/access-token', async (req, res) => {
  try {
    const token = await getAccessToken();
    res.json({ success: true, accessToken: token });
  } catch (err) {
    console.error('Access token error:', err.message);
    res.status(500).json({ success: false, message: err.message });
  }
});

// Backwards-compatible names:
// - POST /api/stkpush  (your original)
// - POST /api/mpesa/stk-push  (frontend service expects this name)
async function handleStkPushRequest(req, res) {
  let requestBody = null;
  try {
    const { phone, amount, accountNumber, accountReference, transactionDesc } = req.body || {};

    if (!phone || !amount) {
      return res.status(400).json({ success: false, message: 'Missing required fields: phone and amount' });
    }

    if (!SHORTCODE || !PASSKEY || !CALLBACK_URL) {
      return res.status(500).json({
        success: false,
        message: 'MPESA_BUSINESS_SHORTCODE, MPESA_PASSKEY or MPESA_CALLBACK_URL not configured on server'
      });
    }

    const sanitizedPhone = sanitizePhone(phone);
    if (!sanitizedPhone || !sanitizedPhone.startsWith('254') || sanitizedPhone.length !== 12) {
      return res.status(400).json({ success: false, message: 'Invalid phone number format. Expect 2547XXXXXXXX' });
    }

    const accessToken = await getAccessToken();

    const timestamp = moment().format('YYYYMMDDHHmmss');
    const password = Buffer.from(SHORTCODE + PASSKEY + timestamp).toString('base64');

    requestBody = {
      BusinessShortCode: SHORTCODE,
      Password: password,
      Timestamp: timestamp,
      TransactionType: TRANSACTION_TYPE,
      Amount: Number(amount),
      PartyA: sanitizedPhone,
      PartyB: process.env.MPESA_PARTYB || SHORTCODE,
      PhoneNumber: sanitizedPhone,
      CallBackURL: CALLBACK_URL,
      AccountReference: accountNumber || accountReference || `Ref-${moment().format('YYYYMMDDHHmmss')}`,
      TransactionDesc: transactionDesc || 'Payment request'
    };

    const stkUrl = `${BASE_URL}/mpesa/stkpush/v1/processrequest`;
    const safResp = await axios.post(stkUrl, requestBody, {
      headers: {
        Authorization: 'Bearer ' + accessToken,
        'Content-Type': 'application/json'
      },
      timeout: 15000
    });

    const data = safResp.data;
    // If response code is 0 (success) we capture the CheckoutRequestID for polling/callback matching
    if (data.ResponseCode === '0' || data.ResponseCode === 0) {
      const checkoutId = data.CheckoutRequestID || data.CheckoutRequestId;
      const merchantReqId = data.MerchantRequestID || data.MerchantRequestId;

      // Track pending request in memory (and optionally persist minimal metadata)
      if (checkoutId) {
        pendingStkRequests.set(checkoutId, {
          phone: sanitizedPhone,
          amount: Number(amount),
          accountReference: requestBody.AccountReference,
          merchantRequestId: merchantReqId,
          createdAt: Date.now(),
          status: 'pending'
        });
      }

      return res.json({
        success: true,
        message: 'STK push initiated. Check your phone to complete the transaction.',
        data
      });
    } else {
      return res.status(400).json({
        success: false,
        message: data.ResponseDescription || 'STK push request rejected by Safaricom',
        data
      });
    }
  } catch (error) {
    console.error('STK push error:', {
      message: error.message,
      status: error.response?.status,
      responseData: error.response?.data,
      requestBody: requestBody ? { ...requestBody, Password: '[REDACTED]' } : null
    });

    return res.status(error.response?.status || 500).json({
      success: false,
      message: error.response?.data?.errorMessage || error.message || 'Failed to initiate STK push',
      error: error.response?.data || null
    });
  }
}
router.post('/api/stkpush', handleStkPushRequest);
router.post('/api/mpesa/stk-push', handleStkPushRequest);

// Callback endpoint that Safaricom will POST to
router.post('/api/mpesa/callback', (req, res) => {
  try {
    if (!req.body || !req.body.Body || !req.body.Body.stkCallback) {
      console.warn('Callback received with unexpected shape', req.body);
      // still respond 200 to Safaricom to avoid retries, but log
      return res.status(200).json({ success: false, message: 'Invalid callback data' });
    }

    const { Body: { stkCallback } } = req.body;
    const checkoutId = stkCallback.CheckoutRequestID;
    const merchantReqId = stkCallback.MerchantRequestID;
    const resultCode = stkCallback.ResultCode;
    const resultDesc = stkCallback.ResultDesc;

    // Save callback JSON for debugging
    saveCallbackJson(checkoutId || merchantReqId, req.body);

    // Update in-memory pending map
    if (checkoutId && pendingStkRequests.has(checkoutId)) {
      const meta = pendingStkRequests.get(checkoutId);
      meta.status = resultCode === 0 ? 'completed' : 'failed';
      meta.completedAt = Date.now();
      meta.resultCode = resultCode;
      meta.resultDesc = resultDesc;
      meta.rawCallback = stkCallback;
      pendingStkRequests.set(checkoutId, meta);
      console.log(`MPESA callback updated pending request ${checkoutId} -> ${meta.status}`);
    } else {
      // Not found in map -> persist callback for later reconciliation
      console.log('Callback did not match any pending request, saving for manual reconciliation.');
    }

    // Respond quickly with 200 to Safaricom
    return res.status(200).json({ success: true, message: 'Callback received' });
  } catch (err) {
    console.error('Callback processing error:', err);
    // Still return 200 to avoid Safaricom retries; but include error in logs
    return res.status(200).json({ success: false, message: 'Callback processed with errors' });
  }
});

// Status endpoint so frontend can poll for result
router.get('/api/mpesa/status/:checkoutRequestId', (req, res) => {
  const id = req.params.checkoutRequestId;
  if (!id) return res.status(400).json({ success: false, message: 'Missing checkoutRequestId' });

  // First check in-memory
  if (pendingStkRequests.has(id)) {
    const meta = pendingStkRequests.get(id);
    return res.json({
      success: true,
      status: meta.status,
      data: meta
    });
  }

  // Fall back to search saved callback files
  try {
    const files = fs.readdirSync(CALLBACK_DIR);
    const match = files.find(f => f.includes(id));
    if (match) {
      const body = JSON.parse(fs.readFileSync(path.join(CALLBACK_DIR, match)));
      const stkCallback = body.Body?.stkCallback;
      const status = stkCallback?.ResultCode === 0 ? 'completed' : 'failed';
      return res.json({ success: true, status, data: stkCallback });
    }
  } catch (err) {
    console.warn('Error checking persisted callbacks:', err.message);
  }

  // Not found -> still pending
  return res.json({ success: true, status: 'pending', message: 'No callback yet' });
});

module.exports = router;

const app = express();
const mpesaRouter = require('./routes/mpesa');

app.use(require('cors')());
app.use(express.json({ limit: '50mb' }));
app.use(mpesaRouter); // mounts routes defined in the file