// Game Odds Service - handles data processing for admin panel
// This service processes the existing preds_dump.json data without needing a separate backend

// Transform predictions data to game odds format
const transformPredictionsToGames = (predictions) => {
  // Reduced logging for performance
  
  const gamesMap = new Map();
  
  if (!predictions || typeof predictions !== 'object') {
    console.warn('Invalid predictions data:', predictions);
    return [];
  }

  try {
    Object.values(predictions).forEach((prediction, index) => {
      if (!prediction || !prediction.selectedOdds || !Array.isArray(prediction.selectedOdds)) {
        console.warn(`Skipping invalid prediction at index ${index}:`, prediction);
        return;
      }

      prediction.selectedOdds.forEach((odd, oddIndex) => {
        if (!odd) {
          console.warn(`Skipping null odd at prediction ${index}, odd ${oddIndex}`);
          return;
        }

        const gameId = odd.gameId || `${odd.competitors?.[0]?.name || 'unknown'}_vs_${odd.competitors?.[1]?.name || 'unknown'}_${odd.Date || Date.now()}`;
        
        if (!gamesMap.has(gameId)) {
          // Extract team names from competitors
          const homeTeam = odd.competitors && odd.competitors[0] && odd.competitors[0].name ? odd.competitors[0].name : 'Unknown Home';
          const awayTeam = odd.competitors && odd.competitors[1] && odd.competitors[1].name ? odd.competitors[1].name : 'Unknown Away';
          
          // Prefer ISO date string over timestamp as the timestamp values seem incorrect
          let gameTimestamp = odd.Date;
          if (!gameTimestamp && odd.DateTimestamp) {
            // Only use DateTimestamp if no Date is available, and if it's a reasonable value
            const timestampMs = parseInt(odd.DateTimestamp);
            if (timestampMs && timestampMs < 4000000000000) { // Reasonable check (year < 2096)
              gameTimestamp = new Date(timestampMs).toISOString();
            }
          }
          
          gamesMap.set(gameId, {
            gameId: gameId,
            homeTeam: homeTeam,
            awayTeam: awayTeam,
            league: 'Various Leagues', // You can enhance this with actual league data
            timestamp: gameTimestamp,
            odds: [],
            oddsCount: 0
          });
        }
        
        // Add odds to game
        const game = gamesMap.get(gameId);
        if (game && game.odds) {
          game.odds.push({
            selectionName: odd.selectionName || 'Unknown Selection',
            odds: odd.odds || 0,
            marketName: odd.marketName || 'Unknown Market',
            periodType: 'Full Time'
          });
          game.oddsCount = game.odds.length;
        }
      });
    });
  } catch (error) {
    console.error('Error in transformPredictionsToGames:', error);
    return [];
  }
  
  const result = Array.from(gamesMap.values());
  // Transform complete
  return result;
};

// Add time categories to games
const addTimeCategories = (games) => {
  // Always use current moment for real-time categorization
  const now = new Date();
  const twoHoursAgo = new Date(now.getTime() - 2 * 60 * 60 * 1000);
  const twoHoursFromNow = new Date(now.getTime() + 2 * 60 * 60 * 1000);
  const twentyFourHoursFromNow = new Date(now.getTime() + 24 * 60 * 60 * 1000);

  // Reduced logging to prevent memory issues
  console.log(`Real-time categorization: Processing ${games.length} games at ${now.toLocaleTimeString()}`);

  const categoryCounts = { past: 0, live: 0, upcoming2h: 0, upcoming24h: 0, future: 0, invalid: 0 };
  // Removed sample games collection to improve performance

  const categorizedGames = games.map((game, index) => {
    let timeCategory = 'invalid';
    
    if (game && game.timestamp && !isNaN(new Date(game.timestamp).getTime())) {
      const gameTime = new Date(game.timestamp);
      
      // Time categorization logic:
      // - past: Games that started more than 2 hours ago
      // - live: Games that have started but are less than 2 hours old (LIVE GAMES)
      // - upcoming2h: Games starting within next 2 hours (haven't started yet)
      // - upcoming24h: ALL games starting within next 24 hours (includes 2hr games)  
      // - future: All games that haven't started yet (beyond 24 hours)
      if (gameTime < twoHoursAgo) {
        timeCategory = 'past';
      } else if (gameTime < now) {
        // Games that have started but are less than 2 hours old (LIVE)
        timeCategory = 'live';
      } else if (gameTime <= twoHoursFromNow) {
        timeCategory = 'upcoming2h';
      } else if (gameTime <= twentyFourHoursFromNow) {
        timeCategory = 'upcoming24h';
      } else {
        timeCategory = 'future';
      }
    }
    
    categoryCounts[timeCategory]++;
    return { ...game, timeCategory };
  });

  // Only log summary to reduce memory usage
  console.log('Categorization complete:', categoryCounts);
  
  return categorizedGames;
};

// Categorize games by time
const categorizeGamesByTime = (games) => {
  const categories = {
    past: 0,
    live: 0,
    upcomingIn2Hours: 0,
    upcomingIn24Hours: 0, // This will include ALL games within 24h (including 2h games)
    future: 0,
    invalidTimestamp: 0
  };

  games.forEach(game => {
    switch (game.timeCategory) {
      case 'past':
        categories.past++;
        break;
      case 'live':
        categories.live++;
        break;
      case 'upcoming2h':
        categories.upcomingIn2Hours++;
        categories.upcomingIn24Hours++; // 2hr games are also within 24hrs
        categories.future++; // 2hr games are also future games (haven't started)
        break;
      case 'upcoming24h':
        categories.upcomingIn24Hours++;
        categories.future++; // 24hr games are also future games (haven't started)
        break;
      case 'future':
        categories.future++; // Beyond 24hr games are future games
        break;
      case 'invalid':
      default:
        categories.invalidTimestamp++;
        break;
    }
  });

  return categories;
};

// Main service class
class GameOddsService {
  constructor() {
    this.predictions = null;
    this.games = null;
    this.lastFetchTime = null;
    this.cacheDuration = 5 * 60 * 1000; // 5 minutes cache
    // Removed mockDataCache since we only use real data now
  }

  // Load predictions data from backend database
  async loadPredictionsData() {
    // List of actual available endpoints (keeping current working system)
    const endpointsToTry = [
      'http://localhost:5000/api/predictions',           // ✅ Working endpoint (confirmed)
      'http://localhost:5000/api/admin/gameOdds',       // ✅ Available endpoint
      'http://localhost:5000/api/lettherebetodayspredictions', // ✅ Available endpoint
      'http://localhost:5000/api/createPredictions'     // ✅ Available endpoint
    ];

    for (const endpoint of endpointsToTry) {
      try {
        if (endpoint !== 'http://localhost:5000/api/predictions') {
          console.log(`Trying endpoint: ${endpoint}`);
        }
        
        const response = await fetch(endpoint, {
          method: 'GET',
          headers: {
            'Content-Type': 'application/json',
            'Authorization': localStorage.getItem('token') ? `Bearer ${localStorage.getItem('token')}` : undefined
          }
        });
        
        if (response.ok) {
          const data = await response.json();
          
          if (endpoint === 'http://localhost:5000/api/predictions') {
            console.log(`✅ Successfully loaded data from primary endpoint`);
          } else {
            console.log(`✅ Success with fallback ${endpoint}:`, {
              type: typeof data,
              isArray: Array.isArray(data),
              keysCount: data && typeof data === 'object' ? Object.keys(data).length : 'N/A'
            });
          }
          
          // Validate data format
          if (!data || (typeof data !== 'object')) {
            console.warn(`Invalid data format from ${endpoint}`);
            continue;
          }
          
          this.predictions = data;
          this.lastFetchTime = Date.now();
          return data;
        } else {
          if (endpoint !== 'http://localhost:5000/api/predictions') {
            console.log(`❌ ${endpoint} failed: HTTP ${response.status} - ${response.statusText}`);
          }
        }
      } catch (error) {
        if (endpoint !== 'http://localhost:5000/api/predictions') {
          console.log(`❌ ${endpoint} error:`, error.message);
        }
      }
    }

    // All endpoints failed, try fallbacks
    console.warn('All backend endpoints failed, trying fallbacks...');
    
    try {
      console.log('Attempting fallback to local preds_dump.json...');
      const fallbackResponse = await fetch('/preds_dump.json');
      if (fallbackResponse.ok) {
        const fallbackData = await fallbackResponse.json();
        console.log('✅ Fallback data loaded successfully:', Object.keys(fallbackData || {}).length, 'records');
        this.predictions = fallbackData;
        this.lastFetchTime = Date.now();
        console.warn('Using local fallback data due to database connection failure');
        return fallbackData;
      }
    } catch (fallbackError) {
      console.error('Fallback also failed:', fallbackError);
    }
    
    // Emergency fallback: use mock data
    const emergencyMockData = this.getMockData();
    this.predictions = emergencyMockData;
    this.lastFetchTime = Date.now();
    console.warn('Using emergency mock data - both database and local file failed');
    return emergencyMockData;
  }

  // Force refresh data from database (clear cache)
  async forceRefreshFromDatabase() {
    console.log('Force refreshing data from database...');
    this.predictions = null;
    this.lastFetchTime = null;
    return await this.loadPredictionsData();
  }

  // REMOVED: adjustDatesForDemo method - now using real timestamps for authentic time categorization

  // Get mock data for testing real-time categorization
  getMockData() {
    const now = new Date();
    // Creating test games
    
    // Create games at specific time intervals to test each category
    const threeHoursAgo = new Date(now.getTime() - 3 * 60 * 60 * 1000); // Past (>2h ago)
    const fiveHoursAgo = new Date(now.getTime() - 5 * 60 * 60 * 1000); // Past (>2h ago)
    const oneHourAgo = new Date(now.getTime() - 1 * 60 * 60 * 1000); // Past (recently started)
    const thirtyMinutesFromNow = new Date(now.getTime() + 30 * 60 * 1000); // Upcoming2h (starting soon)
    const oneHourFromNow = new Date(now.getTime() + 1 * 60 * 60 * 1000); // Upcoming2h (starting within 2h)
    const sixHoursFromNow = new Date(now.getTime() + 6 * 60 * 60 * 1000); // Upcoming24h (starting within 24h)
    const twelvehHoursFromNow = new Date(now.getTime() + 12 * 60 * 60 * 1000); // Upcoming24h (starting within 24h)
    const twoDaysFromNow = new Date(now.getTime() + 2 * 24 * 60 * 60 * 1000); // Future (starting beyond 24h)

    return {
      "test_past_recent": {
        "predictionId": "test_past_1",
        "selectedOdds": [
          {
            "gameId": "test_past_game_1",
            "competitors": [
              { "name": "Liverpool" },
              { "name": "Arsenal" }
            ],
            "selectionName": "Liverpool Win",
            "odds": 2.1,
            "marketName": "Match Winner",
            "Date": threeHoursAgo.toISOString()
          }
        ]
      },
      "test_past_older": {
        "predictionId": "test_past_2",
        "selectedOdds": [
          {
            "gameId": "test_past_game_2",
            "competitors": [
              { "name": "Chelsea" },
              { "name": "Manchester City" }
            ],
            "selectionName": "Over 2.5 Goals",
            "odds": 1.75,
            "marketName": "Total Goals",
            "Date": fiveHoursAgo.toISOString()
          }
        ]
      },
      "test_past_recent_3": {
        "predictionId": "test_past_3",
        "selectedOdds": [
          {
            "gameId": "test_past_game_3",
            "competitors": [
              { "name": "Tottenham" },
              { "name": "West Ham" }
            ],
            "selectionName": "Both Teams to Score",
            "odds": 1.90,
            "marketName": "Both Teams to Score",
            "Date": oneHourAgo.toISOString()
          }
        ]
      },
      "test_upcoming2h_soon": {
        "predictionId": "test_upcoming2h_1",
        "selectedOdds": [
          {
            "gameId": "test_upcoming2h_game_1",
            "competitors": [
              { "name": "Barcelona" },
              { "name": "Real Madrid" }
            ],
            "selectionName": "Barcelona Win",
            "odds": 1.9,
            "marketName": "Match Winner",
            "Date": thirtyMinutesFromNow.toISOString()
          }
        ]
      },
      "test_upcoming2h_medium": {
        "predictionId": "test_upcoming2h_2",
        "selectedOdds": [
          {
            "gameId": "test_upcoming2h_game_2",
            "competitors": [
              { "name": "PSG" },
              { "name": "Bayern Munich" }
            ],
            "selectionName": "Both Teams To Score",
            "odds": 1.6,
            "marketName": "Both Teams To Score",
            "Date": thirtyMinutesFromNow.toISOString()
          }
        ]
      },
      "test_upcoming2h_later": {
        "predictionId": "test_upcoming2h_3",
        "selectedOdds": [
          {
            "gameId": "test_upcoming2h_game_3",
            "competitors": [
              { "name": "Tottenham" },
              { "name": "West Ham" }
            ],
            "selectionName": "Tottenham Win",
            "odds": 1.85,
            "marketName": "Match Winner",
            "Date": oneHourFromNow.toISOString()
          }
        ]
      },
      "test_upcoming24h_afternoon": {
        "predictionId": "test_upcoming24h_1",
        "selectedOdds": [
          {
            "gameId": "test_upcoming24h_game_1",
            "competitors": [
              { "name": "Juventus" },
              { "name": "AC Milan" }
            ],
            "selectionName": "Under 3.5 Goals",
            "odds": 1.8,
            "marketName": "Total Goals",
            "Date": sixHoursFromNow.toISOString()
          }
        ]
      },
      "test_upcoming24h_tonight": {
        "predictionId": "test_upcoming24h_2",
        "selectedOdds": [
          {
            "gameId": "test_upcoming24h_game_2",
            "competitors": [
              { "name": "Inter Milan" },
              { "name": "Napoli" }
            ],
            "selectionName": "Draw",
            "odds": 3.2,
            "marketName": "Match Winner",
            "Date": twelvehHoursFromNow.toISOString()
          }
        ]
      },
      "test_future_weekend": {
        "predictionId": "test_future_1",
        "selectedOdds": [
          {
            "gameId": "test_future_game_1",
            "competitors": [
              { "name": "Atletico Madrid" },
              { "name": "Sevilla" }
            ],
            "selectionName": "Atletico Madrid Win",
            "odds": 2.2,
            "marketName": "Match Winner",
            "Date": twoDaysFromNow.toISOString()
          }
        ]
      }
    };
  }

  // Check if cache is valid
  isCacheValid() {
    return this.lastFetchTime && (Date.now() - this.lastFetchTime) < this.cacheDuration;
  }

  // Get processed games data with real-time categorization
  async getGamesData() {
    // Load fresh predictions data if cache expired
    if (!this.predictions || !this.isCacheValid()) {
      console.log('Loading fresh data from source...');
      await this.loadPredictionsData();
    }

    console.log('Processing predictions data:', {
      predictionsExists: !!this.predictions,
      predictionsType: typeof this.predictions,
      predictionsKeys: this.predictions ? Object.keys(this.predictions).length : 0
    });

    // ALWAYS recalculate time categories using current moment
    // Time categories change continuously, so never cache them
    const transformedGames = transformPredictionsToGames(this.predictions);
    console.log('Transformed games count:', transformedGames.length);
    
    const gamesWithRealTimeCategories = addTimeCategories(transformedGames);
    console.log('Games with categories count:', gamesWithRealTimeCategories.length);
    
    // Data processed  
    return gamesWithRealTimeCategories;
  }

  // Lightweight getter that returns processed games only from cache.
  // It will NOT trigger a network reload. Use this when you want non-blocking reads
  // for UI elements like autocomplete where stale-but-fast data is acceptable.
  getCachedGamesData() {
    try {
      if (!this.predictions) return [];
      const transformedGames = transformPredictionsToGames(this.predictions);
      const gamesWithRealTimeCategories = addTimeCategories(transformedGames);
      return gamesWithRealTimeCategories;
    } catch (e) {
      console.warn('getCachedGamesData failed:', e && e.message ? e.message : e);
      return [];
    }
  }

  // Get dashboard data
  // Get dashboard data
  async getDashboardData() {
    const games = await this.getGamesData();
    const categories = categorizeGamesByTime(games);
    
    // Calculate upcoming games only (exclude past games from display)
    const upcomingGamesTotal = categories.upcomingIn2Hours + categories.upcomingIn24Hours + categories.future;
    
    return {
      totalGamesInDb: upcomingGamesTotal, // Show only upcoming games total
      gamesWithin24Hours: categories.upcomingIn24Hours, // Already includes 2hr games
      gamesWithValidTimestamps: upcomingGamesTotal, // Only count upcoming games
      // Map to UI expected field names
      pastGames: categories.past,
      upcomingIn2Hours: categories.upcomingIn2Hours, 
      upcomingIn24Hours: categories.upcomingIn24Hours, // Already includes 2hr games
      futureGames: categories.future,
      invalidTimestamp: categories.invalidTimestamp,
      // Keep original names for backward compatibility
      ...categories
    };
  }

  // Get filtered and paginated games
  async getFilteredGames({ timeCategory = 'all', search = '', sortBy = 'timestamp', sortOrder = 'desc', page = 1, limit = 20 }) {
    try {
      // Reduced logging for performance
      let games = await this.getGamesData();
      
      if (!Array.isArray(games)) {
        console.error('Games data is not an array:', games);
        return {
          games: [],
          currentPage: page,
          totalPages: 0,
          totalGames: 0,
          hasNextPage: false,
          hasPrevPage: false
        };
      }
      
      // Filter out only past games - show live and upcoming games
      games = games.filter(game => game && game.timeCategory !== 'past' && game.timeCategory !== 'invalid');
      
      // Apply additional filters
      if (timeCategory && timeCategory !== 'all') {
        if (timeCategory === 'upcoming24h') {
          // For 24h filter, include both 2hr and 24hr games
          games = games.filter(game => game && (game.timeCategory === 'upcoming2h' || game.timeCategory === 'upcoming24h'));
        } else if (timeCategory === 'future') {
          // For future filter, include ALL games that haven't started yet (2hr, 24hr, and beyond)
          games = games.filter(game => game && (game.timeCategory === 'upcoming2h' || game.timeCategory === 'upcoming24h' || game.timeCategory === 'future'));
        } else if (timeCategory === 'live') {
          // For live filter, show only live games
          games = games.filter(game => game && game.timeCategory === 'live');
        } else {
          games = games.filter(game => game && game.timeCategory === timeCategory);
        }
      }
      
      // Filter by search term
      if (search) {
        const searchLower = search.toLowerCase();
        games = games.filter(game => 
          game && (
            (game.homeTeam && game.homeTeam.toLowerCase().includes(searchLower)) ||
            (game.awayTeam && game.awayTeam.toLowerCase().includes(searchLower)) ||
            (game.league && game.league.toLowerCase().includes(searchLower)) ||
            (game.gameId && game.gameId.includes(search))
          )
        );
      }
      
      // Sort games
      games.sort((a, b) => {
        if (!a || !b) return 0;
        
        let aValue, bValue;
        
        switch (sortBy) {
          case 'homeTeam':
            aValue = (a.homeTeam || '').toLowerCase();
            bValue = (b.homeTeam || '').toLowerCase();
            break;
          case 'awayTeam':
            aValue = (a.awayTeam || '').toLowerCase();
            bValue = (b.awayTeam || '').toLowerCase();
            break;
          case 'league':
            aValue = (a.league || '').toLowerCase();
            bValue = (b.league || '').toLowerCase();
            break;
          case 'timestamp':
          default:
            // For timestamp sorting, use smart category-based ordering
            const categoryPriority = {
              'upcoming2h': 1,
              'upcoming24h': 2, 
              'future': 3,
              'past': 4,
              'invalid': 5
            };
            
            const aPriority = categoryPriority[a.timeCategory] || 5;
            const bPriority = categoryPriority[b.timeCategory] || 5;
            
            // If different categories, sort by category priority
            if (aPriority !== bPriority) {
              if (sortOrder === 'desc') {
                return bPriority - aPriority; // Reverse category order for desc
              } else {
                return aPriority - bPriority; // Normal category order for asc
              }
            }
            
            // Same category: sort by actual timestamp
            aValue = new Date(a.timestamp || 0).getTime();
            bValue = new Date(b.timestamp || 0).getTime();
            
            // Apply consistent logic based on sortOrder
            // For upcoming games: asc = soonest first, desc = latest first
            // For past games: asc = oldest first, desc = most recent first
            break;
        }
        
        if (sortOrder === 'desc') {
          return aValue > bValue ? -1 : aValue < bValue ? 1 : 0;
        } else {
          return aValue < bValue ? -1 : aValue > bValue ? 1 : 0;
        }
      });
      
      // Pagination
      const pageNum = parseInt(page) || 1;
      const limitNum = parseInt(limit) || 20;
      const startIndex = (pageNum - 1) * limitNum;
      const endIndex = startIndex + limitNum;
      
      const paginatedGames = games.slice(startIndex, endIndex);
      const totalPages = Math.ceil(games.length / limitNum);
      
      const result = {
        games: paginatedGames,
        currentPage: pageNum,
        totalPages: totalPages,
        totalGames: games.length,
        hasNextPage: pageNum < totalPages,
        hasPrevPage: pageNum > 1
      };
      
      // Result prepared
      return result;
    } catch (error) {
      console.error('Error in getFilteredGames:', error);
      return {
        games: [],
        currentPage: page,
        totalPages: 0,
        totalGames: 0,
        hasNextPage: false,
        hasPrevPage: false
      };
    }
  }

  // Get individual game details
  async getGameDetails(gameId) {
    const games = await this.getGamesData();
    const game = games.find(g => g.gameId === gameId);
    
    if (!game) {
      throw new Error(`Game with ID ${gameId} not found`);
    }
    
    return {
      gameInfo: {
        gameId: game.gameId,
        homeTeam: game.homeTeam,
        awayTeam: game.awayTeam,
        league: game.league,
        timestamp: game.timestamp
      },
      timeCategory: game.timeCategory,
      totalOdds: game.oddsCount,
      odds: game.odds
    };
  }

  // Real cleanup function - deletes games more than 2 hours past from Firebase
  async cleanup() {
    try {
      const games = await this.getGamesData();
      const pastGames = games.filter(game => game.timeCategory === 'past');
      
      if (pastGames.length === 0) {
        return {
          message: 'No past games found to cleanup',
          removedCount: 0
        };
      }

      console.log(`Starting cleanup of ${pastGames.length} past games...`);
      
      // Import Firebase database functions
      const { getDatabase, ref, remove, get } = await import('firebase/database');
      const database = getDatabase();
      
      let removedCount = 0;
      const errors = [];

      // Remove past games from Firebase
      for (const game of pastGames) {
        try {
          // Find and remove predictions that contain this game
          const predictionsRef = ref(database, 'predictions');
          const snapshot = await get(predictionsRef);
          
          if (snapshot.exists()) {
            const predictions = snapshot.val();
            
            for (const [predictionKey, prediction] of Object.entries(predictions)) {
              if (prediction && prediction.selectedOdds && Array.isArray(prediction.selectedOdds)) {
                // Check if this prediction contains the past game
                const containsPastGame = prediction.selectedOdds.some(odd => 
                  odd.gameId === game.gameId ||
                  (odd.competitors && 
                   odd.competitors[0]?.name === game.homeTeam &&
                   odd.competitors[1]?.name === game.awayTeam)
                );
                
                if (containsPastGame) {
                  // Remove the entire prediction if it contains past game
                  const predictionRef = ref(database, `predictions/${predictionKey}`);
                  await remove(predictionRef);
                  removedCount++;
                  console.log(`Removed prediction ${predictionKey} containing past game ${game.gameId}`);
                }
              }
            }
          }
        } catch (error) {
          console.error(`Error removing game ${game.gameId}:`, error);
          errors.push(`${game.gameId}: ${error.message}`);
        }
      }

      // Clear local cache to force reload
      this.predictions = null;
      this.lastFetchTime = null;

      const message = removedCount > 0 
        ? `Successfully cleaned up ${removedCount} predictions containing past games`
        : 'No predictions were removed during cleanup';

      if (errors.length > 0) {
        console.warn('Cleanup completed with errors:', errors);
      }

      return {
        message: message,
        removedCount: removedCount,
        errors: errors.length > 0 ? errors : undefined
      };
      
    } catch (error) {
      console.error('Cleanup failed:', error);
      return {
        message: `Cleanup failed: ${error.message}`,
        removedCount: 0,
        error: error.message
      };
    }
  }
}

// Export singleton instance
const gameOddsService = new GameOddsService();
export default gameOddsService;