import React from 'react';
import { enhancePredictionWithFriendlyNames } from '../utils/enhancePrediction';
import { useNavigate } from 'react-router-dom';
// Note: Modal/Backdrop not required in this list view

class ValueBetsPredictionsList extends React.Component {
  state = {
    loading: true,
    valueBetsPredictions: [],
    copyingMap: {},
    sendingMap: {},
    visitedMap: {},
    reports: {},
    confirmedMap: {},
    autobetRunning: false,
    autobetStats: null,
    targetSuccessfulBets: 5
  };

  autobetTimeout = null;
  autobetCancel = false;
  refreshInterval = null;
  isActive = true;

  componentDidMount() {
    this.fetchValueBetsPredictions();
    try { 
      const visited = JSON.parse(localStorage.getItem('copiedValueBetsPredictions') || '{}'); 
      if (visited && typeof visited === 'object') this.setState({ visitedMap: visited }); 
    } catch(e){}
    try { 
      const confirmed = JSON.parse(localStorage.getItem('confirmedValueBetsPredictions') || '{}'); 
      if (confirmed && typeof confirmed === 'object') this.setState({ confirmedMap: confirmed }); 
    } catch(e){}
    try { 
      const reports = JSON.parse(localStorage.getItem('valueBetsPredictionCopyReports') || '{}'); 
      if (reports && typeof reports === 'object') {
        // sanitize any report.message that might be an object
        Object.keys(reports).forEach(k => {
          const r = reports[k];
          if (r && r.message && typeof r.message === 'object') {
            try { reports[k].message = this.normalizeResponseMessage(r.message); } catch (e) { reports[k].message = JSON.stringify(r.message); }
          }
        });
        this.setState({ reports });
      }
    } catch(e){}
    this.refreshInterval = setInterval(() => this.fetchValueBetsPredictions(), 120000);
  }

  componentWillUnmount() {
    this.isActive = false;
    if (this.refreshInterval) clearInterval(this.refreshInterval);
  }

  persistVisited = (m) => { 
    try { 
      localStorage.setItem('copiedValueBetsPredictions', JSON.stringify(m || {})); 
    } catch(e){} 
  };
  persistConfirmed = (m) => { 
    try { 
      localStorage.setItem('confirmedValueBetsPredictions', JSON.stringify(m || {})); 
    } catch(e){} 
  };
  persistReports = (r) => { 
    try { 
      localStorage.setItem('valueBetsPredictionCopyReports', JSON.stringify(r || {})); 
    } catch(e){} 
  };

  parseStakeFromSms = (sms) => { 
    if (!sms || typeof sms !== 'string') return 1; 
    const parts = sms.split('#'); 
    const last = parts[parts.length-1]; 
    const n = parseFloat(last); 
    return Number.isFinite(n) ? n : 1; 
  };

  // Parser for invoice text response
  parseInvoiceText = (text) => {
    const lines = text.split('\n');
    const strippedLines = lines.map(line => line.trim()).filter(line => line.length > 0);

    const prediction = {
      'predictionId': '',
      'predictionName': '',
      'predictiontype': '',
      'price': 0.0,
      'totalOdds': 0.0,
      'numGames': 0,
      'selectedOdds': [],
      'potentialWin': 0.0,
      'createdAt': new Date().toISOString()
    };

    let i = 0;
    const n = strippedLines.length;
    while (i < n) {
      let line = strippedLines[i];
      if (line.startsWith('Prediction ID:')) {
        prediction['predictionId'] = line.split(':', 2)[1].trim();
      } else if (line.startsWith('Prediction Name:')) {
        prediction['predictionName'] = line.split(':', 2)[1].trim();
      } else if (line.startsWith('Prediction Type:')) {
        prediction['predictiontype'] = line.split(':', 2)[1].trim();
      } else if (line.startsWith('Number of Games:')) {
        prediction['numGames'] = parseInt(line.split(':', 2)[1].trim(), 10);
      } else if (line.startsWith('Total Odds:')) {
        prediction['totalOdds'] = parseFloat(line.split(':', 2)[1].trim());
      } else if (line.startsWith('Prediction Price:')) {
        const pricePart = line.split(':', 2)[1].trim();
        const priceMatch = /(\d+(?:\.\d+)?)/.exec(pricePart);
        if (priceMatch) {
          prediction['price'] = parseFloat(priceMatch[1]);
        }
      } else if (line.startsWith('Selected Odds:')) {
        i += 1; // skip separator
        let gameNum = 1;
        while (i < n && gameNum <= prediction['numGames']) {
          if (/^\d+\./.test(strippedLines[i])) {
            const match = /^(\d+)\.\s*(.*)$/.exec(strippedLines[i]);
            if (match) {
              const gNum = parseInt(match[1], 10);
              if (gNum !== gameNum) break;
              const title = match[2];
              const parts = /\s+vs\s+/.exec(title);
              const home = parts ? title.substring(0, parts.index).trim() : title;
              const away = parts ? title.substring(parts.index + 3).trim() : '';
              i += 1;
              let date = 'N/A';
              let time = 'N/A';
              let market = 'N/A';
              let selection = 'N/A';
              let odds = 0.0;
              while (i < n && !/^\d+\./.test(strippedLines[i]) && !strippedLines[i].startsWith('Error generating') && !strippedLines[i].startsWith('Total Amount:')) {
                let l = strippedLines[i];
                if (l.startsWith('Date:')) {
                  date = l.split(':', 2)[1].trim();
                } else if (l.startsWith('Time:')) {
                  time = l.split(':', 2)[1].trim();
                } else if (l.startsWith('Market:')) {
                  market = l.split(':', 2)[1].trim();
                } else if (l.startsWith('Selection:')) {
                  selection = l.split(':', 2)[1].trim();
                } else if (l.startsWith('Odds:')) {
                  i += 1;
                  if (i < n) {
                    const oddsPart = strippedLines[i];
                    const oddsMatch = /(\d+(?:\.\d+)?)/.exec(oddsPart);
                    if (oddsMatch) {
                      odds = parseFloat(oddsMatch[1]);
                    }
                  }
                  break;
                }
                i += 1;
              }
              // DateTimestamp
              let dateTimestamp = Date.now();
              if (date !== 'N/A') {
                try {
                  const dt = new Date(date);
                  if (!isNaN(dt.getTime())) {
                    dateTimestamp = dt.getTime();
                  }
                } catch (e) {}
              }
              const selectedOdd = {
                'Date': date !== 'N/A' ? date : new Date().toISOString().split('T')[0],
                'DateTimestamp': dateTimestamp,
                'time': time || 'N/A',
                'competitors': [
                  { 'name': home },
                  { 'name': away }
                ],
                'gameId': `valuebet_${gameNum}`,
                'marketName': market,
                'odds': odds,
                'selectionName': selection
              };
              prediction['selectedOdds'].push(selectedOdd);
              gameNum += 1;
              i -= 1; // Adjust for outer increment
            }
          }
          i += 1;
        }
        break;
      }
      i += 1;
    }

    prediction['potentialWin'] = prediction['price'] * prediction['totalOdds'];
    prediction['_localId'] = prediction['predictionId'] || `local-${Date.now()}`;

    return { predictions: [prediction] };
  };

  fetchValueBetsPredictions = async () => {
    // Lightweight fetch: only collect small preview data so UI renders fast.
    this.setState({ loading: true });
    try {
      // Try preview-enabled endpoint first (non-breaking):
      const tryUrls = [
        'http://localhost:5000/api/getValueBetsPredictions?preview=true',
        'http://localhost:5000/api/getValueBetsPreviews',
        'http://localhost:5000/api/getValueBetsPredictions'
      ];

      let data = null;
      let lastError = null;
      for (const url of tryUrls) {
        try {
          const res = await fetch(url);
          if (!res.ok) throw new Error(`HTTP ${res.status} from ${url}`);
          try {
            data = await res.json();
          } catch (jsonErr) {
            const text = await res.text().catch(() => '');
            data = this.parseInvoiceText(text);
          }
          console.debug('ValueBets preview fetch from', url, 'shape:', data && (Array.isArray(data) ? 'array' : typeof data));
          // If we got something that looks like predictions, stop trying
          if (data) break;
        } catch (err) {
          lastError = err;
          console.debug('Preview fetch failed for', url, err && err.message ? err.message : err);
          continue;
        }
      }
      if (!data) throw lastError || new Error('Failed to fetch value bets predictions');

      // Accept multiple shapes from backend or local fallback:
      let list = [];
      if (Array.isArray(data)) list = data;
      else if (Array.isArray(data.predictions)) list = data.predictions;
      else if (data && data.predictions && typeof data.predictions === 'object') list = Object.values(data.predictions);
      else if (data && typeof data === 'object') list = Object.values(data);
      else list = [];

      // Map to lightweight preview objects — avoid heavy per-game normalization here.
      const previews = list.map((p, i) => {
        const id = p.predictionId || p.id || p._localId || `valuebet-${i}`;
        const gamesArr = Array.isArray(p.games) ? p.games : Array.isArray(p.selectedOdds) ? p.selectedOdds : (p.legs || []);
        const numGames = Array.isArray(gamesArr) ? gamesArr.length : 0;
        return {
          _localId: p._localId || id,
          // unique instance id to avoid collisions when multiple items share the same predictionId/_localId
          _instanceId: `${id}-${i}`,
          predictionId: p.predictionId || p.id || null,
          predictionName: p.predictionName || p.name || `ValueBet ${id}`,
          numGames,
          totalOdds: p.totalOdds || p.total_odds || null,
          price: p.price || p.predictionPrice || 10,
          potentialWin: p.potentialWin || (p.price ? (p.price * (p.totalOdds || 1)) : null),
          createdAt: p.createdAt || p.created || new Date().toISOString(),
          smsString: p.smsString || null,
          // keep original minimal games reference for on-demand normalization
          _rawGamesRef: gamesArr && gamesArr.slice ? gamesArr.slice(0, 6) : [],
          // keep full raw object reference for actions that need to normalize
          _rawPrediction: p
        };
      }).filter(p => p.numGames >= 1);

      if (this.isActive) this.setState({ valueBetsPredictions: previews, loading: false });
    } catch (e) {
      console.error('fetchValueBetsPredictions error', e);
      if (this.isActive) this.setState({ valueBetsPredictions: [], loading: false });
    }
  };

  // Normalize prediction into the shape Invoice expects
  normalizePrediction = (prediction) => {
    if (!prediction) return null;
    // If this is a lightweight preview, prefer the original raw prediction object for accurate normalization
    if (prediction._rawPrediction && typeof prediction._rawPrediction === 'object') {
      prediction = prediction._rawPrediction;
    }
    const predId = prediction.predictionId || prediction.id || prediction._localId || `valuebet-${Date.now()}`;
    const rawGames = prediction.games || prediction.selectedOdds || prediction.legs || [];
    const selectedOdds = Array.isArray(rawGames) ? rawGames.map(g => {
      const ts = (g.DateTimestamp !== undefined && g.DateTimestamp !== null) ? Number(g.DateTimestamp) : (g.Date ? Date.parse(g.Date) : (g.date ? Date.parse(g.date) : null));
      const tsMs = (ts && ts < 1e12) ? ts * 1000 : ts; // convert seconds->ms when needed
      const iso = tsMs ? new Date(tsMs).toISOString() : (g.Date || g.date || null);
      return {
        Date: g.Date || g.date || (iso ? iso.split('T')[0] : null),
        date: iso || null,
        DateTimestamp: tsMs || null,
        time: g.time || (iso ? new Date(iso).toISOString().split('T')[1] : null),
        competitors: g.competitors || (g.home && g.away ? [{ name: g.home }, { name: g.away }] : g.competitors) || [],
        gameId: g.gameId || g.id || `valuebet_game_${Math.random().toString(36).slice(2,9)}`,
        marketName: g.marketName || g.market_name || g.market || g.market_id || '',
        odds: (typeof g.odds === 'number' ? g.odds : (g.valueBetOdds || g.odds_value || parseFloat(g.odds || 0))) || 0,
        selectionName: g.selectionName || g.outcome_name || g.selection || g.outcome || '',
        raw: g
      };
    }) : [];

    const totalOdds = prediction.totalOdds || selectedOdds.reduce((acc, s) => acc * (s.odds || 1), 1) || 1;
    const price = prediction.price || prediction.predictionPrice || 10;
    const normalized = {
      ...prediction,
      predictionId: predId,
      predictionName: prediction.predictionName || prediction.name || `ValueBet ${predId}`,
      predictiontype: prediction.predictiontype || 'value_bet',
      price,
      totalOdds,
      numGames: selectedOdds.length,
      selectedOdds,
      games: selectedOdds,
      potentialWin: price * totalOdds,
      createdAt: prediction.createdAt || new Date().toISOString(),
      smsString: prediction.smsString || null,
      isValueBet: true,
      originalStrategy: 'value_bets'
    };
    return normalized;
  };

  // Ensure a readable string message from various bridge/server response objects
  normalizeResponseMessage = (res) => {
    try {
      if (!res) return '';
      if (typeof res === 'string') return res;
      if (res.message && typeof res.message === 'string') return res.message;
      if (res.prediction_result && typeof res.prediction_result === 'object') {
        if (res.prediction_result.message) return String(res.prediction_result.message);
        return JSON.stringify(res.prediction_result);
      }
      if (res.confirmation && res.confirmation.message) return String(res.confirmation.message);
      // fallback to JSON string
      return JSON.stringify(res);
    } catch (e) {
      return String(res);
    }
  };

  

  // Adapted from Betlist: handle navigation for details
  handleBuyClick = (prediction) => {
    if (!prediction) return;
    // Use the central normalizer so shape is consistent with copy/send actions
    const normalized = this.normalizePrediction(prediction);

    // Determine instance id for UI state (keep consistent with other handlers)
    const instanceId = prediction._instanceId || (prediction.predictionId || prediction.id || prediction._localId);

    // Mark as visited (use instance id)
    if (instanceId) {
      this.setState(prev => {
        const visitedNext = { ...(prev.visitedMap || {}), [instanceId]: true };
        this.persistVisited(visitedNext);
        return { visitedMap: visitedNext };
      });
    }

    // Navigate to prediction details page with the normalized object
    if (this.props.navigate) {
      this.props.navigate('/prediction-details', {
        state: { prediction: normalized }
      });
    } else {
      // Fallback for development
      alert(`Navigation not available. Prediction: ${normalized.predictionName || 'Value Bet Prediction'}`);
    }
  };

  handleCopySms = async (prediction) => {
    if (!prediction) return;
  const instanceId = prediction._instanceId || (prediction.predictionId || prediction.id || prediction._localId);
  if (!(prediction.predictionId || prediction.id || prediction._localId)) return;
    if (this.state.copyingMap[instanceId]) return;
    this.setState(prev => ({ copyingMap: { ...(prev.copyingMap||{}), [instanceId]: true } }));
    try {
      // Prefer sending the original/raw prediction object to the backend so saved raw metadata (e.g. raw.smsCode)
      // can be used as a fallback when provider mapping fails. Fallback to normalized shape if raw not available.
  const normalized = this.normalizePrediction(prediction);
  const rawToSend = prediction._rawPrediction || enhancePredictionWithFriendlyNames(prediction) || prediction || normalized;
      let sms = normalized.smsString || prediction.smsString || rawToSend.smsString;
      if (!sms) {
        const r = await fetch('http://localhost:5000/api/createSmsBets', { method: 'POST', headers: { 'Content-Type': 'application/json' }, body: JSON.stringify({ predictions: [rawToSend], stake: 1 }) });
        if (!r.ok) throw new Error('createSmsBets failed');
  const d = await r.json(); sms = d.smsBets && d.smsBets[0] && d.smsBets[0].smsString;
        if (!sms) throw new Error('No smsString returned');
        // persist smsString back into state only for this instance
        this.setState(prev => {
          const list = (prev.valueBetsPredictions || []).map(p => ((p._instanceId || (p.predictionId || p.id || p._localId)) === instanceId) ? { ...p, smsString: sms } : p);
          return { valueBetsPredictions: list };
        });
      }

      if (navigator.clipboard && navigator.clipboard.writeText) await navigator.clipboard.writeText(sms);
      else { const ta = document.createElement('textarea'); ta.value = sms; document.body.appendChild(ta); ta.select(); document.execCommand('copy'); document.body.removeChild(ta); }
  const rep = { status: 'success', message: sms, timestamp: new Date().toISOString() };
  this.setState(prev => { const reports = { ...(prev.reports||{}), [instanceId]: rep }; this.persistReports(reports); return { reports }; });
  const visited = { ...(this.state.visitedMap||{}), [instanceId]: true }; this.setState({ visitedMap: visited }); this.persistVisited(visited);
    } catch (err) {
      const msg = err && err.message ? err.message : String(err);
  const rep = { status: 'error', message: msg, timestamp: new Date().toISOString() };
  this.setState(prev => { const reports = { ...(prev.reports||{}), [instanceId]: rep }; this.persistReports(reports); return { reports }; });
      console.error('copy failed', err);
      alert('Failed to copy SMS: ' + msg);
    } finally {
      this.setState(prev => { const cm = { ...(prev.copyingMap||{}) }; delete cm[instanceId]; return { copyingMap: cm }; });
    }
  };

  handleSendTo79079 = async (prediction) => {
    if (!prediction) return;
  const instanceId = prediction._instanceId || (prediction.predictionId || prediction.id || prediction._localId);
  if (!(prediction.predictionId || prediction.id || prediction._localId)) return;
  try {
  // Prefer sending the original/raw prediction object so backend can use saved raw metadata
  const normalized = this.normalizePrediction(prediction);
  const rawToSend = prediction._rawPrediction || enhancePredictionWithFriendlyNames(prediction) || prediction || normalized;
      let sms = normalized.smsString || prediction.smsString || rawToSend.smsString;
      if (!sms) {
        const r = await fetch('http://localhost:5000/api/createSmsBets', { method: 'POST', headers: { 'Content-Type': 'application/json' }, body: JSON.stringify({ predictions: [rawToSend], stake: 1 }) });
        if (!r.ok) throw new Error('createSmsBets failed');
        const d = await r.json(); sms = d.smsBets && d.smsBets[0] && d.smsBets[0].smsString;
        if (sms) {
          this.setState(prev => {
            const list = (prev.valueBetsPredictions || []).map(p => ((p._instanceId || (p.predictionId || p.id || p._localId)) === instanceId) ? { ...p, smsString: sms } : p);
            return { valueBetsPredictions: list };
          });
        }
      }
      if (!sms) throw new Error('No SMS available');
      const url = `sms:79079?body=${encodeURIComponent(sms)}`;
      try { window.location.href = url; } catch (e) { const a = document.createElement('a'); a.href = url; document.body.appendChild(a); a.click(); document.body.removeChild(a); }
      const rep = { status: 'success', message: 'Opened composer', timestamp: new Date().toISOString() };
      this.setState(prev => { const reports = { ...(prev.reports||{}), [instanceId]: rep }; this.persistReports(reports); return { reports }; });
    } catch (err) { const msg = err && err.message ? err.message : String(err); alert('Send failed: ' + msg); }
  };

  handleProgrammaticSend = async (prediction) => {
    if (!prediction) return;
    const instanceId = prediction._instanceId || (prediction.predictionId || prediction.id || prediction._localId);
    const id = prediction.predictionId || prediction.id || prediction._localId;
    if (!id) return;
    if (this.state.sendingMap[instanceId]) return;
    this.setState(prev => ({ sendingMap: { ...(prev.sendingMap||{}), [instanceId]: true } }));
    try {
      // Prefer the original/raw prediction when asking backend to build SMS (so raw.smsCode can be used)
  const normalized = this.normalizePrediction(prediction);
  const rawToSend = prediction._rawPrediction || enhancePredictionWithFriendlyNames(prediction) || prediction || normalized;
      let sms = normalized.smsString || prediction.smsString || rawToSend.smsString;
      if (!sms) {
        const r = await fetch('http://localhost:5000/api/createSmsBets', { method: 'POST', headers: { 'Content-Type': 'application/json' }, body: JSON.stringify({ predictions: [rawToSend], stake: 1 }) });
        if (!r.ok) throw new Error('createSmsBets failed');
        const d = await r.json(); sms = d.smsBets && d.smsBets[0] && d.smsBets[0].smsString;
        if (sms) {
          this.setState(prev => {
            const list = (prev.valueBetsPredictions || []).map(p => ((p._instanceId || (p.predictionId || p.id || p._localId)) === instanceId) ? { ...p, smsString: sms } : p);
            return { valueBetsPredictions: list };
          });
        }
      }
      if (!sms) throw new Error('No SMS to send');
  const controller = new AbortController(); const TIMEOUT = 40000; const tid = setTimeout(() => { try { controller.abort(); } catch(e){} }, TIMEOUT);
      const resp = await fetch('http://localhost:5021/send', { method: 'POST', headers: { 'Content-Type': 'application/json' }, body: JSON.stringify({ to: '79079', message: sms }), signal: controller.signal });
      clearTimeout(tid);
      if (!resp.ok) { const txt = await resp.text().catch(() => ''); throw new Error('Bridge error: ' + txt); }
      const jr = await resp.json().catch(() => null);
      if (jr && jr.confirmation && jr.confirmation.confirmed) {
        const rep = { status: 'success', message: `Confirmed ${jr.confirmation.betId||'?'}`, timestamp: new Date().toISOString() };
        this.setState(prev => {
          const reports = { ...(prev.reports||{}), [instanceId]: rep };
          const confirmed = { ...(prev.confirmedMap||{}), [instanceId]: true };
          this.persistReports(reports);
          this.persistConfirmed(confirmed);
          return { reports, confirmedMap: confirmed, valueBetsPredictions: (prev.valueBetsPredictions||[]).filter(p => (p._instanceId || (p.predictionId||p.id||p._localId)) !== instanceId) };
        });
        fetch('http://localhost:5000/api/confirmValueBetsPrediction', { method: 'POST', headers: { 'Content-Type': 'application/json' }, body: JSON.stringify({ predictionId: id }) }).catch(()=>{});
      } else {
        const readable = this.normalizeResponseMessage(jr || 'sent');
        const rep = { status: 'sent', message: readable, timestamp: new Date().toISOString() };
        this.setState(prev => { const reports = { ...(prev.reports||{}), [instanceId]: rep }; this.persistReports(reports); return { reports }; });
      }
    } catch (err) {
      const msg = err && err.message ? err.message : String(err);
      const rep = { status: 'error', message: msg, timestamp: new Date().toISOString() };
      this.setState(prev => { const reports = { ...(prev.reports||{}), [instanceId]: rep }; this.persistReports(reports); return { reports }; });
    } finally {
      this.setState(prev => { const sm = { ...(prev.sendingMap||{}) }; delete sm[instanceId]; return { sendingMap: sm }; });
    }
  };

  handleConfirmSuccess = (prediction) => {
    if (!prediction) return;
    const instanceId = prediction._instanceId || (prediction.predictionId || prediction.id || prediction._localId);
    const id = prediction.predictionId || prediction.id || prediction._localId;
    if (!id) return;
    const rep = { status: 'confirmed_by_user', message: 'User confirmed', timestamp: new Date().toISOString() };
    this.setState(prev => { const reports = { ...(prev.reports||{}), [instanceId]: rep }; const confirmed = { ...(prev.confirmedMap||{}), [instanceId]: true }; this.persistReports(reports); this.persistConfirmed(confirmed); return { reports, confirmedMap: confirmed, valueBetsPredictions: (prev.valueBetsPredictions||[]).filter(p => (p._instanceId || (p.predictionId||p.id||p._localId)) !== instanceId) }; });
    fetch('http://localhost:5000/api/confirmValueBetsPrediction', { method: 'POST', headers: { 'Content-Type': 'application/json' }, body: JSON.stringify({ predictionId: id }) }).catch(()=>{});
  };

  handleReportError = (prediction) => {
    if (!prediction) return;
    const instanceId = prediction._instanceId || (prediction.predictionId || prediction.id || prediction._localId);
    const id = prediction.predictionId || prediction.id || prediction._localId;
    if (!id) return;
    const rep = { status: 'reported_error', message: 'Reported by user', timestamp: new Date().toISOString() };
    this.setState(prev => { const reports = { ...(prev.reports||{}), [instanceId]: rep }; this.persistReports(reports); return { reports }; });
    fetch('http://localhost:5000/api/markFailedValueBetsPrediction', { method: 'POST', headers: { 'Content-Type': 'application/json' }, body: JSON.stringify({ predictionId: id }) }).catch(()=>{});
  };

  computeReportCounts = () => {
    const { reports } = this.state;
    let success = 0;
    let error = 0;
    if (reports && typeof reports === 'object') {
      Object.values(reports).forEach(r => {
        if (!r || !r.status) return;
        if (r.status === 'success') success += 1;
        else if (r.status === 'error') error += 1;
      });
    }
    return { success, error, total: success + error };
  };

  handleExportReports = () => {
    try {
      const data = JSON.stringify(this.state.reports || {}, null, 2);
      const blob = new Blob([data], { type: 'application/json' });
      const url = URL.createObjectURL(blob);
      const a = document.createElement('a');
      a.href = url;
      a.download = `valuebet_prediction_reports_${new Date().toISOString().replace(/[:.]/g,'-')}.json`;
      document.body.appendChild(a);
      a.click();
      document.body.removeChild(a);
      URL.revokeObjectURL(url);
    } catch (e) {
      console.error('Failed to export reports', e);
      alert('Failed to export reports. See console for details.');
    }
  };

  handleUploadReports = async () => {
    try {
      const body = { reports: this.state.reports || {} };
      const resp = await fetch('http://localhost:5000/api/valueBetsPredictionReports', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(body)
      });
      if (!resp.ok) {
        const text = await resp.text().catch(() => '');
        throw new Error(`${resp.status} ${resp.statusText} ${text}`);
      }
      alert('Value bets prediction reports uploaded successfully.');
    } catch (err) {
      console.error('Failed to upload value bets reports', err);
      alert('Failed to upload reports: ' + (err.message || String(err)));
    }
  };

  // Adapted from Betlist: Emergency stop for autobet
  emergencyStopAutobet = (options = {}) => {
    this.setState({ autobetRunning: false, autobetStats: null });
    if (!options.silent) {
      alert('AutoBet stopped by user.');
    }
  };

  // Adapted from Betlist: Run auto-bet on all unconfirmed value bets
  runAutoBetAll = async (options = {}) => {
    const delayMs = options.delayMs || 1200;
    const targetSuccesses = this.state.targetSuccessfulBets || 5;
    const predictions = this.state.valueBetsPredictions || [];
    const unconfirmedPreds = predictions.filter(p => {
      const id = p.predictionId || p.id || p._localId;
      return !this.state.confirmedMap[id];
    });

    if (unconfirmedPreds.length === 0) {
      alert('No unconfirmed value bet predictions to auto-bet.');
      return;
    }

    this.setState({ 
      autobetRunning: true, 
      autobetStats: { 
        total: unconfirmedPreds.length, 
        remaining: unconfirmedPreds.length, 
        successes: 0, 
        failures: 0, 
        target: targetSuccesses, 
        placedStakeTotal: 0 
      } 
    });

    let successes = 0;
    let failures = 0;
    let placedStakeTotal = 0;

    for (let i = 0; i < unconfirmedPreds.length; i++) {
      if (!this.state.autobetRunning) {
        this.emergencyStopAutobet({ silent: true });
        break;
      }
      if (successes >= targetSuccesses) {
        alert(`AutoBet reached target of ${targetSuccesses} successful bets.`);
        this.setState({ autobetRunning: false });
        break;
      }
      const pred = unconfirmedPreds[i];
      const predId = pred.predictionId || pred.id || pred._localId;

      try {
        // Use the existing programmatic send logic (it handles SMS creation and sending)
        await this.handleProgrammaticSend(pred);
        // Note: handleProgrammaticSend updates state for reports/visited/confirmed/removal internally

        const stake = this.parseStakeFromSms(pred.smsString || '');
        placedStakeTotal += stake;

        successes++;
      } catch (err) {
        failures++;
        console.error('AutoBet failed for prediction', predId, err);
      }

      // Update stats
      try {
        const prevStats = this.state.autobetStats || {};
        const newStats = {
          ...prevStats,
          successes,
          failures,
          remaining: unconfirmedPreds.length - (i + 1),
          placedStakeTotal
        };
        this.setState({ autobetStats: newStats });
      } catch (e) {
        // ignore if state unavailable
      }

      if (i < unconfirmedPreds.length - 1) {
        await new Promise(resolve => setTimeout(resolve, delayMs));
      }
    }

    this.setState({ autobetRunning: false });
    alert(`AutoBet completed: ${successes} successes, ${failures} failures.`);
  };

  renderReportsPanel = () => {
    const { reports } = this.state;
    const keys = reports
      ? Object.keys(reports).filter(k => reports[k] && typeof reports[k] === 'object' && reports[k].status)
      : [];
    if (!keys.length) return null;
    return (
      <div style={{ marginTop: 20, padding: 12, border: '1px solid #ddd', borderRadius: 6 }}>
        <h4 style={{ marginTop: 0 }}>Value Bets Prediction Copy Reports</h4>
        <ul style={{ paddingLeft: 18 }}>
          {keys.map(k => {
            const r = reports[k];
            return (
              <li key={k} style={{ marginBottom: 8 }}>
                <strong>{k}</strong> — {r.status === 'success' ? <span style={{ color: '#0a7' }}>Success</span> : <span style={{ color: '#c00' }}>Error</span>}
                <div style={{ fontSize: 12, color: '#444' }}>{r.message}</div>
                <div style={{ fontSize: 11, color: '#666' }}>{new Date(r.timestamp).toLocaleString()}</div>
              </li>
            );
          })}
        </ul>
        <button onClick={this.handleExportReports} style={{ padding: '8px 16px', marginRight: 8, cursor: 'pointer' }}>Export Reports</button>
        <button onClick={this.handleUploadReports} style={{ padding: '8px 16px', cursor: 'pointer' }}>Upload Reports to Server</button>
      </div>
    );
  };

  // Adapted renderPredictionStatus from Betlist
  // use shared helper: enhancePredictionWithFriendlyNames
  enhancePredictionWithFriendlyNames = (prediction) => {
    if (!prediction) return prediction;
    // Work on a deep clone to avoid mutating state
    let pred = JSON.parse(JSON.stringify(prediction));
    // If this is a lightweight preview, prefer normalized version
    if (pred._rawPrediction && typeof pred._rawPrediction === 'object') {
      pred = JSON.parse(JSON.stringify(pred._rawPrediction));
    }
    // Normalize selectedOdds array
    const games = Array.isArray(pred.selectedOdds) ? pred.selectedOdds : (pred.games || []);
    const enhanced = (games || []).map((g) => {
      const o = JSON.parse(JSON.stringify(g));
      // Normalize marketName '1X2' variants to '3 Way'
      if (o.marketName && String(o.marketName).toLowerCase().replace(/\s+/g, '') === '1x2') {
        o.marketName = '3 Way';
      }

      // If selectionName is sms-style token like 'gameId#selectionId', attempt to resolve
      const sel = String(o.selectionName || '');
      const smsToken = sel.match(/^(\d+)#(\d+)$/);
      const comps = Array.isArray(o.competitors) && o.competitors.length ? o.competitors : (Array.isArray(pred.competitors) ? pred.competitors : []);
      if (smsToken) {
        const selId = smsToken[2];
        // Try to match competitor id fields if available
        let matched = null;
        for (const c of comps) {
          if (!c) continue;
          if (String(c.id) === selId || String(c.selectionId) === selId || String(c.smsId) === selId || String(c.ID) === selId) { matched = c; break; }
        }
        if (matched && matched.name) {
          o.selectionName = matched.name;
        } else if (comps && comps.length >= 2) {
          // If we couldn't match by id, heuristically decide: if selId ends with even/odd or equals 1/2
          if (selId === '1') o.selectionName = comps[0].name || 'Home';
          else if (selId === '2') o.selectionName = comps[1].name || 'Away';
          else {
            // default to first competitor name (home) to avoid leaving token
            o.selectionName = comps[0].name || 'Home';
          }
        }
      } else {
        // If selectionName is simple '1'|'2'|'X', map to competitor names when available
        const normalizedSel = sel.trim().toUpperCase();
        if (['1', '2', 'X'].includes(normalizedSel) && comps && comps.length >= 2) {
          if (normalizedSel === '1') o.selectionName = comps[0].name || o.selectionName;
          else if (normalizedSel === '2') o.selectionName = comps[1].name || o.selectionName;
          else o.selectionName = 'Draw';
        }
      }

      return o;
    });

    // Attach enhanced selectedOdds back
    pred.selectedOdds = enhanced;
    // Keep original fields like predictionId, price, etc.
    return pred;
  }
  
  render() {
    const { valueBetsPredictions, loading, copyingMap, visitedMap, reports, confirmedMap } = this.state;
    const reportCounts = this.computeReportCounts();

    return (
      <div style={{ padding: 20 }}>
        <div style={{ display: 'flex', gap: '16px', marginBottom: 16, flexWrap: 'wrap' }}>
          <div style={{ padding: '12px 20px', backgroundColor: '#f0f8ff', borderRadius: '6px', border: '1px solid #4a90e2' }}>
            <div style={{ fontSize: '24px', fontWeight: 'bold', color: '#4a90e2' }}>{valueBetsPredictions.length}</div>
            <div style={{ fontSize: '12px', color: '#666' }}>Total Predictions</div>
          </div>
          <div style={{ padding: '12px 20px', backgroundColor: '#f0fff0', borderRadius: '6px', border: '1px solid #28a745' }}>
            <div style={{ fontSize: '24px', fontWeight: 'bold', color: '#28a745' }}>
              {valueBetsPredictions.length > 0
                ? (valueBetsPredictions.reduce((sum, p) => sum + (p.totalOdds || 0), 0) / valueBetsPredictions.length).toFixed(2)
                : '0.00'}
            </div>
            <div style={{ fontSize: '12px', color: '#666' }}>Avg Odds</div>
          </div>
          <div style={{ padding: '12px 20px', backgroundColor: '#fff0f0', borderRadius: '6px', border: '1px solid #dc3545' }}>
            <div style={{ fontSize: '24px', fontWeight: 'bold', color: '#dc3545' }}>{Object.keys(confirmedMap).length}</div>
            <div style={{ fontSize: '12px', color: '#666' }}>Confirmed</div>
          </div>
          <div style={{ padding: '12px 20px', backgroundColor: '#f8f9fa', borderRadius: '6px', border: '1px solid #6c757d' }}>
            <div style={{ fontSize: '24px', fontWeight: 'bold', color: '#6c757d' }}>
              {reportCounts.success}/{reportCounts.total}
            </div>
            <div style={{ fontSize: '12px', color: '#666' }}>Success/Total</div>
          </div>
        </div>

        {loading ? (
          <div style={{ textAlign: 'center', padding: '40px' }}>
            <div style={{ fontSize: '18px', color: '#666' }}>Loading value bets predictions...</div>
          </div>
        ) : valueBetsPredictions.length === 0 ? (
          <div style={{ textAlign: 'center', padding: '40px', backgroundColor: '#f8f9fa', borderRadius: '8px' }}>
            <div style={{ fontSize: '48px', marginBottom: '16px' }}>📭</div>
            <div style={{ fontSize: '18px', color: '#666', marginBottom: '8px' }}>No value bets predictions available</div>
            <div style={{ fontSize: '14px', color: '#999' }}>Run the value bets matching and prediction creation workflow to generate predictions</div>
          </div>
        ) : (
          <ol style={{ paddingLeft: '20px' }}>
            {(() => {
              const idCounts = {};
              (valueBetsPredictions || []).forEach((p, i) => {
                const base = p.predictionId || p.id || p._localId || `valuebet-${i}`;
                idCounts[base] = (idCounts[base] || 0) + 1;
              });
              const items = (valueBetsPredictions || []).map((prediction, idx) => {
                const baseId = prediction._localId || prediction.predictionId || prediction.id || `valuebet-${idx}`;
                const uniqueId = (idCounts[baseId] && idCounts[baseId] > 1) ? `${baseId}-${idx}` : baseId;
                const copying = !!(copyingMap && copyingMap[baseId]);
                const visited = !!(visitedMap && visitedMap[baseId]);
                const confirmed = !!(confirmedMap && confirmedMap[baseId]);
                const report = (reports || {})[baseId];
                return (
                  <li key={uniqueId} style={{
                    marginBottom: 12,
                    backgroundColor: report && report.status === 'error' ? '#ffeaea' : '#f8f9fa',
                    padding: '10px',
                    borderRadius: '6px',
                    border: report && report.status === 'error' ? '2px solid #c00' : '2px solid #ffd700'
                  }}>
                    <div style={{ display: 'flex', alignItems: 'center', gap: 12 }}>
                      <div style={{ flex: 1 }}>
                        <strong style={{ color: report && report.status === 'error' ? '#c00' : '#ff9900' }}>
                          💎 {prediction.predictionName || `Value Prediction ${idx + 1}`}
                        </strong>
                        <div style={{ fontSize: 13, color: '#555' }}>
                          {prediction.numGames ? `${prediction.numGames} game(s)` : ''}
                          {prediction.totalOdds ? ` — Odds: ${prediction.totalOdds.toFixed(2)}` : ''}
                          {prediction.price ? ` — Stake: KES ${prediction.price}` : ''}
                          {prediction.potentialWin ? ` — Win: KES ${prediction.potentialWin.toFixed(2)}` : ''}
                        </div>
                        {prediction.smsString && (
                          <div style={{ fontSize: 11, color: '#888', marginTop: 4, fontFamily: 'monospace' }}>
                            SMS: {prediction.smsString}
                          </div>
                        )}
                        {confirmed && (
                          <div style={{ color: '#28a745', fontWeight: 'bold', marginTop: 6, fontSize: 12 }}>
                            ✅ Confirmed - Bet Placed
                          </div>
                        )}
                        {this.renderPredictionStatus(prediction)}
                      </div>
                      <div style={{ display: 'flex', gap: 8, flexWrap: 'wrap' }}>
                        <button
                          onClick={() => this.handleBuyClick(prediction)}
                          style={{
                            backgroundColor: '#007bff',
                            color: 'white',
                            border: 'none',
                            padding: '6px 12px',
                            borderRadius: '4px',
                            cursor: 'pointer',
                            fontWeight: 'bold'
                          }}
                        >
                          View Details
                        </button>
                        <button
                          onClick={() => this.handleCopySms(prediction)}
                          disabled={copying || confirmed}
                          style={{
                            backgroundColor: copying ? '#6c757d' : '#28a745',
                            color: 'white',
                            border: 'none',
                            padding: '6px 12px',
                            borderRadius: '4px',
                            cursor: copying || confirmed ? 'not-allowed' : 'pointer',
                            opacity: copying || confirmed ? 0.6 : 1
                          }}
                        >
                          {copying ? 'Copying...' : (visited ? 'Copy SMS (visited)' : 'Copy SMS')}
                        </button>
                        <button
                          onClick={() => this.handleConfirmSuccess(prediction)}
                          disabled={confirmed}
                          style={{
                            backgroundColor: confirmed ? '#6c757d' : '#ffc107',
                            color: confirmed ? 'white' : '#000',
                            border: 'none',
                            padding: '6px 12px',
                            borderRadius: '4px',
                            cursor: confirmed ? 'not-allowed' : 'pointer',
                            opacity: confirmed ? 0.6 : 1
                          }}
                        >
                          {confirmed ? 'Confirmed ✓' : 'Confirm'}
                        </button>
                        <button
                          onClick={() => this.handleSendTo79079(prediction)}
                          disabled={confirmed}
                          style={{
                            backgroundColor: '#6f42c1',
                            color: 'white',
                            border: 'none',
                            padding: '6px 12px',
                            borderRadius: '4px',
                            cursor: confirmed ? 'not-allowed' : 'pointer',
                            opacity: confirmed ? 0.6 : 1
                          }}
                        >
                          Send to 79079
                        </button>
                        <button
                          onClick={() => this.handleProgrammaticSend(prediction)}
                          disabled={confirmed || (this.state.sendingMap && this.state.sendingMap[baseId])}
                          style={{
                            backgroundColor: (this.state.sendingMap && this.state.sendingMap[baseId]) ? '#6c757d' : '#e83e8c',
                            color: 'white',
                            border: 'none',
                            padding: '6px 12px',
                            borderRadius: '4px',
                            cursor: (confirmed || (this.state.sendingMap && this.state.sendingMap[baseId])) ? 'not-allowed' : 'pointer',
                            opacity: (confirmed || (this.state.sendingMap && this.state.sendingMap[baseId])) ? 0.6 : 1
                          }}
                        >
                          {(this.state.sendingMap && this.state.sendingMap[baseId]) ? 'Sending...' : 'Send via Device'}
                        </button>
                        <button
                          onClick={() => this.handleReportError(prediction)}
                          style={{
                            backgroundColor: '#dc3545',
                            color: 'white',
                            border: 'none',
                            padding: '6px 12px',
                            borderRadius: '4px',
                            cursor: 'pointer'
                          }}
                        >
                          Report
                        </button>
                      </div>
                    </div>
                  </li>
                );
              });
              return items;
            })()}
          </ol>
        )}

        {this.renderReportsPanel()}
      </div>
    );
  }
}

// Wrapper to provide navigate (like in Betlist)
const ValueBetsPredictionsListWithNavigation = (props) => {
  const navigate = useNavigate();
  return <ValueBetsPredictionsList {...props} navigate={navigate} />;
};

export default ValueBetsPredictionsListWithNavigation;