import React, { useState, useEffect, useMemo } from 'react';
import axios from 'axios';
import './ValueBets.css';

const ValueBets = () => {
  const [data, setData] = useState(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);
  const [view, setView] = useState('games'); // 'games' or 'teams'
  const [sortBy, setSortBy] = useState('valuePercent'); // 'valuePercent', 'odds', 'probability'
  const [filterMinValue, setFilterMinValue] = useState(0);
  const [filterMinOdds, setFilterMinOdds] = useState(1.0);
  const [filterMaxOdds, setFilterMaxOdds] = useState(10.0);
  const [searchTeam, setSearchTeam] = useState('');
  const [showRepeatingOnly, setShowRepeatingOnly] = useState(false);
  const [lastUpdated, setLastUpdated] = useState(null);
  const [selectedQuality, setSelectedQuality] = useState('all'); // 'all', 'high', 'medium', 'poor'
  const [cleaningGames, setCleaningGames] = useState(false);
  const [cleanupResult, setCleanupResult] = useState(null);

  // Fetch data from Firebase via backend API
  const fetchValueBets = async () => {
    try {
      setLoading(true);
      setError(null);
      const response = await axios.get('http://localhost:5000/api/getLatestValueBets');
      
      if (response.data.success) {
        // Transform the data to match the expected format
        const transformedData = {
          games: response.data.games,
          summary: response.data.summary,
          scrapedAt: response.data.games[0]?.scrapedAt || new Date().toISOString()
        };
        setData(transformedData);
        setLastUpdated(new Date());
      } else {
        setError('No value bets found');
      }
    } catch (err) {
      console.error('Failed to load data from Firebase:', err);
      setError('Failed to connect to backend. Make sure the server is running.');
      
      // Fallback to local JSON file if Firebase fails
      try {
        const res = await fetch('/football_data_homewin_1x2.json');
        const jsonData = await res.json();
        setData(jsonData);
        setLastUpdated(new Date(jsonData.scrapedAt));
        setError('Loaded from local file (backend unavailable)');
      } catch (fallbackErr) {
        console.error('Fallback also failed:', fallbackErr);
      }
    } finally {
      setLoading(false);
    }
  };

  // Clean outdated games from database
  const cleanOutdatedGames = async () => {
    try {
      setCleaningGames(true);
      setCleanupResult(null);
      setError(null);
      
      const response = await axios.post('http://localhost:5000/api/cleanOutdatedGames');
      
      if (response.data.success) {
        setCleanupResult({
          deleted: response.data.deleted,
          remaining: response.data.remaining,
          message: response.data.message
        });
        
        // Show success message
        alert(`✅ Cleanup Complete!\n\n${response.data.message}\n\nDeleted: ${response.data.deleted} outdated games\nRemaining: ${response.data.remaining} future games`);
      } else {
        setError('Failed to clean outdated games');
      }
    } catch (err) {
      console.error('Error cleaning outdated games:', err);
      setError('Failed to clean outdated games: ' + err.message);
      alert('❌ Error: ' + err.message);
    } finally {
      setCleaningGames(false);
    }
  };

  useEffect(() => {
    fetchValueBets();
  }, []);

  // Filter and clean games data
  const cleanedGames = useMemo(() => {
    if (!data || !data.games) return [];
    
    return data.games.filter(game => {
      // Remove invalid/noise data
      if (!game.homeTeam || !game.awayTeam) return false;
      if (game.homeTeam.includes('window.') || game.homeTeam.includes('function')) return false;
      if (game.homeTeam.includes('.lf-') || game.homeTeam.includes('webkit')) return false;
      if (game.homeTeam === 'Fixtures' || game.homeTeam === 'Alerts') return false;
      if (game.homeTeam.length < 3 || game.awayTeam.length < 3) return false;
      
      // Must have valid odds
      const odds = parseFloat(game.odds);
      if (!game.odds || isNaN(odds) || odds < 1.0) return false;
      
      // Apply filters
      if (odds < filterMinOdds || odds > filterMaxOdds) return false;
      const valuePercent = parseFloat(game.valuePercent);
      if (game.valuePercent && !isNaN(valuePercent) && valuePercent < filterMinValue) return false;
      
      // Quality filter
      if (selectedQuality !== 'all' && game.quality) {
        if (selectedQuality.toLowerCase() !== game.quality.toLowerCase()) {
          return false;
        }
      }
      
      // Search filter
      if (searchTeam) {
        const search = searchTeam.toLowerCase();
        if (!game.homeTeam.toLowerCase().includes(search) && 
            !game.awayTeam.toLowerCase().includes(search)) {
          return false;
        }
      }
      
      return true;
    });
  }, [data, filterMinOdds, filterMaxOdds, filterMinValue, searchTeam, selectedQuality]);

  // Sort games
  const sortedGames = useMemo(() => {
    const games = [...cleanedGames];
    
    games.sort((a, b) => {
      switch(sortBy) {
        case 'valuePercent':
          return parseFloat(b.valuePercent || 0) - parseFloat(a.valuePercent || 0);
        case 'odds':
          return parseFloat(b.odds || 0) - parseFloat(a.odds || 0);
        case 'probability':
          return parseFloat(b.probability || 0) - parseFloat(a.probability || 0);
        case 'date':
          return new Date(b.timestamp) - new Date(a.timestamp);
        default:
          return 0;
      }
    });
    
    return games;
  }, [cleanedGames, sortBy]);

  // Group games by quality
  const gamesByQuality = useMemo(() => {
    const groups = {
      high: [],
      medium: [],
      poor: [],
      unrated: []
    };

    sortedGames.forEach(game => {
      const quality = game.quality ? game.quality.toLowerCase() : 'unrated';
      if (groups[quality]) {
        groups[quality].push(game);
      } else {
        groups.unrated.push(game);
      }
    });

    return groups;
  }, [sortedGames]);

  // Filter teams
  const filteredTeams = useMemo(() => {
    if (!data || !data.repeatingTeams) return [];
    
    let teams = data.repeatingTeams;
    
    if (showRepeatingOnly) {
      teams = teams.filter(t => t.appearances > 1);
    }
    
    if (searchTeam) {
      const search = searchTeam.toLowerCase();
      teams = teams.filter(t => t.team.toLowerCase().includes(search));
    }
    
    return teams;
  }, [data, showRepeatingOnly, searchTeam]);

  if (loading) {
    return (
      <div className="value-bets-container">
        <div className="loading">
          <div className="spinner"></div>
          <p>Loading Value Bets Data from Firebase...</p>
        </div>
      </div>
    );
  }

  if (!data) {
    return (
      <div className="value-bets-container">
        <div className="error">
          <h2>⚠️ Error</h2>
          <p>{error || 'Failed to load data'}</p>
          <button onClick={fetchValueBets} className="retry-button">
            🔄 Retry
          </button>
        </div>
      </div>
    );
  }

  return (
    <div className="value-bets-container">
      {/* Header */}
      <div className="value-bets-header">
        <div className="header-title">
          <h1>⚽ Football Value Bets - Home Wins (1x2)</h1>
          <div className="header-buttons">
            <button onClick={fetchValueBets} className="refresh-button" disabled={loading}>
              🔄 {loading ? 'Refreshing...' : 'Refresh from Firebase'}
            </button>
            <button 
              onClick={cleanOutdatedGames} 
              className="cleanup-button" 
              disabled={cleaningGames}
              title="Remove outdated games from database"
            >
              🧹 {cleaningGames ? 'Cleaning...' : 'Clean Outdated Games'}
            </button>
          </div>
        </div>
        {error && <div className="warning-banner">⚠️ {error}</div>}
        {cleanupResult && (
          <div className="cleanup-result">
            ✅ {cleanupResult.message}
          </div>
        )}
        {lastUpdated && (
          <div className="last-updated">
            Last updated: {lastUpdated.toLocaleString()}
          </div>
        )}
        <div className="header-stats">
          <div className="stat-box">
            <div className="stat-label">Data Source</div>
            <div className="stat-value">🔥 Firebase</div>
          </div>
          <div className="stat-box">
            <div className="stat-label">Total Games</div>
            <div className="stat-value">{data.summary?.totalGames || data.games?.length || 0}</div>
          </div>
          <div className="stat-box">
            <div className="stat-label">Unique Teams</div>
            <div className="stat-value">{data.summary?.uniqueTeams || 0}</div>
          </div>
          <div className="stat-box">
            <div className="stat-label">Cleaned Games</div>
            <div className="stat-value">{cleanedGames.length}</div>
          </div>
          {data.summary?.avgOdds && (
            <div className="stat-box highlight">
              <div className="stat-label">Avg Odds</div>
              <div className="stat-value">{data.summary.avgOdds}</div>
            </div>
          )}
          {data.summary?.avgValuePercent && (
            <div className="stat-box highlight">
              <div className="stat-label">Avg Value</div>
              <div className="stat-value">{data.summary.avgValuePercent}%</div>
            </div>
          )}
        </div>
      </div>

      {/* Controls */}
      <div className="value-bets-controls">
        <div className="view-toggle">
          <button 
            className={view === 'games' ? 'active' : ''}
            onClick={() => setView('games')}
          >
            📋 Games ({sortedGames.length})
          </button>
          <button 
            className={view === 'teams' ? 'active' : ''}
            onClick={() => setView('teams')}
          >
            👥 Teams ({filteredTeams.length})
          </button>
        </div>

        <div className="search-box">
          <input
            type="text"
            placeholder="🔍 Search teams..."
            value={searchTeam}
            onChange={(e) => setSearchTeam(e.target.value)}
          />
        </div>

        {view === 'games' && (
          <div className="filters">
            <div className="filter-group">
              <label>Quality:</label>
              <select value={selectedQuality} onChange={(e) => setSelectedQuality(e.target.value)}>
                <option value="all">All Quality ({sortedGames.length})</option>
                <option value="high">⭐ High ({gamesByQuality.high.length})</option>
                <option value="medium">🔶 Medium ({gamesByQuality.medium.length})</option>
                <option value="poor">🔻 Poor ({gamesByQuality.poor.length})</option>
              </select>
            </div>

            <div className="filter-group">
              <label>Sort By:</label>
              <select value={sortBy} onChange={(e) => setSortBy(e.target.value)}>
                <option value="valuePercent">Value %</option>
                <option value="odds">Odds</option>
                <option value="probability">Probability</option>
                <option value="date">Date</option>
              </select>
            </div>

            <div className="filter-group">
              <label>Min Value %:</label>
              <input
                type="number"
                value={filterMinValue}
                onChange={(e) => setFilterMinValue(Number(e.target.value))}
                step="1"
              />
            </div>

            <div className="filter-group">
              <label>Odds Range:</label>
              <input
                type="number"
                value={filterMinOdds}
                onChange={(e) => setFilterMinOdds(Number(e.target.value))}
                step="0.1"
                min="1"
              />
              <span> - </span>
              <input
                type="number"
                value={filterMaxOdds}
                onChange={(e) => setFilterMaxOdds(Number(e.target.value))}
                step="0.1"
                min="1"
              />
            </div>
          </div>
        )}

        {view === 'teams' && (
          <div className="filters">
            <div className="filter-group">
              <label>
                <input
                  type="checkbox"
                  checked={showRepeatingOnly}
                  onChange={(e) => setShowRepeatingOnly(e.target.checked)}
                />
                Show Repeating Only
              </label>
            </div>
          </div>
        )}
      </div>

      {/* Content */}
      <div className="value-bets-content">
        {view === 'games' ? (
          selectedQuality === 'all' ? (
            // Show categorized by quality when "All" is selected
            <div className="games-view-categorized">
              {['high', 'medium', 'poor'].map(quality => (
                gamesByQuality[quality].length > 0 && (
                  <div key={quality} className="quality-category">
                    <div className={`quality-header quality-${quality}`}>
                      <h2>
                        {quality === 'high' && '⭐ High Quality Bets'}
                        {quality === 'medium' && '🔶 Medium Quality Bets'}
                        {quality === 'poor' && '🔻 Poor Quality Bets'}
                        <span className="category-count">({gamesByQuality[quality].length})</span>
                      </h2>
                    </div>
                    <div className="games-grid">
                      {gamesByQuality[quality].map((game, idx) => (
                        <div key={game.id} className={`game-card quality-${quality}`}>
                          <div className="game-header">
                            <span className="game-number">#{idx + 1}</span>
                            <span className={`quality-badge quality-${quality}`}>
                              {quality.toUpperCase()}
                            </span>
                            <span className="refresh-badge">Refresh {game.refreshNum}</span>
                          </div>
                          
                          <div className="game-teams">
                            <div className="team home-team">
                              <span className="team-label">🏠</span>
                              <span className="team-name">{game.homeTeam}</span>
                            </div>
                            <div className="vs-separator">VS</div>
                            <div className="team away-team">
                              <span className="team-label">✈️</span>
                              <span className="team-name">{game.awayTeam}</span>
                            </div>
                          </div>

                          <div className="game-odds">
                            <div className="odds-item main-odds">
                              <span className="odds-label">Odds</span>
                              <span className="odds-value">{parseFloat(game.odds).toFixed(2)}</span>
                            </div>
                            {game.probability && (
                              <div className="odds-item">
                                <span className="odds-label">Probability</span>
                                <span className="odds-value">{parseFloat(game.probability).toFixed(1)}%</span>
                              </div>
                            )}
                            {game.valuePercent !== null && game.valuePercent !== undefined && (
                              <div className={`odds-item value-item ${parseFloat(game.valuePercent) > 0 ? 'positive' : 'negative'}`}>
                                <span className="odds-label">Value</span>
                                <span className="odds-value">{parseFloat(game.valuePercent).toFixed(2)}%</span>
                              </div>
                            )}
                          </div>

                          {(game.league || game.time) && (
                            <div className="game-meta">
                              {game.league && <span className="meta-league">🏆 {game.league}</span>}
                              {game.time && <span className="meta-date">📅 {game.time}</span>}
                            </div>
                          )}

                          {game.bookmaker && (
                            <div className="game-bookmaker">
                              📊 {game.bookmaker}
                            </div>
                          )}

                          <div className="game-footer">
                            <span className="timestamp">{new Date(game.timestamp).toLocaleString()}</span>
                          </div>
                        </div>
                      ))}
                    </div>
                  </div>
                )
              ))}
              
              {gamesByQuality.unrated.length > 0 && (
                <div className="quality-category">
                  <div className="quality-header quality-unrated">
                    <h2>
                      ❓ Unrated Bets
                      <span className="category-count">({gamesByQuality.unrated.length})</span>
                    </h2>
                  </div>
                  <div className="games-grid">
                    {gamesByQuality.unrated.map((game, idx) => (
                      <div key={game.id} className="game-card">
                        <div className="game-header">
                          <span className="game-number">#{idx + 1}</span>
                          <span className="quality-badge quality-unrated">UNRATED</span>
                          <span className="refresh-badge">Refresh {game.refreshNum}</span>
                        </div>
                        
                        <div className="game-teams">
                          <div className="team home-team">
                            <span className="team-label">🏠</span>
                            <span className="team-name">{game.homeTeam}</span>
                          </div>
                          <div className="vs-separator">VS</div>
                          <div className="team away-team">
                            <span className="team-label">✈️</span>
                            <span className="team-name">{game.awayTeam}</span>
                          </div>
                        </div>

                        <div className="game-odds">
                          <div className="odds-item main-odds">
                            <span className="odds-label">Odds</span>
                            <span className="odds-value">{parseFloat(game.odds).toFixed(2)}</span>
                          </div>
                          {game.probability && (
                            <div className="odds-item">
                              <span className="odds-label">Probability</span>
                              <span className="odds-value">{parseFloat(game.probability).toFixed(1)}%</span>
                            </div>
                          )}
                          {game.valuePercent !== null && game.valuePercent !== undefined && (
                            <div className={`odds-item value-item ${parseFloat(game.valuePercent) > 0 ? 'positive' : 'negative'}`}>
                              <span className="odds-label">Value</span>
                              <span className="odds-value">{parseFloat(game.valuePercent).toFixed(2)}%</span>
                            </div>
                          )}
                        </div>

                        {(game.league || game.time) && (
                          <div className="game-meta">
                            {game.league && <span className="meta-league">🏆 {game.league}</span>}
                            {game.time && <span className="meta-date">📅 {game.time}</span>}
                          </div>
                        )}

                        <div className="game-footer">
                          <span className="timestamp">{new Date(game.timestamp).toLocaleString()}</span>
                        </div>
                      </div>
                    ))}
                  </div>
                </div>
              )}
            </div>
          ) : (
            // Show filtered list when a specific quality is selected
            <div className="games-view">
              <div className="games-grid">
                {sortedGames.map((game, idx) => (
                  <div key={game.id} className={`game-card quality-${game.quality?.toLowerCase() || 'unrated'}`}>
                    <div className="game-header">
                      <span className="game-number">#{idx + 1}</span>
                      {game.quality && (
                        <span className={`quality-badge quality-${game.quality.toLowerCase()}`}>
                          {game.quality.toUpperCase()}
                        </span>
                      )}
                      <span className="refresh-badge">Refresh {game.refreshNum}</span>
                    </div>
                    
                    <div className="game-teams">
                      <div className="team home-team">
                        <span className="team-label">🏠</span>
                        <span className="team-name">{game.homeTeam}</span>
                      </div>
                      <div className="vs-separator">VS</div>
                      <div className="team away-team">
                        <span className="team-label">✈️</span>
                        <span className="team-name">{game.awayTeam}</span>
                      </div>
                    </div>

                    <div className="game-odds">
                      <div className="odds-item main-odds">
                        <span className="odds-label">Odds</span>
                        <span className="odds-value">{parseFloat(game.odds).toFixed(2)}</span>
                      </div>
                      {game.probability && (
                        <div className="odds-item">
                          <span className="odds-label">Probability</span>
                          <span className="odds-value">{parseFloat(game.probability).toFixed(1)}%</span>
                        </div>
                      )}
                      {game.valuePercent !== null && game.valuePercent !== undefined && (
                        <div className={`odds-item value-item ${parseFloat(game.valuePercent) > 0 ? 'positive' : 'negative'}`}>
                          <span className="odds-label">Value</span>
                          <span className="odds-value">{parseFloat(game.valuePercent).toFixed(2)}%</span>
                        </div>
                      )}
                    </div>

                    {(game.league || game.time) && (
                      <div className="game-meta">
                        {game.league && <span className="meta-league">🏆 {game.league}</span>}
                        {game.time && <span className="meta-date">📅 {game.time}</span>}
                      </div>
                    )}

                    {game.bookmaker && (
                      <div className="game-bookmaker">
                        📊 {game.bookmaker}
                      </div>
                    )}

                    <div className="game-footer">
                      <span className="timestamp">{new Date(game.timestamp).toLocaleString()}</span>
                    </div>
                  </div>
                ))}
              </div>

              {sortedGames.length === 0 && (
                <div className="no-data">
                  No games match your filters. Try adjusting the criteria.
                </div>
              )}
            </div>
          )
        ) : (
          <div className="teams-view">
            <div className="teams-grid">
              {filteredTeams.map((team, idx) => (
                <div key={team.team} className="team-card">
                  <div className="team-card-header">
                    <span className="team-rank">#{idx + 1}</span>
                    <h3 className="team-card-name">{team.team}</h3>
                  </div>

                  <div className="team-stats">
                    <div className="team-stat">
                      <span className="stat-icon">🔁</span>
                      <div>
                        <div className="stat-number">{team.appearances}</div>
                        <div className="stat-text">Appearances</div>
                      </div>
                    </div>

                    <div className="team-stat">
                      <span className="stat-icon">📊</span>
                      <div>
                        <div className="stat-number">{parseFloat(team.avgOdds).toFixed(2)}</div>
                        <div className="stat-text">Avg Odds</div>
                      </div>
                    </div>

                    <div className="team-stat">
                      <span className="stat-icon">📈</span>
                      <div>
                        <div className="stat-number">{parseFloat(team.avgProbability).toFixed(1)}%</div>
                        <div className="stat-text">Avg Probability</div>
                      </div>
                    </div>
                  </div>

                  <div className="team-refreshes">
                    <strong>Seen in refreshes:</strong> {team.refreshes.join(', ')}
                  </div>

                  <div className="team-opponents">
                    <strong>Opponents:</strong>
                    <div className="opponents-list">
                      {team.opponents.slice(0, 5).map((opp, i) => (
                        <span key={i} className="opponent-badge">{opp}</span>
                      ))}
                      {team.opponents.length > 5 && (
                        <span className="opponent-badge more">+{team.opponents.length - 5} more</span>
                      )}
                    </div>
                  </div>
                </div>
              ))}
            </div>

            {filteredTeams.length === 0 && (
              <div className="no-data">
                No teams match your search.
              </div>
            )}
          </div>
        )}
      </div>
    </div>
  );
};

export default ValueBets;
