import React, { useState, useEffect } from 'react';
import axios from 'axios';
import './ValueBetPredictions.css';

/**
 * ValueBetPredictions Component
 * 
 * Matches value bets with SportPesa data to create multibet predictions
 * - Fetches value bets from Firebase
 * - Matches with SportPesa allGameData
 * - Finds 3-way markets (1X2)
 * - Generates 100 multibet predictions
 * - Displays predictions in SMS format
 */
function ValueBetPredictions() {
  const [loading, setLoading] = useState(false);
  const [generating, setGenerating] = useState(false);
  const [predictions, setPredictions] = useState([]);
  const [stats, setStats] = useState(null);
  const [error, setError] = useState(null);
  const [selectedPrediction, setSelectedPrediction] = useState(null);

  // Fetch existing predictions on mount
  useEffect(() => {
    fetchPredictions();
  }, []);

  // Fetch existing predictions from Firebase
  const fetchPredictions = async () => {
    try {
      setLoading(true);
      setError(null);
      
      const response = await axios.get('http://localhost:5000/api/getValueBetPredictions');
      
      if (response.data.success) {
        setPredictions(response.data.predictions);
        setStats(response.data.summary);
      } else {
        setError(response.data.message || 'Failed to fetch predictions');
      }
    } catch (err) {
      console.error('Error fetching predictions:', err);
      setError(err.response?.data?.error || err.message);
    } finally {
      setLoading(false);
    }
  };

  // Generate new predictions by matching value bets with SportPesa
  const generatePredictions = async () => {
    try {
      setGenerating(true);
      setError(null);
      
      console.log('🎲 Generating multibet predictions from value bets...');
      
      const response = await axios.get('http://localhost:5000/api/matchValueBetsAndCreatePredictions');
      
      if (response.data.success) {
        console.log('✅ Predictions generated successfully!');
        console.log('Stats:', response.data.stats);
        
        // Refresh predictions list
        await fetchPredictions();
        
        alert(`Success! Generated ${response.data.stats.predictionsGenerated} predictions\n\n` +
              `Matched: ${response.data.stats.matched} games\n` +
              `Unmatched: ${response.data.stats.unmatched} games`);
      } else {
        setError(response.data.error || 'Failed to generate predictions');
      }
    } catch (err) {
      console.error('Error generating predictions:', err);
      setError(err.response?.data?.error || err.message);
      
      if (err.response?.data?.details) {
        console.error('Details:', err.response.data.details);
      }
    } finally {
      setGenerating(false);
    }
  };

  // Copy SMS command to clipboard
  const copySMS = (smsCommand) => {
    navigator.clipboard.writeText(smsCommand);
    alert('SMS command copied to clipboard!');
  };

  // View prediction details
  const viewDetails = (prediction) => {
    setSelectedPrediction(prediction);
  };

  // Close details modal
  const closeDetails = () => {
    setSelectedPrediction(null);
  };

  return (
    <div className="value-bet-predictions-container">
      <div className="header">
        <h1>🎯 Value Bet Predictions</h1>
        <p className="subtitle">
          Multibet predictions generated from value bets matched with SportPesa data
        </p>
      </div>

      {/* Controls */}
      <div className="controls">
        <button 
          className="btn btn-primary"
          onClick={generatePredictions}
          disabled={generating}
        >
          {generating ? '🔄 Generating...' : '🎲 Generate New Predictions'}
        </button>
        
        <button 
          className="btn btn-secondary"
          onClick={fetchPredictions}
          disabled={loading}
        >
          {loading ? '🔄 Loading...' : '🔃 Refresh'}
        </button>
      </div>

      {/* Stats Summary */}
      {stats && (
        <div className="stats-card">
          <h3>📊 Summary</h3>
          <div className="stats-grid">
            <div className="stat-item">
              <span className="stat-label">Total Predictions</span>
              <span className="stat-value">{stats.totalPredictions}</span>
            </div>
            <div className="stat-item">
              <span className="stat-label">Avg Odds</span>
              <span className="stat-value">{stats.avgOdds}</span>
            </div>
            <div className="stat-item">
              <span className="stat-label">Avg Games/Bet</span>
              <span className="stat-value">{stats.avgGamesPerBet}</span>
            </div>
            <div className="stat-item">
              <span className="stat-label">Total Potential Win</span>
              <span className="stat-value">KES {stats.totalPotentialWin}</span>
            </div>
          </div>
        </div>
      )}

      {/* Error Message */}
      {error && (
        <div className="error-message">
          <strong>⚠️ Error:</strong> {error}
        </div>
      )}

      {/* Loading State */}
      {loading && (
        <div className="loading-state">
          <div className="spinner"></div>
          <p>Loading predictions...</p>
        </div>
      )}

      {/* Predictions List */}
      {!loading && predictions.length > 0 && (
        <div className="predictions-list">
          <h3>🎲 Predictions ({predictions.length})</h3>
          <div className="predictions-grid">
            {predictions.map((prediction) => (
              <div key={prediction.id} className="prediction-card">
                <div className="prediction-header">
                  <h4>{prediction.predictionName}</h4>
                  <span className="odds-badge">
                    {prediction.totalOdds}x
                  </span>
                </div>

                <div className="prediction-info">
                  <div className="info-row">
                    <span>🎯 Games:</span>
                    <span>{prediction.numGames}</span>
                  </div>
                  <div className="info-row">
                    <span>💰 Stake:</span>
                    <span>KES {prediction.price}</span>
                  </div>
                  <div className="info-row">
                    <span>🏆 Potential Win:</span>
                    <span className="win-amount">KES {prediction.potentialWin}</span>
                  </div>
                </div>

                {/* Games List */}
                <div className="games-list">
                  {prediction.selectedGames?.slice(0, 3).map((game, index) => (
                    <div key={index} className="game-item">
                      <span className="game-number">{index + 1}</span>
                      <div className="game-details">
                        <div className="teams">
                          {game.homeTeam} vs {game.awayTeam}
                        </div>
                        <div className="game-meta">
                          <span className="selection">{game.selection}</span>
                          <span className="game-odds">{game.odds}</span>
                          {game.quality && (
                            <span className={`quality-tag quality-${game.quality.toLowerCase()}`}>
                              {game.quality}
                            </span>
                          )}
                        </div>
                      </div>
                    </div>
                  ))}
                </div>

                {/* Metadata */}
                {prediction.metadata && (
                  <div className="prediction-metadata">
                    <span className="meta-item">
                      📈 Avg Value: {prediction.metadata.avgValuePercent}
                    </span>
                    <span className="meta-item">
                      🎯 Match Confidence: {prediction.metadata.avgMatchConfidence}
                    </span>
                  </div>
                )}

                {/* SMS Command */}
                <div className="sms-section">
                  <code className="sms-command">{prediction.smsCommand}</code>
                  <button 
                    className="btn btn-copy"
                    onClick={() => copySMS(prediction.smsCommand)}
                  >
                    📋 Copy
                  </button>
                </div>

                {/* Actions */}
                <div className="prediction-actions">
                  <button 
                    className="btn btn-details"
                    onClick={() => setSelectedPrediction({ ...prediction, isOpposite: true, originalStrategy: 'contrarian_betting' })}
                  >
                    👁️ View Details
                  </button>
                </div>
              </div>
            ))}
          </div>
        </div>
      )}

      {/* Empty State */}
      {!loading && predictions.length === 0 && (
        <div className="empty-state">
          <h3>No Predictions Yet</h3>
          <p>Click "Generate New Predictions" to create multibet predictions from value bets</p>
        </div>
      )}

      {/* Details Modal */}
      {selectedPrediction && (
        <div className="modal-overlay" onClick={closeDetails}>
          <div className="modal-content" onClick={(e) => e.stopPropagation()}>
            <div className="modal-header">
              <h2>{selectedPrediction.predictionName || 'Value Bet Prediction'}</h2>
              <button className="btn-close" onClick={closeDetails}>×</button>
            </div>
            <div className="modal-body">
              <h2>
                {selectedPrediction.price ? `KES ${selectedPrediction.price}` : ''} 
                {selectedPrediction.createdAt ? ` - ${new Date(selectedPrediction.createdAt).toLocaleDateString()}` : ''}
              </h2>
              <p>
                {selectedPrediction.description || 'No description available.'}
              </p>
              <div style={{ margin: '12px 0', fontWeight: 'bold' }}>
                {selectedPrediction.numGames ? `${selectedPrediction.numGames} game(s)` : ''}
                {selectedPrediction.totalOdds ? ` — Odds: ${selectedPrediction.totalOdds}` : ''}
                {selectedPrediction.potentialWin ? ` — Win: KES ${selectedPrediction.potentialWin}` : ''}
              </div>
              {selectedPrediction.smsCommand && (
                <div style={{ marginBottom: '16px' }}>
                  <strong>SMS String:</strong>
                  <div style={{
                    backgroundColor: '#fff',
                    border: '1px solid #ddd',
                    padding: '8px',
                    borderRadius: '4px',
                    fontFamily: 'monospace',
                    fontSize: '12px',
                    wordBreak: 'break-all',
                    marginTop: '8px'
                  }}>
                    {selectedPrediction.smsCommand}
                  </div>
                </div>
              )}
              {selectedPrediction.selectedGames && selectedPrediction.selectedGames.length > 0 && (
                <div>
                  <h4>Games:</h4>
                  <ul>
                    {selectedPrediction.selectedGames.map((game, idx) => (
                      <li key={idx}>
                        <strong>{game.homeTeam || game.home} vs {game.awayTeam || game.away}</strong>
                        <div style={{ fontSize: '13px', color: '#666' }}>
                          <div><strong>Market:</strong> {game.market || 'N/A'}</div>
                          <div><strong>Selection:</strong> {game.selection || 'N/A'}</div>
                          <div><strong>Odds:</strong> {game.odds || 'N/A'}</div>
                          {game.startTimestamp && <div><strong>Date:</strong> {new Date(game.startTimestamp).toLocaleString()}</div>}
                          {game.smsCode && <div><strong>SMS Code:</strong> {game.smsCode}</div>}
                        </div>
                      </li>
                    ))}
                  </ul>
                </div>
              )}
              <div style={{ marginTop: '16px' }}>
                <button onClick={() => copySMS(selectedPrediction.smsCommand)} style={{ marginRight: 8 }}>
                  Copy SMS
                </button>
                {/* Add more actions here if you implement them */}
              </div>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}

export default ValueBetPredictions;
