import React from 'react';
import '../components/Navigation/Styles.css';
import '../components/Navigation/css';
import { AuthContext } from '../context/auth-context';
import Modal from '../components/Modal/Modal';
import './Events.css';
import EventList from '../components/Events/EventList/EventList';

class Today_Betlist extends React.Component {
  state = {
    creating: true,
    products: [],
    predictions: [],
    selectedEvent: null,
    copyingMap: {},
    copiedMap: {},
    visitedMap: {},
    reports: {}
  };

  isActive = true;

  static contextType = AuthContext;

  componentDidMount() {
    this.fetchEvents();
    this.fetchPredictions();

    try {
      const storedVisited = JSON.parse(localStorage.getItem('copiedPredictions') || '{}');
      if (storedVisited && typeof storedVisited === 'object') {
        this.setState({ visitedMap: storedVisited });
      }
    } catch (e) {
      console.warn('Failed to load copiedPredictions from localStorage', e);
    }
    try {
      const storedReports = JSON.parse(localStorage.getItem('predictionCopyReports') || '{}');
      if (storedReports && typeof storedReports === 'object') {
        const validReports = {};
        Object.keys(storedReports).forEach(key => {
          if (storedReports[key] && typeof storedReports[key] === 'object' && storedReports[key].status) {
            validReports[key] = storedReports[key];
          }
        });
        this.setState({ reports: validReports });
      }
    } catch (e) {
      console.warn('Failed to load predictionCopyReports from localStorage', e);
    }
  }

  persistVisited = (visitedMap) => {
    try {
      localStorage.setItem('copiedPredictions', JSON.stringify(visitedMap || {}));
    } catch (e) {
      console.warn('Failed to persist visitedMap', e);
    }
  };

  persistReports = (reports) => {
    try {
      const validReports = {};
      Object.keys(reports || {}).forEach(key => {
        if (reports[key] && typeof reports[key] === 'object' && reports[key].status) {
          validReports[key] = reports[key];
        }
      });
      localStorage.setItem('predictionCopyReports', JSON.stringify(validReports));
    } catch (e) {
      console.warn('Failed to persist reports', e);
    }
  };

  fetchEvents() {
    const requestBody = {
      query: `
        query {
          products {
            _id
            name
            description
            date
            category
            brand
            image
            price
            creator {
              _id
              email
            }
          }
        }
      `
    };

    fetch('http://localhost:5000/graphql', {
      method: 'POST',
      body: JSON.stringify(requestBody),
      headers: {
        'Content-Type': 'application/json'
      }
    })
      .then(res => {
        if (res.status !== 200 && res.status !== 201) {
          throw new Error('Failed!');
        }
        return res.json();
      })
      .then(resData => {
        const products = resData.data.products;
        this.setState({ products: products });
      })
      .catch(err => {
        console.log(err);
      });
  }

  fetchPredictions() {
    fetch('http://localhost:5000/api/todayspredictions')
      .then(res => {
        if (res.status !== 200 && res.status !== 201) {
          throw new Error('Failed to fetch predictions!');
        }
        return res.json();
      })
      .then(predictionsData => {
        const predictionsArray = Array.isArray(predictionsData)
          ? predictionsData
          : Object.keys(predictionsData || {}).map(key => predictionsData[key]);

        const idSet = new Set();
        predictionsArray.forEach((p, i) => {
          if (!p.predictionId) {
            console.warn(`Prediction at index ${i} missing predictionId`, p);
          } else if (idSet.has(p.predictionId)) {
            console.warn(`Duplicate predictionId: ${p.predictionId}`);
          } else {
            idSet.add(p.predictionId);
          }
        });

        const sortedPredictions = predictionsArray.sort((a, b) => {
          const timeA = new Date(a.predictionId.replace("Prediction_", "")).getTime();
          const timeB = new Date(b.predictionId.replace("Prediction_", "")).getTime();
          return timeA - timeB;
        });

        const withLocalIds = sortedPredictions.map((p, i) => ({
          _localId: p.predictionId || `local-${i}`,
          ...p
        }));

        this.setState(prev => {
          const newPredictions = withLocalIds;
          const newVisitedMap = {};
          const newReports = {};
          newPredictions.forEach(p => {
            const id = p.predictionId || p._localId;
            newVisitedMap[id] = prev.visitedMap[id] || false;
            newReports[id] = prev.reports[id] || {};
          });
          this.persistVisited(newVisitedMap);
          this.persistReports(newReports);
          return {
            predictions: newPredictions,
            visitedMap: newVisitedMap,
            reports: newReports
          };
        });
      })
      .catch(err => {
        console.log(err);
      });
  }

  showDetailHandler = productId => {
    this.setState(prevState => {
      const selectedEvent = prevState.products.find(e => e._id === productId);
      return { selectedEvent: selectedEvent };
    });
  };

  bookEventHandler = () => {
    if (!this.context.token) {
      this.setState({ selectedEvent: null });
      return;
    }
    const requestBody = {
      query: `
        mutation BookEvent($id: ID!) {
          bookEvent(productId: $id) {
            _id
            createdAt
            updatedAt
          }
        }
      `,
      variables: {
        id: this.state.selectedEvent._id
      }
    };

    fetch('http://localhost:8000/graphql', {
      method: 'POST',
      body: JSON.stringify(requestBody),
      headers: {
        'Content-Type': 'application/json',
        Authorization: 'Bearer ' + this.context.token
      }
    })
      .then(res => {
        if (res.status !== 200 && res.status !== 201) {
          throw new Error('Failed!');
        }
        return res.json();
      })
      .then(resData => {
        console.log(resData);
        this.setState({ selectedEvent: null });
      })
      .catch(err => {
        console.log(err);
      });
  };

  componentWillUnmount() {
    this.isActive = false;
  }

  handleBuyClick = (prediction, index) => {
    if (!prediction || !Array.isArray(prediction.selectedOdds)) {
      console.error('Invalid prediction object or selectedOdds is not an array:', prediction);
      return;
    }

    const serializablePrediction = {
      predictionId: prediction.predictionId,
      predictionName: prediction.predictionName,
      predictiontype: prediction.predictiontype,
      numGames: prediction.numGames,
      totalOdds: prediction.totalOdds,
      selectedOdds: prediction.selectedOdds.map((odd) => ({
        marketName: odd.marketName,
        selectionName: odd.selectionName,
        odds: odd.odds,
        gameId: odd.gameId,
        date: odd.Date,
        competitors: Array.isArray(odd.competitors)
          ? odd.competitors.map((competitor) => ({
              id: competitor.id,
              name: competitor.name,
            }))
          : [],
      })),
      copySuccess: prediction.copySuccess,
      copyReport: prediction.copyReport
    };

    this.props.history.push({
      pathname: '/prediction-details',
      state: {
        prediction: serializablePrediction,
        predictions: this.state.predictions,
        currentIndex: index
      }
    });
  };

  handleCopySms = async (prediction) => {
    if (!prediction) {
      console.warn('Missing prediction', prediction);
      return;
    }

    const predId = prediction.predictionId || prediction._localId;
    if (!predId) {
      console.warn('Prediction missing id and _localId', prediction);
      return;
    }

    if (this.state.copyingMap && this.state.copyingMap[predId]) return;

    this.setState(prev => ({ copyingMap: { ...prev.copyingMap, [predId]: true } }));

    try {
      let smsText = prediction.smsString;

      if (!smsText) {
        const resp = await fetch('http://localhost:5000/api/createSmsBets', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({ predictions: [prediction], stake: 1 })
        });

        if (!resp.ok) {
          const body = await resp.text().catch(() => '');
          throw new Error(`createSmsBets failed: ${resp.status} ${resp.statusText} ${body}`);
        }

        const data = await resp.json();
        if (!data || !data.smsBets || !data.smsBets[0] || !data.smsBets[0].smsString) {
          throw new Error('Invalid response when creating sms bets');
        }
        smsText = data.smsBets[0].smsString;

        this.setState(prev => ({
          predictions: prev.predictions.map(p => {
            const id = p.predictionId || p._localId;
            return id === predId ? { ...p, smsString: smsText } : p;
          })
        }));
      }

      if (navigator && navigator.clipboard && navigator.clipboard.writeText) {
        await navigator.clipboard.writeText(smsText);
      } else {
        const ta = document.createElement('textarea');
        ta.value = smsText;
        ta.style.position = 'fixed';
        ta.style.left = '-999999px';
        ta.style.top = '-999999px';
        document.body.appendChild(ta);
        ta.focus();
        ta.select();
        try {
          document.execCommand('copy');
        } catch (err) {
          console.error('Unable to copy to clipboard', err);
        }
        document.body.removeChild(ta);
      }

      const successReport = {
        status: 'success',
        message: smsText,
        timestamp: new Date().toISOString()
      };
      await fetch('http://localhost:5000/api/updatePrediction', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          Authorization: 'Bearer ' + this.context.token
        },
        body: JSON.stringify({
          predictionId: predId,
          copySuccess: true,
          copyReport: successReport
        })
      })
        .then(res => {
          if (!res.ok) throw new Error('Failed to update prediction');
          return res.json();
        })
        .then(() => {
          this.setState(prev => ({
            predictions: prev.predictions.map(p => {
              const id = p.predictionId || p._localId;
              return id === predId ? { ...p, copySuccess: true, copyReport: successReport } : p;
            })
          }));
        })
        .catch(err => {
          console.error('Failed to update prediction in database:', err);
          alert('Failed to update prediction in database: ' + err.message);
        });

      this.setState(prev => ({ copiedMap: { ...prev.copiedMap, [predId]: true } }));

      this.setState(prev => {
        const visitedNext = { ...(prev.visitedMap || {}), [predId]: true };
        this.persistVisited(visitedNext);
        return { visitedMap: visitedNext };
      });

      this.setState(prev => {
        const reportsNext = { ...(prev.reports || {}), [predId]: successReport };
        this.persistReports(reportsNext);
        return { reports: reportsNext };
      });

      setTimeout(() => {
        this.setState(prev => {
          const cm = { ...(prev.copiedMap || {}) };
          delete cm[predId];
          return { copiedMap: cm };
        });
      }, 4000);
    } catch (err) {
      const errMsg = err && err.message ? err.message : String(err);
      const errorReport = {
        status: 'error',
        message: errMsg,
        timestamp: new Date().toISOString()
      };

      await fetch('http://localhost:5000/api/updatePrediction', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          Authorization: 'Bearer ' + this.context.token
        },
        body: JSON.stringify({
          predictionId: predId,
          copySuccess: false,
          copyReport: errorReport
        })
      })
        .then(res => {
          if (!res.ok) throw new Error('Failed to update prediction');
          return res.json();
        })
        .then(() => {
          this.setState(prev => ({
            predictions: prev.predictions.map(p => {
              const id = p.predictionId || p._localId;
              return id === predId ? { ...p, copySuccess: false, copyReport: errorReport } : p;
            })
          }));
        })
        .catch(err => {
          console.error('Failed to update prediction in database:', err);
          alert('Failed to update prediction in database: ' + err.message);
        });

      this.setState(prev => {
        const reportsNext = { ...(prev.reports || {}), [predId]: errorReport };
        this.persistReports(reportsNext);
        return { reports: reportsNext };
      });
      console.error('Failed to copy SMS for prediction', predId, err);
      alert('Failed to copy SMS: ' + errMsg);
    } finally {
      this.setState(prev => {
        const cm = { ...(prev.copyingMap || {}) };
        delete cm[predId];
        return { copyingMap: cm };
      });
    }
  };

  handleResetInteractions = () => {
    if (!window.confirm('Are you sure you want to reset all interaction history? This will clear copied statuses and reports.')) {
      return;
    }

    try {
      localStorage.removeItem('copiedPredictions');
      localStorage.removeItem('predictionCopyReports');
      Promise.all(this.state.predictions.map(pred => {
        const predId = pred.predictionId || pred._localId;
        return fetch('http://localhost:5000/api/updatePrediction', {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
            Authorization: 'Bearer ' + this.context.token
          },
          body: JSON.stringify({
            predictionId: predId,
            copySuccess: false,
            copyReport: null
          })
        }).catch(err => {
          console.error(`Failed to reset prediction ${predId}:`, err);
        });
      })).then(() => {
        this.setState({
          copyingMap: {},
          copiedMap: {},
          visitedMap: {},
          reports: {},
          selectedEvent: null,
          predictions: this.state.predictions.map(p => ({
            ...p,
            copySuccess: false,
            copyReport: null
          }))
        });
      });
    } catch (e) {
      console.error('Failed to reset interactions', e);
      alert('Failed to reset interactions. See console for details.');
    }
  };

  computeReportCounts = () => {
    const { reports } = this.state;
    let success = 0;
    let error = 0;
    if (reports && typeof reports === 'object') {
      Object.values(reports).forEach(r => {
        if (!r || !r.status) return;
        if (r.status === 'success') success += 1;
        else if (r.status === 'error') error += 1;
      });
    }
    return { success, error, total: success + error };
  };

  handleExportReports = () => {
    try {
      const data = JSON.stringify(this.state.reports || {}, null, 2);
      const blob = new Blob([data], { type: 'application/json' });
      const url = URL.createObjectURL(blob);
      const a = document.createElement('a');
      a.href = url;
      a.download = `prediction_reports_${new Date().toISOString().replace(/[:.]/g,'-')}.json`;
      document.body.appendChild(a);
      a.click();
      document.body.removeChild(a);
      URL.revokeObjectURL(url);
    } catch (e) {
      console.error('Failed to export reports', e);
      alert('Failed to export reports. See console for details.');
    }
  };

  handleUploadReports = async () => {
    try {
      const body = { reports: this.state.reports || {} };
      const resp = await fetch('http://localhost:5000/api/reports', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(body)
      });
      if (!resp.ok) {
        const text = await resp.text().catch(() => '');
        throw new Error(`${resp.status} ${resp.statusText} ${text}`);
      }
      alert('Reports uploaded successfully for system improvement.');
    } catch (err) {
      console.error('Failed to upload reports', err);
      alert('Failed to upload reports: ' + (err.message || String(err)));
    }
  };

  renderPredictionStatus = (pred) => {
    const id = pred.predictionId || pred._localId;
    const { copiedMap } = this.state;
    const copied = !!(copiedMap && copiedMap[id]);
    const { copySuccess, copyReport } = pred;
    if (copied) {
      return <span style={{ marginLeft: 10, color: '#0a7' }}>Copied — bet placed</span>;
    }
    if (copyReport && copyReport.status) {
      if (copyReport.status === 'success') {
        return <span style={{ marginLeft: 10, color: '#0a7', fontSize: 12 }}>
          Last sent at {new Date(copyReport.timestamp).toLocaleString()}
        </span>;
      }
      return <span style={{ marginLeft: 10, color: '#c00', fontSize: 12 }}>
        Error: {copyReport.message}
      </span>;
    }
    return copySuccess ? (
      <span style={{ marginLeft: 10, color: '#0a7', fontSize: 12 }}>Previously copied successfully</span>
    ) : null;
  };

  renderReportsPanel = () => {
    const { reports } = this.state;
    const keys = reports
      ? Object.keys(reports).filter(k => reports[k] && typeof reports[k] === 'object' && reports[k].status)
      : [];
    if (!keys.length) return null;
    return (
      <div style={{ marginTop: 20, padding: 12, border: '1px solid #ddd', borderRadius: 6 }}>
        <h4 style={{ marginTop: 0 }}>Copy reports (last per prediction, local)</h4>
        <ul style={{ paddingLeft: 18 }}>
          {keys.map(k => {
            const r = reports[k];
            return (
              <li key={k} style={{ marginBottom: 8 }}>
                <strong>{k}</strong> — {r.status === 'success' ? <span style={{ color: '#0a7' }}>Success</span> : <span style={{ color: '#c00' }}>Error</span>}
                <div style={{ fontSize: 12, color: '#444' }}>{r.message}</div>
                <div style={{ fontSize: 11, color: '#666' }}>{new Date(r.timestamp).toLocaleString()}</div>
              </li>
            );
          })}
        </ul>
        <button onClick={this.handleExportReports}>Export Reports</button>
        <button onClick={this.handleUploadReports} style={{ marginLeft: 8 }}>Upload Reports to Server</button>
      </div>
    );
  };

  render() {
    return (
      <React.Fragment>
        {this.state.selectedEvent && (
          <Modal
            title={this.state.selectedEvent.title}
            canCancel
            canConfirm
            onCancel={() => this.setState({ selectedEvent: null })}
            onConfirm={this.bookEventHandler}
            confirmText={this.context.token ? 'Book' : 'Confirm'}
          >
            <h1>{this.state.selectedEvent.name}</h1>
            <h2>
              ${this.state.selectedEvent.price} -{' '}
              {new Date(this.state.selectedEvent.date).toLocaleDateString()}
            </h2>
            <p>{this.state.selectedEvent.description}</p>
          </Modal>
        )}

        <div>
          <EventList
            products={this.state.products}
            authUserId={this.context.userId}
            onViewDetail={this.showDetailHandler}
          />
        </div>

        <div style={{ marginTop: 24 }}>
          <h2>Today's Predictions</h2>
          <button
            onClick={this.handleResetInteractions}
            style={{ marginBottom: 12, background: '#c00', color: '#fff', padding: '8px 16px', borderRadius: 4 }}
          >
            Reset Interactions
          </button>
          {this.state.predictions.length === 0 ? (
            <p>No predictions available.</p>
          ) : (
            <ol>
              {this.state.predictions.map((pred, idx) => {
                const id = pred.predictionId || pred._localId || `local-${idx}`;
                const copying = !!this.state.copyingMap[id];
                const visited = !!this.state.visitedMap[id];
                return (
                  <li key={id} style={{ marginBottom: 12 }}>
                    <div style={{ display: 'flex', alignItems: 'center', gap: 12 }}>
                      <div style={{ flex: 1 }}>
                        <strong>{pred.predictionName || `Prediction ${id}`}</strong>
                        <div style={{ fontSize: 13, color: '#555' }}>
                          {pred.numGames ? `${pred.numGames} game(s)` : ''}
                          {pred.totalOdds ? ` — Odds: ${pred.totalOdds}` : ''}
                        </div>
                      </div>
                      <div>
                        <button onClick={() => this.handleBuyClick(pred, idx)} style={{ marginRight: 8 }}>
                          View
                        </button>
                        <button
                          onClick={() => this.handleCopySms(pred)}
                          disabled={copying}
                          style={visited ? { color: '#551A8B', textDecoration: 'underline' } : {}}
                        >
                          {copying ? (
                            <span>
                              Copying… <span style={{ fontSize: 12 }}>(⏳)</span>
                            </span>
                          ) : (visited ? 'Copy SMS (visited)' : 'Copy SMS')}
                        </button>
                        {this.renderPredictionStatus(pred)}
                      </div>
                    </div>
                  </li>
                );
              })}
            </ol>
          )}
        </div>
        {this.renderReportsPanel()}
      </React.Fragment>
    );
  }
}

export default Today_Betlist;