import React, { useContext, useEffect, useState } from 'react';
import { AuthContext } from '../context/auth-context';
import { useNavigate, Link } from 'react-router-dom';

export default function ProfilePage() {
  const { user } = useContext(AuthContext);
  const [profile, setProfile] = useState({ 
    name: '', 
    contact: '', 
    bio: '', 
    avatarUrl: '' 
  });
  const [loading, setLoading] = useState(true);
  const [saving, setSaving] = useState(false);
  const [dirty, setDirty] = useState(false);
  const [successMsg, setSuccessMsg] = useState('');
  const [errorMsg, setErrorMsg] = useState('');
  const navigate = useNavigate();
  const [posts, setPosts] = useState([]);
  const [postsLoading, setPostsLoading] = useState(false);
  const [postsError, setPostsError] = useState('');
  const [filterText, setFilterText] = useState('');
  const [filterType, setFilterType] = useState(''); // e.g., 'text', 'image'
  const [page, setPage] = useState(1);
  const [pageSize, setPageSize] = useState(5);
  const [kpis, setKpis] = useState({ followers: 0, totalPosts: 0, avgEngagement: 0 });

  // Sync profile from auth user immediately (e.g., displayName, email, photoURL)
  useEffect(() => {
    if (user) {
      setProfile(prev => ({
        ...prev,
        name: user.displayName || prev.name || '',
        contact: user.email || prev.contact || '',
        avatarUrl: user.photoURL || prev.avatarUrl || ''
      }));
      setDirty(false); // Reset dirty since this is auto-loaded
    }
  }, [user]);

  // Load additional profile from localStorage or backend (merges with auth data)
  useEffect(() => {
    try {
      const raw = localStorage.getItem('user_profile');
      if (raw) {
        const parsed = JSON.parse(raw);
        if (parsed.expiresAt && Date.now() < parsed.expiresAt && parsed.profile) {
          setProfile(prev => ({ ...prev, ...parsed.profile })); // Merge with auth-synced data
          setLoading(false);
          return;
        }
      }
    } catch (e) {
      // ignore
    }

    (async () => {
      try {
        if (!user) { setLoading(false); return; }
        const token = await user.getIdToken();
        if (!token) { setLoading(false); return; }
        const res = await fetch('http://localhost:5000/api/profile', { 
          headers: { Authorization: `Bearer ${token}` } 
        });
        if (!res.ok) { setLoading(false); return; }
        const json = await res.json();
        if (json.profile) {
          setProfile(prev => ({ ...prev, ...json.profile })); // Merge backend with auth/local
        }
      } catch (e) {
        console.warn('Failed to load profile', e);
      } finally {
        setLoading(false);
      }
    })();
  }, [user]);

  // Load user's posts and calculate KPIs
  useEffect(() => {
    (async () => {
      setPostsLoading(true);
      setPostsError('');
      try {
        if (!user) { setPosts([]); setKpis({ followers: 0, totalPosts: 0, avgEngagement: 0 }); setPostsLoading(false); return; }
        const token = await user.getIdToken();
        if (!token) { setPosts([]); setPostsLoading(false); return; }
        const res = await fetch('http://localhost:5000/api/posts', { 
          headers: { Authorization: `Bearer ${token}` } 
        });
        if (!res.ok) throw new Error('Failed to fetch posts');
        const json = await res.json();
        const userPosts = json.posts.filter(p => p.authorId === user.uid) || [];
        setPosts(userPosts);

        // Calculate KPIs
        const totalPosts = userPosts.length;
        const totalEngagement = userPosts.reduce((sum, p) => sum + (p.likes || 0) + (p.comments || 0), 0);
        const avgEngagement = totalPosts > 0 ? Math.round(totalEngagement / totalPosts) : 0;
        setKpis({ 
          followers: json.followerCount || 0, // Assume backend provides this
          totalPosts, 
          avgEngagement 
        });
      } catch (e) {
        setPostsError(e.message || 'Error fetching posts');
      } finally {
        setPostsLoading(false);
      }
    })();
  }, [user]);

  const validate = ({ name, contact, bio }) => {
    if (!name || String(name).trim().length < 2) return 'Please provide your full name (at least 2 characters).';
    if (!contact || String(contact).trim().length === 0) return 'Please provide an email or phone number.';
    const c = String(contact).trim();
    const emailRe = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    const phoneRe = /^\+?\d{7,15}$/;
    if (!emailRe.test(c) && !phoneRe.test(c)) return 'Contact must be a valid email or international phone number.';
    if (bio && String(bio).trim().length > 500) return 'Bio must be under 500 characters.';
    return null;
  };

  const save = async (e) => {
    e.preventDefault();
    setErrorMsg('');
    const err = validate(profile);
    if (err) {
      setErrorMsg(err);
      return;
    }
    setSaving(true);
    try {
      if (user) {
        const token = await user.getIdToken();
        if (token) {
          // Update Firebase user for displayName/photoURL if changed
          if (profile.name !== user.displayName || profile.avatarUrl !== user.photoURL) {
            await user.updateProfile({
              displayName: profile.name,
              photoURL: profile.avatarUrl
            });
          }
          await fetch('http://localhost:5000/api/profile', { 
            method: 'POST', 
            headers: { 'Content-Type': 'application/json', Authorization: `Bearer ${token}` }, 
            body: JSON.stringify(profile) 
          });
        }
      }
      const expiresAt = Date.now() + 130 * 60 * 1000;
      localStorage.setItem('user_profile', JSON.stringify({ profile, expiresAt }));
      setSuccessMsg('Profile updated successfully!');
      setDirty(false);
      setTimeout(() => setSuccessMsg(''), 3000);
    } catch (e) {
      setErrorMsg('Failed to update profile: ' + (e.message || e));
    } finally {
      setSaving(false);
    }
  };

  // Filtered and paginated posts
  const getFilteredPosts = () => {
    const lower = filterText.trim().toLowerCase();
    return posts.filter(p => {
      if (filterType && p.type !== filterType) return false;
      if (!lower) return true;
      return p.title?.toLowerCase().includes(lower) || p.content?.toLowerCase().includes(lower);
    });
  };

  if (loading) return <div style={{ padding: 20 }}>Loading profile…</div>;

  const filteredPosts = getFilteredPosts();
  const total = filteredPosts.length;
  const pages = Math.max(1, Math.ceil(total / pageSize));
  const p = Math.min(Math.max(1, page), pages);
  const start = (p - 1) * pageSize;
  const pageItems = filteredPosts.slice(start, start + pageSize);

  return (
    <div style={{ padding: 20, maxWidth: 800, margin: '0 auto' }}>
      <h1 style={{ textAlign: 'center', marginBottom: 20 }}>Your Social Profile</h1>
      
      {/* Profile Edit Form */}
      <form onSubmit={save} style={{ background: '#f9f9f9', padding: 20, borderRadius: 8, marginBottom: 24 }}>
        <div style={{ display: 'grid', gridTemplateColumns: '1fr 1fr', gap: 12, marginBottom: 12 }}>
          <div>
            <label style={{ display: 'block', marginBottom: 6 }}>Full Name</label>
            <input 
              value={profile.name} 
              onChange={e => { setProfile(p => ({ ...p, name: e.target.value })); setDirty(true); }} 
              style={{ width: '100%', padding: 8, border: '1px solid #ddd', borderRadius: 4 }} 
            />
          </div>
          <div>
            <label style={{ display: 'block', marginBottom: 6 }}>Contact (email or phone)</label>
            <input 
              value={profile.contact} 
              onChange={e => { setProfile(p => ({ ...p, contact: e.target.value })); setDirty(true); }} 
              style={{ width: '100%', padding: 8, border: '1px solid #ddd', borderRadius: 4 }} 
            />
          </div>
        </div>
        <div style={{ marginBottom: 12 }}>
          <label style={{ display: 'block', marginBottom: 6 }}>Bio (optional, max 500 chars)</label>
          <textarea 
            value={profile.bio} 
            onChange={e => { setProfile(p => ({ ...p, bio: e.target.value })); setDirty(true); }} 
            rows={3}
            style={{ width: '100%', padding: 8, border: '1px solid #ddd', borderRadius: 4 }} 
          />
        </div>
        <div style={{ marginBottom: 12 }}>
          <label style={{ display: 'block', marginBottom: 6 }}>Avatar URL (optional)</label>
          <input 
            value={profile.avatarUrl} 
            onChange={e => { setProfile(p => ({ ...p, avatarUrl: e.target.value })); setDirty(true); }} 
            placeholder="https://example.com/avatar.jpg"
            style={{ width: '100%', padding: 8, border: '1px solid #ddd', borderRadius: 4 }} 
          />
          {profile.avatarUrl && (
            <img src={profile.avatarUrl} alt="Avatar preview" style={{ width: 50, height: 50, marginTop: 8, borderRadius: '50%' }} />
          )}
        </div>
        {errorMsg && <div style={{ color: 'red', marginBottom: 8 }}>{errorMsg}</div>}
        {successMsg && <div style={{ color: 'green', marginBottom: 8 }}>{successMsg}</div>}
        <div style={{ display: 'flex', gap: 8 }}>
          <button 
            type="submit" 
            disabled={saving || !dirty} 
            style={{ padding: '10px 16px', background: '#007bff', color: 'white', border: 'none', borderRadius: 4 }} 
          >
            {saving ? 'Saving...' : 'Update Profile'}
          </button>
          <button 
            type="button" 
            onClick={() => navigate('/')} 
            style={{ padding: '10px 16px', background: '#6c757d', color: 'white', border: 'none', borderRadius: 4 }} 
          >
            Back to Feed
          </button>
        </div>
      </form>

      {/* KPI Cards */}
      <div style={{ display: 'grid', gridTemplateColumns: 'repeat(auto-fit, minmax(150px, 1fr))', gap: 16, marginBottom: 24 }}>
        <div style={{ background: '#e9ecef', padding: 16, borderRadius: 8, textAlign: 'center' }}>
          <h3 style={{ margin: 0, color: '#6c757d' }}>Followers</h3>
          <p style={{ fontSize: 24, fontWeight: 'bold', margin: 4 }}>{kpis.followers}</p>
        </div>
        <div style={{ background: '#e9ecef', padding: 16, borderRadius: 8, textAlign: 'center' }}>
          <h3 style={{ margin: 0, color: '#6c757d' }}>Total Posts</h3>
          <p style={{ fontSize: 24, fontWeight: 'bold', margin: 4 }}>{kpis.totalPosts}</p>
        </div>
        <div style={{ background: '#e9ecef', padding: 16, borderRadius: 8, textAlign: 'center' }}>
          <h3 style={{ margin: 0, color: '#6c757d' }}>Avg Engagement</h3>
          <p style={{ fontSize: 24, fontWeight: 'bold', margin: 4 }}>{kpis.avgEngagement}</p>
        </div>
      </div>

      {/* Posts Section */}
      <h2 style={{ marginBottom: 12 }}>Your Posts</h2>
      {postsLoading ? (
        <div>Loading posts…</div>
      ) : postsError ? (
        <div style={{ color: 'red' }}>{postsError}</div>
      ) : posts.length === 0 ? (
        <div>No posts yet. <Link to="/create-post">Create one now!</Link></div>
      ) : (
        <div>
          <div style={{ marginBottom: 12, display: 'flex', gap: 8, alignItems: 'center', flexWrap: 'wrap' }}>
            <input 
              placeholder="Search by title or content" 
              value={filterText} 
              onChange={e => { setFilterText(e.target.value); setPage(1); }} 
              style={{ padding: 8, flex: 1, minWidth: 200, border: '1px solid #ddd', borderRadius: 4 }} 
            />
            <select 
              value={filterType} 
              onChange={e => { setFilterType(e.target.value); setPage(1); }} 
              style={{ padding: 8, border: '1px solid #ddd', borderRadius: 4 }}
            >
              <option value="">All types</option>
              <option value="text">Text</option>
              <option value="image">Image</option>
              <option value="video">Video</option>
            </select>
            <select 
              value={pageSize} 
              onChange={e => { setPageSize(parseInt(e.target.value, 10)); setPage(1); }} 
              style={{ padding: 8, border: '1px solid #ddd', borderRadius: 4 }}
            >
              <option value={5}>5 / page</option>
              <option value={10}>10 / page</option>
              <option value={25}>25 / page</option>
            </select>
          </div>

          <div>
            {pageItems.map(post => (
              <div 
                key={post.id} 
                style={{ 
                  border: '1px solid #eee', 
                  padding: 16, 
                  marginBottom: 12, 
                  borderRadius: 8,
                  display: 'flex', 
                  gap: 12,
                  flexDirection: 'column'
                }} 
              >
                {post.imageUrl && (
                  <img 
                    src={post.imageUrl} 
                    alt="Post media" 
                    style={{ maxWidth: '100%', height: 200, objectFit: 'cover', borderRadius: 4 }} 
                  />
                )}
                <div>
                  <strong>Title:</strong> <Link to={`/posts/${post.id}`}>{post.title}</Link>
                </div>
                <div><strong>Content:</strong> {post.content?.substring(0, 100)}...</div>
                <div><strong>Type:</strong> {post.type}</div>
                <div><strong>Likes:</strong> {post.likes || 0} | <strong>Comments:</strong> {post.comments || 0}</div>
                <div><strong>Posted:</strong> {post.createdAt ? new Date(post.createdAt).toLocaleString() : '—'}</div>
              </div>
            ))}

            <div style={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', marginTop: 16 }}>
              <div>Showing {start + 1}–{Math.min(start + pageSize, total)} of {total} posts</div>
              <div style={{ display: 'flex', gap: 8 }}>
                <button 
                  disabled={p <= 1} 
                  onClick={() => setPage(p - 1)} 
                  style={{ padding: '8px 12px', border: '1px solid #ddd', borderRadius: 4, background: p > 1 ? '#fff' : '#f8f9fa' }} 
                >
                  Prev
                </button>
                <button 
                  disabled={p >= pages} 
                  onClick={() => setPage(p + 1)} 
                  style={{ padding: '8px 12px', border: '1px solid #ddd', borderRadius: 4, background: p < pages ? '#fff' : '#f8f9fa' }} 
                >
                  Next
                </button>
              </div>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}