import React, { useEffect, useState, useContext } from 'react';
import { Link } from 'react-router-dom';
import { AuthContext } from '../context/auth-context';

const Orders = () => {
  const { user } = useContext(AuthContext);
  const [orders, setOrders] = useState([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);

  useEffect(() => {
    const load = async () => {
      if (!user) {
        setLoading(false);
        return;
      }
      try {
        const token = await user.getIdToken();
        const res = await fetch('http://localhost:5000/api/orders', { headers: { Authorization: `Bearer ${token}` } });
        if (!res.ok) throw new Error('Failed to fetch orders');
        const json = await res.json();
        setOrders(json.orders || []);
      } catch (e) {
        setError(e.message || 'Error');
      } finally {
        setLoading(false);
      }
    };
    load();
  }, [user]);

  if (loading) return <div>Loading orders...</div>;
  if (!user) return <div>Please log in to see your orders.</div>;
  if (error) return <div style={{ color: 'red' }}>{error}</div>;

  return (
    <div style={{ padding: 16 }}>
      <h2>Your Orders</h2>
      {orders.length === 0 ? (
        <p>No orders yet.</p>
      ) : (
        <div>
          {orders.map(o => (
            <div key={o.orderId} style={{ border: '1px solid #ddd', padding: 12, marginBottom: 8 }}>
              <div><strong>Order ID:</strong> <Link to={`/orders/${o.orderId}`}>{o.orderId}</Link></div>
              <div><strong>Created:</strong> {new Date(o.createdAt).toLocaleString()}</div>
              <div><strong>Amount:</strong> Ksh. {o.amount}.00</div>
              <div><strong>Status:</strong> {o.status}</div>
              <div><strong>Prediction:</strong> {o.prediction && o.prediction.predictionName}</div>
            </div>
          ))}
        </div>
      )}
    </div>
  );
};

export default Orders;
