import React, { useEffect, useState, useContext } from 'react';
import { useParams } from 'react-router-dom';
import { AuthContext } from '../context/auth-context';

const OrderDetail = () => {
  const { id } = useParams();
  const { user } = useContext(AuthContext);
  const [order, setOrder] = useState(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);

  useEffect(() => {
    const load = async () => {
      if (!user) return setLoading(false);
      try {
        const token = await user.getIdToken();
        // check claims to decide endpoint
        const idTokenResult = await user.getIdTokenResult();
        const isAdmin = idTokenResult && idTokenResult.claims && idTokenResult.claims.role === 'superuser';
        const endpoint = isAdmin ? `http://localhost:5000/api/admin/orders/${encodeURIComponent(id)}` : `http://localhost:5000/api/orders/${encodeURIComponent(id)}`;
        const res = await fetch(endpoint, { headers: { Authorization: `Bearer ${token}` } });
        if (!res.ok) throw new Error('Failed to fetch order');
        const json = await res.json();
        setOrder(json.order);
      } catch (e) {
        setError(e.message || 'Error');
      } finally {
        setLoading(false);
      }
    };
    load();
    // polling until paid
    let pollId = null;
    const startPolling = () => {
      pollId = setInterval(async () => {
        try {
          const token = await user.getIdToken();
          const idTokenResult = await user.getIdTokenResult();
          const isAdmin = idTokenResult && idTokenResult.claims && idTokenResult.claims.role === 'superuser';
          const endpoint = isAdmin ? `http://localhost:5000/api/admin/orders/${encodeURIComponent(id)}` : `http://localhost:5000/api/orders/${encodeURIComponent(id)}`;
          const res = await fetch(endpoint, { headers: { Authorization: `Bearer ${token}` } });
          if (!res.ok) return;
          const json = await res.json();
          setOrder(json.order);
          if (json.order && json.order.status === 'paid') {
            clearInterval(pollId);
          }
        } catch (e) {
          // ignore poll errors
        }
      }, 10000);
    };
    startPolling();
    return () => { if (pollId) clearInterval(pollId); };
  }, [user, id]);

  if (loading) return <div>Loading order...</div>;
  if (!user) return <div>Please log in to see the order.</div>;
  if (error) return <div style={{ color: 'red' }}>{error}</div>;
  if (!order) return <div>Order not found.</div>;

  return (
    <div style={{ padding: 16 }}>
      <h2>Order {order.orderId}</h2>
      <div><strong>User:</strong> {order.uid}</div>
      <div><strong>Amount:</strong> Ksh. {order.amount}.00</div>
      <div><strong>Status:</strong> {order.status}</div>
      <div style={{ marginTop: 8 }}>
        <button onClick={async () => {
          try {
            const token = await user.getIdToken();
            await fetch(`http://localhost:5000/api/orders/${encodeURIComponent(order.orderId)}/status`, { method: 'POST', headers: { 'Content-Type': 'application/json', Authorization: `Bearer ${token}` }, body: JSON.stringify({ status: 'paid' }) });
            const token2 = await user.getIdToken();
            const res = await fetch(`http://localhost:5000/api/orders/${encodeURIComponent(id)}`, { headers: { Authorization: `Bearer ${token2}` } });
            const json = await res.json();
            setOrder(json.order);
          } catch (e) {
            alert('Failed to update status');
          }
        }} className="px-3 py-1 mr-2" style={{ background: '#0a7', color: '#fff' }}>Mark Paid</button>
        <button onClick={async () => {
          try {
            const token = await user.getIdToken();
            await fetch(`http://localhost:5000/api/orders/${encodeURIComponent(order.orderId)}/status`, { method: 'POST', headers: { 'Content-Type': 'application/json', Authorization: `Bearer ${token}` }, body: JSON.stringify({ status: 'failed' }) });
            const token2 = await user.getIdToken();
            const res = await fetch(`http://localhost:5000/api/orders/${encodeURIComponent(id)}`, { headers: { Authorization: `Bearer ${token2}` } });
            const json = await res.json();
            setOrder(json.order);
          } catch (e) {
            alert('Failed to update status');
          }
        }} className="px-3 py-1" style={{ background: '#c00', color: '#fff' }}>Mark Failed</button>
      </div>
      <div><strong>Created:</strong> {new Date(order.createdAt).toLocaleString()}</div>
      {order.paidAt && <div><strong>Paid At:</strong> {new Date(order.paidAt).toLocaleString()}</div>}
      <h3>Prediction</h3>
      <pre style={{ whiteSpace: 'pre-wrap', background: '#f8f8f8', padding: 8 }}>{JSON.stringify(order.prediction, null, 2)}</pre>
      <h3>MPesa Callback (raw)</h3>
      <pre style={{ whiteSpace: 'pre-wrap', background: '#f8f8f8', padding: 8 }}>{JSON.stringify(order.mpesaCallback || order.callback || {}, null, 2)}</pre>
    </div>
  );
};

export default OrderDetail;
