import React from 'react';
import { useNavigate } from 'react-router-dom';
import '../components/Navigation/Styles.css';
import '../components/Navigation/css';
import { AuthContext } from '../context/auth-context';
import Modal from '../components/Modal/Modal';
import './Events.css';
import EventList from '../components/Events/EventList/EventList';

class OppositeBetlist extends React.Component {
  state = {
    creating: true,
    products: [],
    oppositePredictions: [],
    selectedEvent: null,
    copyingMap: {},
    sendingMap: {},
    copiedMap: {},
    visitedMap: {},
    reports: {},
    confirmedMap: {},
    targetSuccessfulBets: 20
  };

  // Autobet state
  // autobetStats: { total, remaining, successes, failures, placedStakeTotal, balanceRemaining }

  isActive = true;

  static contextType = AuthContext;

  componentDidMount() {
    this.fetchOppositePredictions();

    // load local confirmed predictions first
    try {
      const storedConfirmed = JSON.parse(localStorage.getItem('confirmedOppositePredictions') || '{}');
      if (storedConfirmed && typeof storedConfirmed === 'object') {
        this.setState({ confirmedMap: storedConfirmed });
      }
    } catch (e) {
      console.warn('Failed to load confirmedOppositePredictions from localStorage', e);
    }

    // Fetch server-side confirmed predictions (imported via apply_reports or confirmed by other clients)
    fetch('http://localhost:5000/api/confirmedOppositePredictions')
      .then(r => r.ok ? r.json() : Promise.reject(new Error('Failed to fetch confirmed opposite predictions')))
      .then(data => {
        if (data && Array.isArray(data.confirmed)) {
          const serverMap = {};
          data.confirmed.forEach(id => { if (id) serverMap[id] = true; });
          this.setState(prev => {
            const merged = Object.assign({}, prev.confirmedMap || {}, serverMap || {});
            try { localStorage.setItem('confirmedOppositePredictions', JSON.stringify(merged)); } catch (e) { /* ignore */ }
            return { confirmedMap: merged };
          }, () => this.fetchOppositePredictions());
        } else {
          this.fetchOppositePredictions();
        }
      })
      .catch(err => {
        console.warn('Could not load server confirmed opposite predictions, falling back to local only', err && err.message);
        this.fetchOppositePredictions();
      });

    try {
      const storedVisited = JSON.parse(localStorage.getItem('copiedOppositePredictions') || '{}');
      if (storedVisited && typeof storedVisited === 'object') {
        this.setState({ visitedMap: storedVisited });
      }
    } catch (e) {
      console.warn('Failed to load copiedOppositePredictions from localStorage', e);
    }
    try {
      const storedReports = JSON.parse(localStorage.getItem('oppositePredictionCopyReports') || '{}');
      if (storedReports && typeof storedReports === 'object') {
        const validReports = {};
        Object.keys(storedReports).forEach(key => {
          if (storedReports[key] && typeof storedReports[key] === 'object' && storedReports[key].status) {
            validReports[key] = storedReports[key];
          }
        });
        this.setState({ reports: validReports });
      }
    } catch (e) {
      console.warn('Failed to load oppositePredictionCopyReports from localStorage', e);
    }
  }

  // helper to parse stake from sms string (assumes smsString ends with #<stake>)
  parseStakeFromSms = (sms) => {
    if (!sms || typeof sms !== 'string') return 1;
    const parts = sms.split('#');
    const last = parts[parts.length - 1];
    const n = parseFloat(last);
    return Number.isFinite(n) ? n : 1;
  }

  // parse SportPesa reply messages (shared earlier). Returns { success, amount, possibleWin, balance, raw }
  parseSportPesaMessage = (msg) => {
    if (!msg || typeof msg !== 'string') return { success: false };
    const s = msg.replace(/\s+/g, ' ');
    const confirmed = /confirm(?:ed)?\b/i.test(s) || /Your bet is confirmed/i.test(s) || /BetID.*confirmed/i.test(s);
    const amountMatch = s.match(/Amount\s*:\s*KSH\s*([\d,]+(?:\.\d+)?)/i) || s.match(/Amount\s*:\s*([\d,]+(?:\.\d+)?)/i);
    const possibleWinMatch = s.match(/Possible win(?: after tax)?:\s*KSH\s*([\d,]+(?:\.\d+)?)/i) || s.match(/Possible win(?: after tax)?:\s*([\d,]+(?:\.\d+)?)/i);
    const balanceMatch = s.match(/Your\s*S-?PESA\s+balance\s*:\s*KSH\s*([\d,]+(?:\.\d+)?)/i) || s.match(/balance\s*:\s*KSH\s*([\d,]+(?:\.\d+)?)/i);
    const amount = amountMatch ? parseFloat(String(amountMatch[1]).replace(/,/g, '')) : null;
    const possibleWin = possibleWinMatch ? parseFloat(String(possibleWinMatch[1]).replace(/,/g, '')) : null;
    const balance = balanceMatch ? parseFloat(String(balanceMatch[1]).replace(/,/g, '')) : null;
    return { success: !!confirmed, amount, possibleWin, balance, raw: msg };
  }

  persistVisited = (visitedMap) => {
    try {
      localStorage.setItem('copiedOppositePredictions', JSON.stringify(visitedMap || {}));
    } catch (e) {
      console.warn('Failed to persist visitedMap', e);
    }
  };

  persistConfirmed = (confirmedMap) => {
    try {
      localStorage.setItem('confirmedOppositePredictions', JSON.stringify(confirmedMap || {}));
    } catch (e) {
      console.warn('Failed to persist confirmedMap', e);
    }
  };

  persistReports = (reports) => {
    try {
      const validReports = {};
      Object.keys(reports || {}).forEach(key => {
        if (reports[key] && typeof reports[key] === 'object' && reports[key].status) {
          validReports[key] = reports[key];
        }
      });
      localStorage.setItem('oppositePredictionCopyReports', JSON.stringify(validReports));
    } catch (e) {
      console.warn('Failed to persist reports', e);
    }
  };

  fetchEvents() {
    const requestBody = {
      query: `
        query {
          products {
            _id
            name
            description
            date
            category
            brand
            image
            price
            creator {
              _id
              email
            }
          }
        }
      `
    };

    fetch('http://localhost:5000/graphql', {
      method: 'POST',
      body: JSON.stringify(requestBody),
      headers: {
        'Content-Type': 'application/json'
      }
    })
      .then(res => {
        if (res.status !== 200 && res.status !== 201) {
          throw new Error('Failed!');
        }
        return res.json();
      })
      .then(resData => {
        const products = resData.data.products;
        this.setState({ products: products });
      })
      .catch(err => {
        console.log(err);
      });
  }

  fetchOppositePredictions() {
    console.log('Starting fetchOppositePredictions...');
    fetch('http://localhost:5000/api/oppositepredictions')
      .then(res => {
        if (res.status !== 200 && res.status !== 201) {
          throw new Error('Failed to fetch opposite predictions!');
        }
        return res.json();
      })
      .then(oppositePredictionsData => {
        // Support new backend format: { results: [...] }
        let predictionsArray = [];
        if (Array.isArray(oppositePredictionsData)) {
          predictionsArray = oppositePredictionsData;
        } else if (Array.isArray(oppositePredictionsData.results)) {
          predictionsArray = oppositePredictionsData.results;
        } else if (typeof oppositePredictionsData === 'object') {
          predictionsArray = Object.values(oppositePredictionsData);
        }

        // Add _localId for React keying
        const withLocalIds = predictionsArray.map((p, i) => ({
          _localId: p.predictionId || `opposite-local-${i}`,
          ...p
        }));

        // filter out confirmed predictions so they don't repeat
        const confirmed = this.state && this.state.confirmedMap ? this.state.confirmedMap : {};
        const filtered = withLocalIds.filter(p => {
          const id = p.predictionId || p._localId;
          return !confirmed[id];
        });

        this.setState(prev => {
          const newVisitedMap = {};
          const newReports = {};
          filtered.forEach(p => {
            const id = p.predictionId || p._localId;
            newVisitedMap[id] = prev.visitedMap[id] || false;
            newReports[id] = prev.reports[id] || {};
          });
          this.persistVisited(newVisitedMap);
          this.persistReports(newReports);
          return {
            oppositePredictions: filtered,
            visitedMap: newVisitedMap,
            reports: newReports
          };
        });
      })
      .catch(err => {
        console.log(err);
      });
  }

  showDetailHandler = productId => {
    this.setState(prevState => {
      const selectedEvent = prevState.products.find(e => e._id === productId);
      return { selectedEvent: selectedEvent };
    });
  };

  handleBuyClick = (prediction, index) => {
    if (!prediction) return;
    
    // Mark as visited
    const predId = prediction.predictionId || prediction._localId;
    if (predId) {
      this.setState(prev => {
        const visitedNext = { ...(prev.visitedMap || {}), [predId]: true };
        this.persistVisited(visitedNext);
        return { visitedMap: visitedNext };
      });
    }
    
    // Navigate to prediction details page with opposite prediction data
    if (this.props.navigate) {
      this.props.navigate('/prediction-details', {
        state: { 
          prediction: {
            ...prediction,
            isOpposite: true, // Flag to indicate this is an opposite prediction
            originalStrategy: 'contrarian_betting'
          }
        }
      });
    } else {
      // Fallback for development - show alert if navigate is not available
      alert(`Navigation not available. Prediction: ${prediction.predictionName || 'Opposite Prediction'}`);
    }
  };

  bookEventHandler = () => {
    if (!this.context.token) {
      this.setState({ selectedEvent: null });
      return;
    }
    const requestBody = {
      query: `
        mutation BookEvent($id: ID!) {
          bookEvent(productId: $id) {
            _id
            createdAt
            updatedAt
          }
        }
      `,
      variables: {
        id: this.state.selectedEvent._id
      }
    };

    fetch('http://localhost:8000/graphql', {
      method: 'POST',
      body: JSON.stringify(requestBody),
      headers: {
        'Content-Type': 'application/json',
        Authorization: 'Bearer ' + this.context.token
      }
    })
      .then(res => {
        if (res.status !== 200 && res.status !== 201) {
          throw new Error('Failed!');
        }
        return res.json();
      })
      .then(resData => {
        console.log(resData);
        this.setState({ selectedEvent: null });
      })
      .catch(err => {
        console.log(err);
      });
  };

  componentWillUnmount() {
    this.isActive = false;
    // Clean up autobet when component unmounts
    this.emergencyStopAutobet();
  }

  handleCopySms = async (prediction) => {
    if (!prediction) {
      console.warn('Missing opposite prediction', prediction);
      return;
    }

    const predId = prediction.predictionId || prediction._localId;
    if (!predId) {
      console.warn('Opposite prediction missing id and _localId', prediction);
      return;
    }

    if (this.state.copyingMap && this.state.copyingMap[predId]) return;

    this.setState(prev => ({ copyingMap: { ...prev.copyingMap, [predId]: true } }));

    try {
      let smsText = prediction.smsString;

      if (!smsText) {
        const resp = await fetch('http://localhost:5000/api/createSmsBets', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({ predictions: [prediction], stake: 1 })
        });

        if (!resp.ok) {
          const body = await resp.text().catch(() => '');
          throw new Error(`createSmsBets failed: ${resp.status} ${resp.statusText} ${body}`);
        }

        const data = await resp.json();
        // If server couldn't build smsBets, inspect errors to decide whether to skip gracefully
        if (!data || !data.smsBets || !data.smsBets[0] || !data.smsBets[0].smsString) {
          const errs = (data && data.errors) || [];
          const startedErr = errs.find(e => e && e.note === 'smsId_not_found_or_game_started');
          if (startedErr) {
            const skipMsg = 'One or more games in this opposite prediction have already started and cannot be SMS-bet. Prediction skipped.';
            const errorReport = { status: 'error', message: skipMsg, timestamp: new Date().toISOString() };
            this.setState(prev => {
              const reportsNext = { ...(prev.reports || {}), [predId]: errorReport };
              this.persistReports(reportsNext);
              return { reports: reportsNext };
            });
            alert(skipMsg);
            return;
          }
          throw new Error('Invalid response when creating sms bets for opposite prediction');
        }
        smsText = data.smsBets[0].smsString;

        this.setState(prev => ({
          oppositePredictions: prev.oppositePredictions.map(p => {
            const id = p.predictionId || p._localId;
            return id === predId ? { ...p, smsString: smsText } : p;
          })
        }));
      }

      if (navigator && navigator.clipboard && navigator.clipboard.writeText) {
        await navigator.clipboard.writeText(smsText);
      } else {
        const ta = document.createElement('textarea');
        ta.value = smsText;
        ta.style.position = 'fixed';
        ta.style.left = '-999999px';
        ta.style.top = '-999999px';
        document.body.appendChild(ta);
        ta.select();
        try {
          document.execCommand('copy');
        } catch (err) {
          console.error('Unable to copy to clipboard', err);
        }
        document.body.removeChild(ta);
      }

      const successReport = {
        status: 'success',
        message: smsText,
        timestamp: new Date().toISOString()
      };
      this.setState(prev => {
        const reportsNext = { ...(prev.reports || {}), [predId]: successReport };
        this.persistReports(reportsNext);
        return { reports: reportsNext };
      });

      this.setState(prev => ({ copiedMap: { ...prev.copiedMap, [predId]: true } }));

      this.setState(prev => {
        const visitedNext = { ...(prev.visitedMap || {}), [predId]: true };
        this.persistVisited(visitedNext);
        return { visitedMap: visitedNext };
      });

      setTimeout(() => {
        this.setState(prev => {
          const cm = { ...(prev.copiedMap || {}) };
          delete cm[predId];
          return { copiedMap: cm };
        });
      }, 4000);
    } catch (err) {
      const errMsg = err && err.message ? err.message : String(err);
      const errorReport = {
        status: 'error',
        message: errMsg,
        timestamp: new Date().toISOString()
      };
      this.setState(prev => {
        const reportsNext = { ...(prev.reports || {}), [predId]: errorReport };
        this.persistReports(reportsNext);
        return { reports: reportsNext };
      });
      console.error('Failed to copy SMS for opposite prediction', predId, err);
    } finally {
      this.setState(prev => {
        const cm = { ...(prev.copyingMap || {}) };
        delete cm[predId];
        return { copyingMap: cm };
      });
    }
  };

  handleSendTo79079 = async (prediction) => {
    if (!prediction) {
      console.warn('Missing opposite prediction', prediction);
      return;
    }

    const predId = prediction.predictionId || prediction._localId;
    if (!predId) {
      console.warn('Opposite prediction missing id and _localId', prediction);
      return;
    }

    // Build or retrieve sms text (same logic as handleCopySms without clipboard use)
    try {
      let smsText = prediction.smsString;

      if (!smsText) {
        const resp = await fetch('http://localhost:5000/api/createSmsBets', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({ predictions: [prediction], stake: 1 })
        });

        if (!resp.ok) {
          const body = await resp.text().catch(() => '');
          throw new Error(`createSmsBets failed: ${resp.status} ${resp.statusText} ${body}`);
        }

        const data = await resp.json();
        if (!data || !data.smsBets || !data.smsBets[0] || !data.smsBets[0].smsString) {
          const errs = (data && data.errors) || [];
          const startedErr = errs.find(e => e && e.note === 'smsId_not_found_or_game_started');
          if (startedErr) {
            const skipMsg = 'One or more games in this opposite prediction have already started and cannot be SMS-bet. Prediction skipped.';
            const errorReport = { status: 'error', message: skipMsg, timestamp: new Date().toISOString() };
            this.setState(prev => {
              const reportsNext = { ...(prev.reports || {}), [predId]: errorReport };
              this.persistReports(reportsNext);
              return { reports: reportsNext };
            });
            alert(skipMsg);
            return;
          }
          throw new Error('Invalid response when creating sms bets for opposite prediction');
        }
        smsText = data.smsBets[0].smsString;

        this.setState(prev => ({
          oppositePredictions: prev.oppositePredictions.map(p => {
            const id = p.predictionId || p._localId;
            return id === predId ? { ...p, smsString: smsText } : p;
          })
        }));
      }

      if (!smsText || typeof smsText !== 'string' || !smsText.trim()) {
        throw new Error('SMS body is empty');
      }

      // Prepare sms: URL. Use body param and encode properly. Android and iOS differ; for Android sms:79079?body=... should work.
      const encoded = encodeURIComponent(smsText);
      const smsUrl = `sms:79079?body=${encoded}`;

      // Try to open using window.location (will open native SMS app on mobile). If in desktop, it may fail.
      let opened = false;
      try {
        window.location.href = smsUrl;
        opened = true;
      } catch (e) {
        // ignore, we'll try other fallbacks
        opened = false;
      }

      // Another attempt via anchor click to be robust in some browsers
      if (!opened) {
        try {
          const a = document.createElement('a');
          a.href = smsUrl;
          a.style.display = 'none';
          document.body.appendChild(a);
          a.click();
          document.body.removeChild(a);
          opened = true;
        } catch (e) {
          opened = false;
        }
      }

      if (!opened) {
        // Couldn't open SMS composer — record an error report for this prediction
        const errorReport = {
          status: 'error',
          message: 'Failed to open SMS composer for 79079',
          timestamp: new Date().toISOString()
        };
        this.setState(prev => {
          const reportsNext = { ...(prev.reports || {}), [predId]: errorReport };
          this.persistReports(reportsNext);
          return { reports: reportsNext };
        });
        alert('Unable to open SMS composer on this device/browser. SMS not sent.');
        return;
      }

      // If opened, mark as success (user still needs to send). We record the sms body so it can be reviewed.
      const successReport = {
        status: 'success',
        message: `Opened composer for 79079 — body: ${smsText}`,
        timestamp: new Date().toISOString()
      };
      this.setState(prev => {
        const reportsNext = { ...(prev.reports || {}), [predId]: successReport };
        this.persistReports(reportsNext);
        return { reports: reportsNext };
      });

    } catch (err) {
      const errMsg = err && err.message ? err.message : String(err);
      const errorReport = {
        status: 'error',
        message: errMsg,
        timestamp: new Date().toISOString()
      };
      this.setState(prev => {
        const reportsNext = { ...(prev.reports || {}), [predId]: errorReport };
        this.persistReports(reportsNext);
        return { reports: reportsNext };
      });
      console.error('Failed to send SMS to 79079 for opposite prediction', predId, err);
      alert('Failed to prepare SMS for 79079: ' + errMsg);
    }
  };

  handleProgrammaticSend = async (prediction) => {
    if (!prediction) return;
    const predId = prediction.predictionId || prediction._localId;
    if (!predId) return;
    if (this.state.sendingMap && this.state.sendingMap[predId]) return;

    this.setState(prev => ({ sendingMap: { ...(prev.sendingMap || {}), [predId]: true } }));
    try {
      let smsText = prediction.smsString;
      if (!smsText) {
        const resp = await fetch('http://localhost:5000/api/createSmsBets', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({ predictions: [prediction], stake: 1 })
        });
        if (!resp.ok) {
          const body = await resp.text().catch(() => '');
          throw new Error(`createSmsBets failed: ${resp.status} ${resp.statusText} ${body}`);
        }
        const data = await resp.json();
        if (!data || !data.smsBets || !data.smsBets[0] || !data.smsBets[0].smsString) {
          const errReport = { status: 'error', message: 'Invalid response when creating sms bets for opposite prediction', timestamp: new Date().toISOString() };
          this.setState(prev => {
            const reportsNext = { ...(prev.reports || {}), [predId]: errReport };
            this.persistReports(reportsNext);
            return { reports: reportsNext };
          });
          return { report: errReport, smsString: smsText };
        }
        smsText = data.smsBets[0].smsString;
        this.setState(prev => ({
          oppositePredictions: prev.oppositePredictions.map(p => {
            const id = p.predictionId || p._localId;
            return id === predId ? { ...p, smsString: smsText } : p;
          })
        }));
      }

      if (!smsText || !smsText.trim()) throw new Error('SMS body is empty');

        // Use the Gradle bridge to invoke the helper via adb and wait for prediction_result.json
        const controller = new AbortController();
        const TIMEOUT_MS = 40000; // 40 seconds
        let timedOut = false;
        const timeoutId = setTimeout(() => {
          timedOut = true;
          try { controller.abort(); } catch (e) { /* ignore */ }
        }, TIMEOUT_MS);

        let jr = null;
        try {
          // Check for emergency stop before sending
          if (this.autobetCancel) {
            console.log('Emergency stop: Cancelling bridge send');
            clearTimeout(timeoutId);
            return { report: { status: 'cancelled', message: 'Emergency stop activated', timestamp: new Date().toISOString() } };
          }

          const bridgeResp = await fetch('http://localhost:5021/send', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ to: '79079', message: smsText }),
            signal: controller.signal
          });
          clearTimeout(timeoutId);
            if (!bridgeResp.ok) {
              const txt = await bridgeResp.text().catch(() => '');
              // record as error and remove prediction
              const errMsg = `Bridge returned ${bridgeResp.status} ${bridgeResp.statusText} ${txt}`;
              const errorReport = { status: 'error', message: errMsg, timestamp: new Date().toISOString() };
              this.setState(prev => {
                const reportsNext = { ...(prev.reports || {}), [predId]: errorReport };
                this.persistReports(reportsNext);
                const confirmedNext = { ...(prev.confirmedMap || {}), [predId]: true };
                this.persistConfirmed(confirmedNext);
                const preds = (prev.oppositePredictions || []).filter(p => (p.predictionId || p._localId) !== predId);
                return { reports: reportsNext, confirmedMap: confirmedNext, oppositePredictions: preds };
              });
              alert('Bridge error: ' + errMsg);
              return { report: errorReport, smsString: smsText };
            }
            jr = await bridgeResp.json().catch(() => null);
        } catch (err) {
          clearTimeout(timeoutId);
          if (timedOut) {
            // Timeout: treat as no response and remove from list
            const errMsg = 'No response from device controller (timeout after 40s)';
            const errorReport = { status: 'error', message: errMsg, timestamp: new Date().toISOString() };
            this.setState(prev => {
              const reportsNext = { ...(prev.reports || {}), [predId]: errorReport };
              this.persistReports(reportsNext);
              const confirmedNext = { ...(prev.confirmedMap || {}), [predId]: true };
              this.persistConfirmed(confirmedNext);
              const preds = (prev.oppositePredictions || []).filter(p => (p.predictionId || p._localId) !== predId);
              return { reports: reportsNext, confirmedMap: confirmedNext, oppositePredictions: preds };
            });
            alert('No response from device (timed out after 40s). Marked as error and removed from list.');
            return;
          }
          // other fetch error
          const errMsg = err && err.message ? err.message : String(err);
          const errorReport = { status: 'error', message: errMsg, timestamp: new Date().toISOString() };
          this.setState(prev => {
            const reportsNext = { ...(prev.reports || {}), [predId]: errorReport };
            this.persistReports(reportsNext);
            return { reports: reportsNext };
          });
          throw err;
        }

      // Analyze controller response
      // jr.confirmation may contain { confirmed: true, betId } or { confirmed: false, error, reason }
      if (jr && jr.confirmation && jr.confirmation.confirmed) {
        const successReport = {
          status: 'success',
          message: `Sent via device; confirmed BetID ${jr.confirmation.betId || '(unknown)'}`,
          timestamp: new Date().toISOString()
        };
        this.setState(prev => {
          const reportsNext = { ...(prev.reports || {}), [predId]: successReport };
          this.persistReports(reportsNext);
          // mark confirmed so the bet disappears from the list and won't repeat
          const confirmedNext = { ...(prev.confirmedMap || {}), [predId]: true };
          this.persistConfirmed(confirmedNext);
          // remove from predictions array
          const preds = (prev.oppositePredictions || []).filter(p => (p.predictionId || p._localId) !== predId);
          return { reports: reportsNext, confirmedMap: confirmedNext, oppositePredictions: preds };
        });
        // persist confirmation on server for global dedupe
        try {
          fetch('http://localhost:5000/api/confirmOppositePrediction', { method: 'POST', headers: { 'Content-Type': 'application/json' }, body: JSON.stringify({ predictionId: predId }) }).catch(e => console.warn('confirmOppositePrediction failed', e));
        } catch (e) { /* ignore */ }
        alert(`Opposite bet confirmed (BetID: ${jr.confirmation.betId || 'unknown'})`);
        return { report: successReport, smsString: smsText };
      } else if (jr && jr.confirmation && jr.confirmation.error) {
        // Device reported an explicit error (treat as processed and remove)
        const errorReport = { status: 'error', message: `Device error: ${jr.confirmation.error} ${jr.confirmation.reason || ''}`, timestamp: new Date().toISOString() };
        this.setState(prev => {
          const reportsNext = { ...(prev.reports || {}), [predId]: errorReport };
          this.persistReports(reportsNext);
          const confirmedNext = { ...(prev.confirmedMap || {}), [predId]: true };
          this.persistConfirmed(confirmedNext);
          const preds = (prev.oppositePredictions || []).filter(p => (p.predictionId || p._localId) !== predId);
          return { reports: reportsNext, confirmedMap: confirmedNext, oppositePredictions: preds };
        });
        alert('Device reported an error; opposite prediction removed from list.');
        return { report: errorReport, smsString: smsText };
      } else if (jr && (jr.sent || (jr.sendResult && jr.sendResult.status === 'sent'))) {
        // Sent but no immediate confirmation
        const pendingReport = {
          status: 'success',
          message: `Sent via device; awaiting confirmation (no immediate confirmation)`,
          timestamp: new Date().toISOString()
        };
        this.setState(prev => {
          const reportsNext = { ...(prev.reports || {}), [predId]: pendingReport };
          this.persistReports(reportsNext);
          return { reports: reportsNext };
        });
        alert('Message sent via device. Waiting for confirmation (server will update queue status).');
        return { report: pendingReport, smsString: smsText };
      } else if (jr && jr.confirmation && jr.confirmation.error) {
        const errorReport = { status: 'error', message: `Device reported error: ${jr.confirmation.error} ${jr.confirmation.reason || ''}`, timestamp: new Date().toISOString() };
        this.setState(prev => {
          const reportsNext = { ...(prev.reports || {}), [predId]: errorReport };
          this.persistReports(reportsNext);
          return { reports: reportsNext };
        });
        alert('Device reported an error while sending. See reports panel for details.');
        return { report: errorReport, smsString: smsText };
      } else {
        // Default fallback — record the raw response for debugging
        const infoReport = { status: 'error', message: `Unexpected controller response: ${JSON.stringify(jr)}`, timestamp: new Date().toISOString() };
        this.setState(prev => {
          const reportsNext = { ...(prev.reports || {}), [predId]: infoReport };
          this.persistReports(reportsNext);
          return { reports: reportsNext };
        });
        alert('Unexpected controller response; recorded in reports.');
        return { report: infoReport, smsString: smsText };
      }

    } catch (err) {
      const errMsg = err && err.message ? err.message : String(err);
      const errorReport = { status: 'error', message: errMsg, timestamp: new Date().toISOString() };
      this.setState(prev => {
        const reportsNext = { ...(prev.reports || {}), [predId]: errorReport };
        this.persistReports(reportsNext);
        return { reports: reportsNext };
      });
      console.error('Programmatic send failed for opposite prediction', err);
      alert('Programmatic send failed: ' + errMsg);
      return { report: errorReport, smsString: null };
    } finally {
      this.setState(prev => {
        const s = { ...(prev.sendingMap || {}) };
        delete s[predId];
        return { sendingMap: s };
      });
    }
  };

  handlePreviewSms = async (prediction) => {
    if (!prediction) return;
    try {
      const resp = await fetch('http://localhost:5000/api/createSmsBets', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ predictions: [prediction], stake: 1 })
      });
      if (!resp.ok) {
        const body = await resp.text().catch(() => '');
        throw new Error(`createSmsBets failed: ${resp.status} ${resp.statusText} ${body}`);
      }
      const data = await resp.json();
      if (data && data.smsBets && data.smsBets[0] && data.smsBets[0].smsString) {
        const sms = data.smsBets[0].smsString;
        const isPartial = !!data.smsBets[0].partial;
        const perLegErrors = data.smsBets[0].perLegErrors || [];
        const msg = `Preview SMS (Opposite Prediction):\n${sms}\n${isPartial ? '\n(Note: this is a PARTIAL bet — some legs could not be resolved)' : ''}${perLegErrors.length ? '\nPer-leg issues: ' + JSON.stringify(perLegErrors) : ''}`;
        alert(msg);
      } else {
        const errs = (data && data.errors) || [];
        const startedErr = errs.find(e => e && e.note === 'smsId_not_found_or_game_started');
        if (startedErr) {
          alert('Cannot preview: one or more games already started and no sms could be built for this opposite prediction.');
          return;
        }
        alert('Unable to create SMS preview for opposite prediction (invalid server response).');
      }
    } catch (err) {
      console.error('Preview SMS failed for opposite prediction', err);
      alert('Preview SMS failed: ' + (err && err.message ? err.message : String(err)));
    }
  };

  // Check for emergency stop flag from server
  checkEmergencyStop = async () => {
    try {
      const response = await fetch('http://localhost:5000/api/emergency-stop-autobet');
      if (response.ok) {
        const data = await response.json();
        return data.stopped === true;
      }
    } catch (e) {
      // If we can't check, don't stop - server might be down
      console.warn('Failed to check emergency stop flag', e);
    }
    return false;
  };

  // Emergency stop function to completely halt autobet
  emergencyStopAutobet = () => {
    this.autobetCancel = true;
    if (this.autobetTimeout) {
      clearTimeout(this.autobetTimeout);
      this.autobetTimeout = null;
    }
    this.setState({ autobetRunning: false, autobetStats: null });
    // Clear localStorage autobet state if it exists
    try {
      localStorage.removeItem('autobetRunning');
      localStorage.removeItem('autobetStats');
    } catch (e) {
      console.warn('Failed to clear autobet localStorage', e);
    }
    console.log('EMERGENCY STOP: Autobet cancelled');
    alert('EMERGENCY STOP: Autobet has been cancelled and cleared');
    
    // Also set server-side emergency stop flag
    fetch('http://localhost:5000/api/emergency-stop-autobet', {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' }
    }).catch(e => console.warn('Failed to set server emergency stop flag', e));
  };

  // Run auto-bet across all visible opposite predictions that are not confirmed or errored
  runAutoBetAll = async (opts = { delayMs: 1500 }) => {
    // Get target from user input or use default
    const targetInput = prompt(`Enter number of successful bets to place (default: ${this.state.targetSuccessfulBets}):`);
    const target = targetInput ? parseInt(targetInput) : this.state.targetSuccessfulBets;
    
    if (isNaN(target) || target <= 0) {
      alert('Please enter a valid number greater than 0');
      return;
    }

    // autobet control - clear any existing timeouts
    if (this.autobetTimeout) {
      clearTimeout(this.autobetTimeout);
      this.autobetTimeout = null;
    }
    this.autobetCancel = false;
    this.setState({ autobetRunning: true, autobetStats: null });

    const preds = (this.state.oppositePredictions || []).filter(p => {
      const id = p.predictionId || p._localId;
      const r = this.state.reports && this.state.reports[id];
      // skip ones that already have status 'success' or 'error' (no repeats)
      if (r && (r.status === 'success' || r.status === 'error')) return false;
      return true;
    });

    const total = preds.length;
    let successes = 0, failures = 0, placedStakeTotal = 0;
    // read starting balance from a previously parsed report if available, else undefined
    let balanceRemaining = undefined;

    if (total === 0) {
      alert('No available opposite predictions to autobet (all have been processed)');
      this.setState({ autobetRunning: false });
      return;
    }

    for (let i = 0; i < preds.length && successes < target; i++) {
      if (this.autobetCancel) {
        console.log('Autobet cancelled by user');
        break;
      }
      
      // Check for server-side emergency stop
      const emergencyStop = await this.checkEmergencyStop();
      if (emergencyStop) {
        console.log('Autobet stopped by server emergency flag');
        this.emergencyStopAutobet();
        break;
      }
      const p = preds[i];
      const id = p.predictionId || p._localId;
      // call programmatic send and await result (we already mark timeouts as errors)
      try {
        const sendResult = await this.handleProgrammaticSend(p);
        const rep = (sendResult && sendResult.report) ? sendResult.report : ((this.state.reports || {})[id] || {});
        const smsStr = (sendResult && sendResult.smsString) || p.smsString || '';
        // if the bridge included a parsed device reply in report.message (e.g., contains S-PESA balance), parse it
        if (rep && rep.message) {
          const parsed = this.parseSportPesaMessage(rep.message || '');
          if (parsed && typeof parsed.balance === 'number') {
            balanceRemaining = parsed.balance; // prefer latest
          }
        }
        if (rep.status === 'success') {
          successes += 1;
          // subtract stake from balance if we can parse
          const stake = this.parseStakeFromSms(smsStr || '');
          placedStakeTotal += stake;
          if (typeof balanceRemaining === 'number') balanceRemaining -= stake;
        } else {
          failures += 1;
          // mark failed on server so legs go to failed pool
          try {
            await fetch('http://localhost:5000/api/markFailedOppositePrediction', {
              method: 'POST', headers: { 'Content-Type': 'application/json' }, body: JSON.stringify({ predictionId: id, prediction: p, reason: rep.message || 'auto-failed' })
            });
          } catch (e) {
            console.warn('Failed to mark failed opposite prediction on server', e);
          }
        }
      } catch (err) {
        failures += 1;
      }

      // update progress UI
      this.setState({ autobetStats: { total, successes, failures, placedStakeTotal, balanceRemaining, remaining: total - successes - failures, currentIndex: i + 1, target: target } });

      // Stop if we reached the target
      if (successes >= target) {
        alert(`Target reached! Successfully placed ${successes} opposite bets.`);
        break;
      }

      // stop if balance low (if known)
      if (typeof balanceRemaining === 'number') {
        const minStake = (opts && opts.minStake) || 1;
        if (balanceRemaining < minStake) {
          alert('Stopping Auto-Bet: insufficient balance (' + balanceRemaining + ')');
          break;
        }
      }

      // simple delay between sends - but check for cancellation during delay
      await new Promise(resolve => {
        this.autobetTimeout = setTimeout(() => {
          this.autobetTimeout = null;
          resolve();
        }, opts.delayMs || 1000);
      });
      
      // Check again after delay
      if (this.autobetCancel) {
        console.log('Autobet cancelled during delay');
        break;
      }
    }

    const remaining = total - successes - failures;
    const stats = { total, remaining, successes, failures, placedStakeTotal, balanceRemaining, target: target };
    this.setState({ autobetStats: stats, autobetRunning: false });
    
    const finalMessage = successes >= target 
      ? `Target reached! Successfully placed ${successes} opposite bets out of ${target} target.`
      : `AutoBet completed for opposite predictions.\nSuccesses: ${successes}/${target}\nFailures: ${failures}\nRemaining: ${remaining}`;
    alert(finalMessage);
    return stats;
  }

  handleConfirmSuccess = (prediction) => {
    const predId = prediction.predictionId || prediction._localId;
    if (!predId) return;
    const report = { status: 'success', message: 'User confirmed success', timestamp: new Date().toISOString() };
    this.setState(prev => {
      const reportsNext = { ...(prev.reports || {}), [predId]: report };
      this.persistReports(reportsNext);
      const confirmedNext = { ...(prev.confirmedMap || {}), [predId]: true };
      this.persistConfirmed(confirmedNext);
      const preds = (prev.oppositePredictions || []).filter(p => (p.predictionId || p._localId) !== predId);
      return { reports: reportsNext, confirmedMap: confirmedNext, oppositePredictions: preds };
    });
  };

  handleReportError = (prediction) => {
    const predId = prediction.predictionId || prediction._localId;
    if (!predId) return;
    const msg = window.prompt('Paste SportPesa error message (or short description):');
    if (msg === null) return;
    const report = {
      status: 'error',
      message: msg || 'Reported error (no message provided)',
      timestamp: new Date().toISOString()
    };
    this.setState(prev => {
      const reportsNext = { ...(prev.reports || {}), [predId]: report };
      this.persistReports(reportsNext);
      return { reports: reportsNext };
    });
  };

  computeReportCounts = () => {
    const { reports } = this.state;
    let success = 0;
    let error = 0;
    if (reports && typeof reports === 'object') {
      Object.values(reports).forEach(r => {
        if (!r || !r.status) return;
        if (r.status === 'success') success += 1;
        else if (r.status === 'error') error += 1;
      });
    }
    return { success, error, total: success + error };
  };

  handleExportReports = () => {
    try {
      const data = JSON.stringify(this.state.reports || {}, null, 2);
      const blob = new Blob([data], { type: 'application/json' });
      const url = URL.createObjectURL(blob);
      const a = document.createElement('a');
      a.href = url;
      a.download = `opposite_prediction_reports_${new Date().toISOString().replace(/[:.]/g,'-')}.json`;
      document.body.appendChild(a);
      a.click();
      document.body.removeChild(a);
      URL.revokeObjectURL(url);
    } catch (e) {
      console.error('Failed to export reports', e);
      alert('Failed to export reports. See console for details.');
    }
  };

  handleUploadReports = async () => {
    try {
      const body = { reports: this.state.reports || {} };
      const resp = await fetch('http://localhost:5000/api/oppositeReports', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(body)
      });
      if (!resp.ok) {
        const text = await resp.text().catch(() => '');
        throw new Error(`${resp.status} ${resp.statusText} ${text}`);
      }
      alert('Opposite prediction reports uploaded successfully for system improvement.');
    } catch (err) {
      console.error('Failed to upload opposite reports', err);
      alert('Failed to upload opposite reports: ' + (err.message || String(err)));
    }
  };

  renderPredictionStatus = (pred) => {
    const id = pred.predictionId || pred._localId;
    const { copiedMap, reports } = this.state;
    const copied = !!(copiedMap && copiedMap[id]);
    const report = reports && reports[id];
    if (copied) {
      return <span style={{ marginLeft: 10, color: '#0a7' }}>Copied — opposite bet placed</span>;
    }
    if (report && report.status) {
      if (report.status === 'success') {
        return <span style={{ marginLeft: 10, color: '#0a7', fontSize: 12 }}>
          Last sent at {new Date(report.timestamp).toLocaleString()}
        </span>;
      }
      return <span style={{ marginLeft: 10, color: '#c00', fontSize: 12 }}>
        Error: {report.message}
      </span>;
    }
    return null;
  };

  renderReportsPanel = () => {
    const { reports } = this.state;
    const keys = reports
      ? Object.keys(reports).filter(k => reports[k] && typeof reports[k] === 'object' && reports[k].status)
      : [];
    if (!keys.length) return null;
    return (
      <div style={{ marginTop: 20, padding: 12, border: '1px solid #ddd', borderRadius: 6 }}>
        <h4 style={{ marginTop: 0 }}>Opposite Prediction Copy reports (last per prediction)</h4>
        <ul style={{ paddingLeft: 18 }}>
          {keys.map(k => {
            const r = reports[k];
            return (
              <li key={k} style={{ marginBottom: 8 }}>
                <strong>{k}</strong> — {r.status === 'success' ? <span style={{ color: '#0a7' }}>Success</span> : <span style={{ color: '#c00' }}>Error</span>}
                <div style={{ fontSize: 12, color: '#444' }}>{r.message}</div>
                <div style={{ fontSize: 11, color: '#666' }}>{new Date(r.timestamp).toLocaleString()}</div>
              </li>
            );
          })}
        </ul>
        <button onClick={this.handleExportReports}>Export Reports</button>
        <button onClick={this.handleUploadReports} style={{ marginLeft: 8 }}>Upload Reports to Server</button>
      </div>
    );
  };

  render() {
    return (
      <React.Fragment>
        {this.state.selectedEvent && (
          <Modal
            title={this.state.selectedEvent.title}
            canCancel
            canConfirm
            onCancel={() => this.setState({ selectedEvent: null })}
            onConfirm={this.bookEventHandler}
            confirmText={this.context.token ? 'Book' : 'Confirm'}
          >
            <h1>{this.state.selectedEvent.name}</h1>
            <h2>
              ${this.state.selectedEvent.price} -{' '}
              {new Date(this.state.selectedEvent.date).toLocaleDateString()}
            </h2>
            <p>{this.state.selectedEvent.description}</p>
          </Modal>
        )}

        <div>
          <EventList
            products={this.state.products}
            authUserId={this.context.userId}
            onViewDetail={this.showDetailHandler}
          />
        </div>

        <div style={{ marginTop: 24 }}>
          <h2>Today's Opposite Predictions</h2>
          <div style={{ marginBottom: 12, backgroundColor: '#fff3cd', padding: '12px', borderRadius: '6px', border: '1px solid #ffeaa7' }}>
            <strong>⚠️ Notice:</strong> These are <strong>opposite predictions</strong> - selections have been flipped (e.g., UNDER becomes OVER). 
            This is a contrarian betting strategy where you bet against the original predictions.
          </div>
          <div style={{ marginBottom: 12 }}>
            <button onClick={() => this.runAutoBetAll({ delayMs: 1200 })} style={{ padding: '12px 20px', fontSize: 16, background: '#e67e22', color: '#fff', border: 'none', borderRadius: 6 }} disabled={this.state.autobetRunning}>
              Auto-Bet All Opposite (send unconfirmed)
            </button>
            <button onClick={this.emergencyStopAutobet} style={{ padding: '12px 16px', marginLeft: 8, backgroundColor: '#dc3545', color: 'white', border: 'none', borderRadius: 4 }}>🛑 EMERGENCY STOP</button>
            {this.state.autobetStats && (
              <div style={{ marginTop: 8, fontSize: 13 }}>
                <strong>AutoBet (Opposite):</strong> successes {this.state.autobetStats.successes}/{this.state.autobetStats.target || 'N/A'}, failures {this.state.autobetStats.failures}, remaining {this.state.autobetStats.remaining}
                {this.state.autobetStats.placedStakeTotal && ` | Stake: ${this.state.autobetStats.placedStakeTotal}`}
                {this.state.autobetStats.balanceRemaining && ` | Balance: KES ${this.state.autobetStats.balanceRemaining}`}
                {this.state.autobetStats.balanceRemaining !== undefined && `, balance: KES ${this.state.autobetStats.balanceRemaining}`}
              </div>
            )}
          </div>
          {(console.log('Render check: oppositePredictions.length =', this.state.oppositePredictions.length), this.state.oppositePredictions.length === 0) ? (
            <p>No opposite predictions available.</p>
          ) : (
            <ol>
              {this.state.oppositePredictions.map((pred, idx) => {
                const id = pred.predictionId || pred._localId || `opposite-local-${idx}`;
                const copying = !!this.state.copyingMap[id];
                const visited = !!this.state.visitedMap[id];
                const uniqueKey = `${id}-${idx}`; // Make key unique with index
                return (
                  <li key={uniqueKey} style={{ marginBottom: 12, backgroundColor: pred.status === 'error' ? '#ffeaea' : '#f8f9fa', padding: '10px', borderRadius: '6px', border: pred.status === 'error' ? '2px solid #c00' : '2px solid #e67e22' }}>
                    <div style={{ display: 'flex', alignItems: 'center', gap: 12 }}>
                      <div style={{ flex: 1 }}>
                        <strong style={{ color: pred.status === 'error' ? '#c00' : '#e67e22' }}>🔄 {pred.predictionName || `Opposite Prediction ${id}`}</strong>
                        <div style={{ fontSize: 13, color: '#555' }}>
                          {pred.numGames ? `${pred.numGames} game(s)` : ''}
                          {pred.totalOdds ? ` — Odds: ${pred.totalOdds}` : ''}
                          <span style={{ color: pred.status === 'error' ? '#c00' : '#e67e22', fontWeight: 'bold' }}> (OPPOSITE SELECTIONS)</span>
                        </div>
                        {pred.status === 'error' && (
                          <div style={{ color: '#c00', fontWeight: 'bold', marginTop: 6 }}>
                            Error: {pred.error}
                          </div>
                        )}
                        {pred.status === 'success' && pred.smsString && (
                          <div style={{ color: '#0a7', fontWeight: 'bold', marginTop: 6 }}>
                            SMS: {pred.smsString}
                          </div>
                        )}
                      </div>
                      <div>
                        <button 
                          onClick={() => this.handleBuyClick(pred, idx)} 
                          style={{ 
                            marginRight: 8, 
                            backgroundColor: pred.status === 'error' ? '#c00' : '#e67e22', 
                            color: 'white', 
                            border: 'none', 
                            padding: '6px 12px', 
                            borderRadius: '4px',
                            cursor: 'pointer'
                          }}
                        >
                          View Details
                        </button>
                        <button
                          onClick={() => this.handleCopySms(pred)}
                          disabled={copying || pred.status === 'error'}
                          style={visited ? { color: '#551A8B', textDecoration: 'underline' } : {}}
                        >
                          {copying ? (
                            <span>
                              Copying… <span style={{ fontSize: 12 }}>(⏳)</span>
                            </span>
                          ) : (visited ? 'Copy SMS (visited)' : 'Copy SMS')}
                        </button>
                        <button onClick={() => this.handlePreviewSms(pred)} style={{ marginLeft: 8 }} disabled={pred.status === 'error'}>
                          Preview SMS
                        </button>
                        <button
                          onClick={() => this.handleSendTo79079(pred)}
                          style={{ marginLeft: 8 }}
                          disabled={pred.status === 'error'}
                        >
                          Send to 79079
                        </button>
                        <button
                          onClick={() => this.handleProgrammaticSend(pred)}
                          style={{ marginLeft: 8 }}
                          disabled={pred.status === 'error'}
                        >
                          Send via device
                        </button>
                        <button onClick={() => this.handleConfirmSuccess(pred)} style={{ marginLeft: 8 }}>
                          Confirm Success
                        </button>
                        <button onClick={() => this.handleReportError(pred)} style={{ marginLeft: 8 }}>
                          Report Error
                        </button>
                        {this.renderPredictionStatus(pred)}
                      </div>
                    </div>
                  </li>
                );
              })}
            </ol>
          )}
        </div>
        {this.renderReportsPanel()}
      </React.Fragment>
    );
  }
}

// Wrapper component to provide navigate function to class component
const OppositeBetlistWithNavigation = () => {
  const navigate = useNavigate();
  return <OppositeBetlist navigate={navigate} />;
};

export default OppositeBetlistWithNavigation;