import React, { useContext, useRef, useState } from 'react';
import '../components/Navigation/Styles.css';
import '../components/Navigation/css';
import { AuthContext } from '../context/auth-context';
import { signInWithEmailAndPassword, RecaptchaVerifier, signInWithPhoneNumber } from 'firebase/auth';
import { auth } from '../firebase';
import { useNavigate } from 'react-router-dom';

export default function LoginPage() {
  const ctx = useContext(AuthContext);
  const emailEl = useRef(null);
  const passwordEl = useRef(null);
  const phoneEl = useRef(null);
  const codeEl = useRef(null);
  const [smsSent, setSmsSent] = useState(false);
  const [confirmationResult, setConfirmationResult] = useState(null);
  const navigate = useNavigate();

  // If already authenticated, redirect to /home
  React.useEffect(() => {
    if (ctx && ctx.user) {
      navigate('/home', { replace: true });
    }
  }, [ctx, navigate]);

  const loginHandler = async (e) => {
    e.preventDefault();
    const email = emailEl.current?.value || '';
    const password = passwordEl.current?.value || '';
    if (!email || !password) return alert('Enter email and password');
    try {
      const cred = await signInWithEmailAndPassword(auth, email, password);
      const user = cred.user;
      const token = await user.getIdToken();
      if (ctx && ctx.login) ctx.login(user, token);

      // fetch backend profile and save to localStorage with 130-minute expiry
      (async () => {
        try {
          const res = await fetch('http://localhost:5000/api/profile', { headers: { Authorization: `Bearer ${token}` } });
          if (res.ok) {
            const json = await res.json();
            const profile = json.profile || { displayName: user.displayName, contact: user.email };
            const expiresAt = Date.now() + 130 * 60 * 1000; // 130 minutes
            localStorage.setItem('user_profile', JSON.stringify({ profile, expiresAt }));
          }
        } catch (e) {
          // ignore profile fetch errors
        }
      })();

      // navigate to home; AuthContext updates will update UI without a full reload
      navigate('/', { replace: true });
    } catch (err) {
      console.error(err);
      alert('Login failed: ' + (err.message || err));
    }
  };

  const sendSmsHandler = async (e) => {
    e.preventDefault();
    const phone = phoneEl.current?.value || '';
    if (!phone) return alert('Enter phone number with country code (e.g. +2547...)');
    try {
      if (!window.recaptchaVerifier) {
        window.recaptchaVerifier = new RecaptchaVerifier('recaptcha-container', { size: 'invisible' }, auth);
        await window.recaptchaVerifier.render();
      }
      const result = await signInWithPhoneNumber(auth, phone, window.recaptchaVerifier);
      setConfirmationResult(result);
      setSmsSent(true);
      alert('SMS sent');
    } catch (err) {
      console.error('sendSmsHandler', err);
      alert('Failed to send SMS: ' + (err.message || err));
    }
  };

  const verifyCodeHandler = async (e) => {
    e.preventDefault();
    const code = codeEl.current?.value || '';
    if (!confirmationResult) return alert('No SMS request in progress');
    try {
      const cred = await confirmationResult.confirm(code);
      const user = cred.user;
      const token = await user.getIdToken();
      if (ctx && ctx.login) ctx.login(user, token);
      // fetch backend profile and save to localStorage with 130-minute expiry
      (async () => {
        try {
          const res = await fetch('http://localhost:5000/api/profile', { headers: { Authorization: `Bearer ${token}` } });
          if (res.ok) {
            const json = await res.json();
            const profile = json.profile || { displayName: user.displayName, contact: user.email };
            const expiresAt = Date.now() + 130 * 60 * 1000; // 130 minutes
            localStorage.setItem('user_profile', JSON.stringify({ profile, expiresAt }));
          }
        } catch (e) {
          // ignore profile fetch errors
        }
      })();

      // navigate to home; AuthContext updates will update UI without a full reload
      navigate('/', { replace: true });
    } catch (err) {
      console.error('verifyCodeHandler', err);
      alert('Code verification failed: ' + (err.message || err));
    }
  };

  return (
    <div>
      <aside id="notifications">
        <div className="container" />
      </aside>

      <div className="breadcrumb_container " data-depth={1}>
        <div className="container">
          <nav data-depth={1} className="breadcrumb">
            <ol itemScope itemType="http://schema.org/BreadcrumbList">
              <li itemProp="itemListElement" itemScope itemType="http://schema.org/ListItem">
                <span itemProp="name">Home</span>
                <meta itemProp="position" content={1} />
              </li>
            </ol>
          </nav>
        </div>
      </div>

      <div id="wrapper">
        <div className="container">
          <div className="row">
            <div id="content-wrapper" className="col-xs-12">
              <section id="main">
                <header className="page-header">
                  <h1>Log in to your account</h1>
                </header>

                <section id="content" className="page-content card card-block">
                  <section className="login-form">
                    <form id="login-form" onSubmit={loginHandler}>
                      <section>
                        <div className="form-group row ">
                          <label className="col-md-3 form-control-label required">Email</label>
                          <div className="col-md-6">
                            <input className="form-control" name="email" type="email" placeholder="your@email.com" ref={emailEl} />
                          </div>
                        </div>

                        <div className="form-group row ">
                          <label className="col-md-3 form-control-label required">Password</label>
                          <div className="col-md-6">
                            <div className="input-group js-parent-focus">
                              <input className="form-control js-child-focus js-visible-password" name="password" type="password" placeholder="Enter Password" id="password" ref={passwordEl} />
                            </div>
                          </div>
                        </div>

                        <div className="form-group row ">
                          <div className="col-md-9 offset-md-3">
                            <button className="btn btn-primary" type="submit">Sign in with Email</button>
                          </div>
                        </div>

                        <hr />

                        <div className="form-group row ">
                          <label className="col-md-3 form-control-label">Phone</label>
                          <div className="col-md-6">
                            <input className="form-control" name="phone" type="tel" placeholder="+2547..." ref={phoneEl} />
                            <div id="recaptcha-container"></div>
                            {!smsSent ? (
                              <button className="btn" onClick={sendSmsHandler} type="button">Send SMS code</button>
                            ) : (
                              <div style={{ marginTop: 8 }}>
                                <input ref={codeEl} placeholder="Enter SMS code" />
                                <button className="btn" onClick={verifyCodeHandler} type="button">Verify code</button>
                              </div>
                            )}
                          </div>
                        </div>

                      </section>
                    </form>
                  </section>

                  <hr />

                  <div className="no-account">
                    <a href="/register">No account? Create one here</a>
                  </div>

                </section>
              </section>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}
