import React, { useState, useEffect } from 'react';
import '../components/Navigation/Styles.css';
import '../components/Navigation/css';
import { useNavigate } from 'react-router-dom'; // Updated for v6
import Modal from '../components/Modal/Modal';
import './Events.css';
import EventList from '../components/Events/EventList/EventList';
import Prediction from '../components/Prediction'; // Import the Prediction component

const Home = () => {
  const navigate = useNavigate(); // React Router v6 hook

  const [state, setState] = useState({
    creating: true,
    products: [],
    predictions: [],
    selectedEvent: null
  });

  const { products, predictions, selectedEvent } = state;

  useEffect(() => {
    fetchEvents();
    fetchPredictions();

    // Cleanup on unmount (optional)
    return () => {
      // Any cleanup if needed
    };
  }, []);

  const fetchEvents = () => {
    const requestBody = {
      query: `
        query {
          products {
            _id
            name
            description
            date
            category
            brand
            image
            price
            creator {
              _id
              email
            }
          }
        }
      `
    };

    fetch('http://localhost:5000/graphql', {
      method: 'POST',
      body: JSON.stringify(requestBody),
      headers: {
        'Content-Type': 'application/json'
      }
    })
      .then(res => {
        if (res.status !== 200 && res.status !== 201) {
          throw new Error('Failed!');
        }
        return res.json();
      })
      .then(resData => {
        const fetchedProducts = resData.data.products;
        setState(prev => ({ ...prev, products: fetchedProducts }));
      })
      .catch(err => {
        console.log(err);
      });
  };

  const fetchPredictions = () => {
    fetch('http://localhost:5000/api/predictions')
      .then(res => {
        if (res.status !== 200 && res.status !== 201) {
          throw new Error('Failed to fetch predictions!');
        }
        return res.json();
      })
      .then(predictionsData => {
        const predictionsArray = Object.keys(predictionsData).map(key => predictionsData[key]);
        setState(prev => ({ ...prev, predictions: predictionsArray }));
      })
      .catch(err => {
        console.log(err);
      });
  };

  const showDetailHandler = productId => {
    const selectedEvent = products.find(e => e._id === productId);
    setState(prev => ({ ...prev, selectedEvent: selectedEvent }));
  };

  const bookEventHandler = () => {
    console.log(selectedEvent);
    const requestBody = {
      query: `
        mutation BookEvent($id: ID!) {
          bookEvent(productId: $id) {
            _id
            createdAt
            updatedAt
          }
        }
      `,
      variables: {
        id: selectedEvent._id
      }
    };

    fetch('http://localhost:8000/graphql', {
      method: 'POST',
      body: JSON.stringify(requestBody),
      headers: {
        'Content-Type': 'application/json'
      }
    })
      .then(res => {
        if (res.status !== 200 && res.status !== 201) {
          throw new Error('Failed!');
        }
        return res.json();
      })
      .then(resData => {
        setState(prev => ({ ...prev, selectedEvent: null }));
      })
      .catch(err => {
        console.log(err);
      });
  };

  const modalCancelHandler = () => {
    setState(prev => ({ ...prev, selectedEvent: null }));
  };

  const handleBuyClick = (prediction) => {
    if (!prediction || !Array.isArray(prediction.selectedOdds)) {
      console.error('Invalid prediction object or selectedOdds is not an array:', prediction);
      return;
    }

    const serializablePrediction = {
      predictionId: prediction.predictionId,
      predictionName: prediction.predictionName,
      predictiontype: prediction.predictiontype,
      numGames: prediction.numGames,
      totalOdds: prediction.totalOdds,
      price: prediction.price,
      selectedOdds: prediction.selectedOdds.map((odd) => ({
        marketName: odd.marketName,
        selectionName: odd.selectionName,
        odds: odd.odds,
        gameId: odd.gameId,
        date: odd.Date,
        competitors: Array.isArray(odd.competitors)
          ? odd.competitors.map((competitor) => ({
              id: competitor.id,
              name: competitor.name,
            }))
          : [],
      })),
    };

    navigate('/prediction-details', {
      state: { prediction: serializablePrediction }
    });
  };

  const categorizeByTime = (predictions) => {
    const categories = {
      today: [],
      thisWeek: [],
      thisMonth: [],
      later: []
    };

    predictions.forEach(prediction => {
      const predictionDate = new Date(prediction.date);
      const today = new Date();
      const oneWeekLater = new Date();
      oneWeekLater.setDate(today.getDate() + 7);
      const oneMonthLater = new Date();
      oneMonthLater.setMonth(today.getMonth() + 1);

      if (predictionDate.toDateString() === today.toDateString()) {
        categories.today.push(prediction);
      } else if (predictionDate < oneWeekLater) {
        categories.thisWeek.push(prediction);
      } else if (predictionDate < oneMonthLater) {
        categories.thisMonth.push(prediction);
      } else {
        categories.later.push(prediction);
      }
    });

    return categories;
  };

  const categorizeByPrice = (predictions) => {
    const categories = {
      free: [],
      below100: [],
      between100and500: [],
      above500: []
    };

    predictions.forEach(prediction => {
      const price = prediction.price || 0;
      if (price === 0) {
        categories.free.push(prediction);
      } else if (price < 100) {
        categories.below100.push(prediction);
      } else if (price >= 100 && price <= 500) {
        categories.between100and500.push(prediction);
      } else {
        categories.above500.push(prediction);
      }
    });

    return categories;
  };

  const categorizedByTime = categorizeByTime(predictions);
  const categorizedByPrice = categorizeByPrice(predictions);

  const limitItems = (category, maxItems) => category.slice(0, maxItems);

  return (
    <React.Fragment>
      {selectedEvent && (
        <Modal
          title={selectedEvent.title}
          canCancel
          canConfirm
          onCancel={modalCancelHandler}
          onConfirm={bookEventHandler}
          confirmText="Book"
        >
          <h1>{selectedEvent.name}</h1>
          <h2>
            ${selectedEvent.price} -{' '}
            {new Date(selectedEvent.date).toLocaleDateString()}
          </h2>
          <p>{selectedEvent.description}</p>
        </Modal>
      )}

      <div style={{ display: 'flex' }}>
        <div
          style={{
            width: '250px',
            height: '100vh',
            backgroundColor: '#1F2937', // Dark background
            color: 'white',
            position: 'fixed',
            top: 0,
            left: 0,
            padding: '10px',
            boxSizing: 'border-box',
            display: 'flex',
            flexDirection: 'column',
            overflowY: 'auto',
            borderRadius: '8px', // Rounded corners
            boxShadow: '4px 0 6px rgba(0, 0, 0, 0.1)', // Soft shadow
          }}
        >
          <h2 style={{
            fontSize: '1.2rem', 
            fontWeight: 'bold', 
            marginBottom: '10px', 
            color: '#fff',
            borderBottom: '2px solid #E5E7EB', // Border for elegance
            paddingBottom: '8px',
          }}>
            Predictions by Time
          </h2>
          {Object.keys(categorizedByTime).map((timeCategory) => (
            <div key={timeCategory} style={{ marginBottom: '15px' }}>
              <h4 style={{
                color: '#F59E0B', 
                fontSize: '1rem', 
                marginBottom: '8px', 
                textTransform: 'capitalize'
              }}>
                {timeCategory.charAt(0).toUpperCase() + timeCategory.slice(1)}
              </h4>
              <ul style={{ paddingLeft: '10px', marginTop: '5px' }}>
                {limitItems(categorizedByTime[timeCategory], 3).map(prediction => (
                  <li key={prediction.predictionId} style={{ marginBottom: '8px' }}>
                    <button
                      type="button"
                      onClick={() => handleBuyClick(prediction)}
                      style={{
                        background: 'none',
                        border: 'none',
                        padding: 0,
                        color: 'white',
                        textDecoration: 'none',
                        fontSize: '0.9rem',
                        lineHeight: '1.4',
                        cursor: 'pointer'
                      }}
                    >
                      {prediction.predictionName} - {prediction.date}
                    </button>
                  </li>
                ))}
              </ul>
            </div>
          ))}

          <h2 style={{
            fontSize: '1.2rem', 
            fontWeight: 'bold', 
            marginBottom: '10px', 
            color: '#fff',
            borderBottom: '2px solid #E5E7EB', // Border for elegance
            paddingBottom: '8px',
          }}>
            Predictions by Price
          </h2>
          {Object.keys(categorizedByPrice).map((priceCategory) => (
            <div key={priceCategory} style={{ marginBottom: '15px' }}>
              <h4 style={{
                color: '#F59E0B', 
                fontSize: '1rem', 
                marginBottom: '8px', 
                textTransform: 'capitalize'
              }}>
                {priceCategory.charAt(0).toUpperCase() + priceCategory.slice(1)}
              </h4>
              <ul style={{ paddingLeft: '10px', marginTop: '5px' }}>
                {limitItems(categorizedByPrice[priceCategory], 3).map(prediction => (
                  <li key={prediction.predictionId} style={{ marginBottom: '8px' }}>
                    <button
                      type="button"
                      onClick={() => handleBuyClick(prediction)}
                      style={{
                        background: 'none',
                        border: 'none',
                        color: 'white',
                        textDecoration: 'none',
                        fontSize: '0.9rem',
                        lineHeight: '1.4',
                        cursor: 'pointer',
                        padding: 0
                      }}
                    >
                      {prediction.predictionName} - KES {prediction.price || 'Free'}
                    </button>
                  </li>
                ))}
              </ul>
            </div>
          ))}
        </div>

        <div style={{ marginLeft: '250px', padding: '20px', width: '100%' }}>
          <EventList
            products={products}
            authUserId={null}
            onViewDetail={showDetailHandler}
          />

          <div style={{ marginTop: 40 }}>
            <h2>Latest Predictions (with Pricing)</h2>
            {predictions.length === 0 ? (
              <p>No predictions available yet. <button onClick={fetchPredictions}>Refresh</button></p>
            ) : (
              <div style={{ display: 'grid', gridTemplateColumns: 'repeat(auto-fit, minmax(300px, 1fr))', gap: 20 }}>
                {predictions.slice(0, 6).map((prediction, index) => (
                  <div key={prediction.predictionId || index} style={{
                    padding: 16,
                    border: '1px solid #ddd',
                    borderRadius: 8,
                    background: '#f9f9f9',
                    boxShadow: '0 2px 4px rgba(0,0,0,0.1)'
                  }}>
                    <h3>{prediction.predictionName}</h3>
                    <p><strong>Type:</strong> {prediction.predictiontype}</p>
                    <p><strong>Games:</strong> {prediction.numGames}</p>
                    <p><strong>Total Odds:</strong> {prediction.totalOdds.toFixed(2)}</p>
                    <p style={{ color: '#007bff', fontWeight: 'bold' }}>
                      <strong>Price:</strong> KES {prediction.price || 'Free'}
                    </p>
                    <button
                      onClick={() => handleBuyClick(prediction)}
                      style={{
                        marginTop: 10,
                        padding: 8,
                        background: '#007bff',
                        color: 'white',
                        border: 'none',
                        borderRadius: 4,
                        cursor: 'pointer'
                      }}
                    >
                      Buy Now (View Details)
                    </button>
                  </div>
                ))}
              </div>
            )}
            <div style={{ textAlign: 'center', marginTop: 20 }}>
              <a href="/betlist" style={{ color: '#007bff', textDecoration: 'none' }}>View All Predictions →</a>
            </div>
          </div>

          <Prediction
            predictions={predictions}
            onBuyClick={handleBuyClick}
          />
        </div>
      </div>
    </React.Fragment>
  );
};

export default Home;
