import React from 'react';
import { useNavigate } from 'react-router-dom';
import '../components/Navigation/Styles.css';
import '../components/Navigation/css';
import { AuthContext } from '../context/auth-context';
import Modal from '../components/Modal/Modal';
import './Events.css';
import EventList from '../components/Events/EventList/EventList';

class Betlist extends React.Component {
  state = {
    creating: true,
    products: [],
    predictions: [],
    selectedEvent: null,
    copyingMap: {},
    sendingMap: {},
    copiedMap: {},
    visitedMap: {},
    reports: {},
    confirmedMap: {},
    targetSuccessfulBets: 20,
    runningPipeline: false,
    pipelineStatus: null
  };

  // Autobet state
  // autobetStats: { total, remaining, successes, failures, placedStakeTotal, balanceRemaining }

  isActive = true;

  static contextType = AuthContext;

  componentDidMount() {
    this.fetchPredictions();

    // load local confirmed predictions first
    try {
      const storedConfirmed = JSON.parse(localStorage.getItem('confirmedPredictions') || '{}');
      if (storedConfirmed && typeof storedConfirmed === 'object') {
        this.setState({ confirmedMap: storedConfirmed });
      }
    } catch (e) {
      console.warn('Failed to load confirmedPredictions from localStorage', e);
    }

    // Fetch server-side confirmed predictions (optional - skip if endpoint doesn't exist)
    // fetch('http://localhost:5000/api/confirmedPredictions')
    //   .then(r => r.ok ? r.json() : Promise.reject(new Error('Failed to fetch confirmed predictions')))
    //   .then(data => {
    //     if (data && Array.isArray(data.confirmed)) {
    //       const serverMap = {};
    //       data.confirmed.forEach(id => { if (id) serverMap[id] = true; });
    //       this.setState(prev => {
    //         const merged = Object.assign({}, prev.confirmedMap || {}, serverMap || {});
    //         try { localStorage.setItem('confirmedPredictions', JSON.stringify(merged)); } catch (e) { /* ignore */ }
    //         return { confirmedMap: merged };
    //       }, () => this.fetchPredictions());
    //     } else {
    //       this.fetchPredictions();
    //     }
    //   })
    //   .catch(err => {
    //     console.warn('Could not load server confirmed predictions, falling back to local only', err && err.message);
    //     this.fetchPredictions();
    //   });

    try {
      const storedVisited = JSON.parse(localStorage.getItem('copiedPredictions') || '{}');
      if (storedVisited && typeof storedVisited === 'object') {
        this.setState({ visitedMap: storedVisited });
      }
    } catch (e) {
      console.warn('Failed to load copiedPredictions from localStorage', e);
    }
    try {
      const storedReports = JSON.parse(localStorage.getItem('predictionCopyReports') || '{}');
      if (storedReports && typeof storedReports === 'object') {
        const validReports = {};
        Object.keys(storedReports).forEach(key => {
          if (storedReports[key] && typeof storedReports[key] === 'object' && storedReports[key].status) {
            validReports[key] = storedReports[key];
          }
        });
        this.setState({ reports: validReports });
      }
    } catch (e) {
      console.warn('Failed to load predictionCopyReports from localStorage', e);
    }
  }

  // helper to parse stake from sms string (assumes smsString ends with #<stake>)
  parseStakeFromSms = (sms) => {
    if (!sms || typeof sms !== 'string') return 1;
    const parts = sms.split('#');
    const last = parts[parts.length - 1];
    const n = parseFloat(last);
    return Number.isFinite(n) ? n : 1;
  }

  // parse SportPesa reply messages (shared earlier). Returns { success, amount, possibleWin, balance, raw }
  parseSportPesaMessage = (msg) => {
    if (!msg || typeof msg !== 'string') return { success: false };
    const s = msg.replace(/\s+/g, ' ');
    const confirmed = /confirm(?:ed)?\b/i.test(s) || /Your bet is confirmed/i.test(s) || /BetID.*confirmed/i.test(s);
    const amountMatch = s.match(/Amount\s*:\s*KSH\s*([\d,]+(?:\.\d+)?)/i) || s.match(/Amount\s*:\s*([\d,]+(?:\.\d+)?)/i);
    const possibleWinMatch = s.match(/Possible win(?: after tax)?:\s*KSH\s*([\d,]+(?:\.\d+)?)/i) || s.match(/Possible win(?: after tax)?:\s*([\d,]+(?:\.\d+)?)/i);
    const balanceMatch = s.match(/Your\s*S-?PESA\s+balance\s*:\s*KSH\s*([\d,]+(?:\.\d+)?)/i) || s.match(/balance\s*:\s*KSH\s*([\d,]+(?:\.\d+)?)/i);
    const amount = amountMatch ? parseFloat(String(amountMatch[1]).replace(/,/g, '')) : null;
    const possibleWin = possibleWinMatch ? parseFloat(String(possibleWinMatch[1]).replace(/,/g, '')) : null;
    const balance = balanceMatch ? parseFloat(String(balanceMatch[1]).replace(/,/g, '')) : null;
    return { success: !!confirmed, amount, possibleWin, balance, raw: msg };
  }

  persistVisited = (visitedMap) => {
    try {
      localStorage.setItem('copiedPredictions', JSON.stringify(visitedMap || {}));
    } catch (e) {
      console.warn('Failed to persist visitedMap', e);
    }
  };

  persistConfirmed = (confirmedMap) => {
    try {
      localStorage.setItem('confirmedPredictions', JSON.stringify(confirmedMap || {}));
    } catch (e) {
      console.warn('Failed to persist confirmedMap', e);
    }
  };

  persistReports = (reports) => {
    try {
      const validReports = {};
      Object.keys(reports || {}).forEach(key => {
        if (reports[key] && typeof reports[key] === 'object' && reports[key].status) {
          validReports[key] = reports[key];
        }
      });
      localStorage.setItem('predictionCopyReports', JSON.stringify(validReports));
    } catch (e) {
      console.warn('Failed to persist reports', e);
    }
  };

  fetchEvents() {
    const requestBody = {
      query: `
        query {
          products {
            _id
            name
            description
            date
            category
            brand
            image
            price
            creator {
              _id
              email
            }
          }
        }
      `
    };

    fetch('http://localhost:5000/graphql', {
      method: 'POST',
      body: JSON.stringify(requestBody),
      headers: {
        'Content-Type': 'application/json'
      }
    })
      .then(res => {
        if (res.status !== 200 && res.status !== 201) {
          throw new Error('Failed!');
        }
        return res.json();
      })
      .then(resData => {
        const products = resData.data.products;
        this.setState({ products: products });
      })
      .catch(err => {
        console.log(err);
      });
  }

  fetchPredictions() {
    console.log('Starting fetchPredictions...');
    fetch('http://localhost:5000/api/predictions')
      .then(res => {
        console.log('Predictions API response status:', res.status);
        if (res.status !== 200 && res.status !== 201) {
          throw new Error('Failed to fetch predictions!');
        }
        return res.json();
      })
      .then(predictionsData => {
        console.log('Predictions data received:', predictionsData);
        console.log('Type of predictionsData:', typeof predictionsData);
        
        // Support new backend format: { results: [...] }
        let predictionsArray = [];
        if (Array.isArray(predictionsData)) {
          console.log('Data is array, length:', predictionsData.length);
          predictionsArray = predictionsData;
        } else if (predictionsData && Array.isArray(predictionsData.results)) {
          console.log('Data has results array, length:', predictionsData.results.length);
          predictionsArray = predictionsData.results;
        } else if (typeof predictionsData === 'object' && predictionsData !== null) {
          console.log('Data is object, converting to array');
          const values = Object.values(predictionsData);
          console.log('Object values length:', values.length);
          // Filter out non-prediction objects (like malformed, etc.)
          predictionsArray = values.filter(v => 
            v && typeof v === 'object' && (v.predictionId || v.predictionName || v.selectedOdds)
          );
          console.log('Filtered predictions array length:', predictionsArray.length);
        } else {
          console.warn('Unexpected predictions data format:', predictionsData);
        }

        console.log('Final predictions array length:', predictionsArray.length);

        // Add _localId for React keying
        const withLocalIds = predictionsArray.map((p, i) => ({
          _localId: p.predictionId || `local-${i}`,
          ...p
        }));

        // filter out confirmed predictions so they don't repeat
        const confirmed = this.state && this.state.confirmedMap ? this.state.confirmedMap : {};
        const filtered = withLocalIds.filter(p => {
          const id = p.predictionId || p._localId;
          return !confirmed[id];
        });

        console.log('Filtered predictions (excluding confirmed):', filtered.length);

        this.setState(prev => {
          const newVisitedMap = {};
          const newReports = {};
          filtered.forEach(p => {
            const id = p.predictionId || p._localId;
            newVisitedMap[id] = prev.visitedMap[id] || false;
            newReports[id] = prev.reports[id] || {};
          });
          this.persistVisited(newVisitedMap);
          this.persistReports(newReports);
          return {
            predictions: filtered,
            visitedMap: newVisitedMap,
            reports: newReports
          };
        });
      })
      .catch(err => {
        console.error('Error fetching predictions:', err);
        console.error('Error details:', err.message, err.stack);
      });
  }

  showDetailHandler = productId => {
    this.setState(prevState => {
      const selectedEvent = prevState.products.find(e => e._id === productId);
      return { selectedEvent: selectedEvent };
    });
  };

  handleBuyClick = (prediction, index) => {
    if (!prediction) return;
    
    // Mark as visited
    const predId = prediction.predictionId || prediction._localId;
    if (predId) {
      this.setState(prev => {
        const visitedNext = { ...(prev.visitedMap || {}), [predId]: true };
        this.persistVisited(visitedNext);
        return { visitedMap: visitedNext };
      });
    }
    
    // Navigate to prediction details page
    if (this.props.navigate) {
      this.props.navigate('/prediction-details', {
        state: { 
          prediction: {
            ...prediction,
            isOpposite: false,
            originalStrategy: 'standard_betting'
          }
        }
      });
    } else {
      // Fallback for development - show alert if navigate is not available
      alert(`Navigation not available. Prediction: ${prediction.predictionName || 'Prediction'}`);
    }
  };

  bookEventHandler = () => {
    if (!this.context.token) {
      this.setState({ selectedEvent: null });
      return;
    }
    const requestBody = {
      query: `
        mutation BookEvent($id: ID!) {
          bookEvent(productId: $id) {
            _id
            createdAt
            updatedAt
          }
        }
      `,
      variables: {
        id: this.state.selectedEvent._id
      }
    };

    fetch('http://localhost:5000/graphql', {
      method: 'POST',
      body: JSON.stringify(requestBody),
      headers: {
        'Content-Type': 'application/json',
        Authorization: 'Bearer ' + this.context.token
      }
    })
      .then(res => {
        if (res.status !== 200 && res.status !== 201) {
          throw new Error('Failed!');
        }
        return res.json();
      })
      .then(resData => {
        console.log(resData);
        this.setState({ selectedEvent: null });
      })
      .catch(err => {
        console.log(err);
      });
  };

  modalCancelHandler = () => {
    this.setState({ creating: false, selectedEvent: null });
  };

  handleCopySms = async (prediction) => {
    if (!prediction) return;
    const predId = prediction.predictionId || prediction._localId;
    if (!predId) return;
    if (this.state.copyingMap && this.state.copyingMap[predId]) return;

    this.setState(prev => ({ copyingMap: { ...(prev.copyingMap || {}), [predId]: true } }));
    try {
      let smsText = prediction.smsString;
      if (!smsText) {
        const resp = await fetch('http://localhost:5000/api/createSmsBets', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({ predictions: [prediction], stake: 1 })
        });
        if (!resp.ok) throw new Error('Failed to create SMS string');
        const data = await resp.json();
        if (data && data.smsBets && data.smsBets[0] && data.smsBets[0].smsString) {
          smsText = data.smsBets[0].smsString;
        } else {
          throw new Error('No SMS string returned');
        }
      }

      if (!navigator.clipboard || !navigator.clipboard.writeText) {
        const successReport = {
          status: 'success',
          message: `SMS string (clipboard unavailable): ${smsText}`,
          timestamp: new Date().toISOString()
        };
        this.setState(prev => {
          const reportsNext = { ...(prev.reports || {}), [predId]: successReport };
          this.persistReports(reportsNext);
          const visitedNext = { ...(prev.visitedMap || {}), [predId]: true };
          this.persistVisited(visitedNext);
          return { reports: reportsNext, visitedMap: visitedNext };
        });
        alert('Clipboard not available. SMS string:\n' + smsText);
        return;
      }

      await navigator.clipboard.writeText(smsText);
      const successReport = {
        status: 'success',
        message: `Copied SMS to clipboard: ${smsText}`,
        timestamp: new Date().toISOString()
      };
      this.setState(prev => {
        const reportsNext = { ...(prev.reports || {}), [predId]: successReport };
        this.persistReports(reportsNext);
        const visitedNext = { ...(prev.visitedMap || {}), [predId]: true };
        this.persistVisited(visitedNext);
        return { reports: reportsNext, visitedMap: visitedNext };
      });
      alert('SMS string copied to clipboard! Send it to 79079.');
    } catch (err) {
      const errMsg = err && err.message ? err.message : String(err);
      const errorReport = {
        status: 'error',
        message: errMsg,
        timestamp: new Date().toISOString()
      };
      this.setState(prev => {
        const reportsNext = { ...(prev.reports || {}), [predId]: errorReport };
        this.persistReports(reportsNext);
        return { reports: reportsNext };
      });
      console.error('Failed to copy SMS for prediction', predId, err);
      alert('Failed to copy SMS: ' + errMsg);
    } finally {
      this.setState(prev => ({ copyingMap: { ...(prev.copyingMap || {}), [predId]: false } }));
    }
  };

  handlePreviewSms = async (prediction) => {
    if (!prediction) return;
    try {
      let smsText = prediction.smsString;
      if (!smsText) {
        const resp = await fetch('http://localhost:5000/api/createSmsBets', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({ predictions: [prediction], stake: 1 })
        });
        if (!resp.ok) throw new Error('Failed to create SMS string');
        const data = await resp.json();
        if (data && data.smsBets && data.smsBets[0] && data.smsBets[0].smsString) {
          smsText = data.smsBets[0].smsString;
        } else {
          throw new Error('No SMS string returned');
        }
      }
      alert('SMS Preview:\n\nTo: 79079\nMessage:\n' + smsText);
    } catch (err) {
      console.error('Failed to preview SMS', err);
      alert('Failed to preview SMS: ' + (err && err.message ? err.message : String(err)));
    }
  };

  handleSendTo79079 = async (prediction) => {
    if (!prediction) return;
    const predId = prediction.predictionId || prediction._localId;
    if (!predId) return;

    try {
      let smsText = prediction.smsString;
      if (!smsText) {
        const resp = await fetch('http://localhost:5000/api/createSmsBets', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({ predictions: [prediction], stake: 1 })
        });
        if (!resp.ok) throw new Error('Failed to create SMS string');
        const data = await resp.json();
        if (data && data.smsBets && data.smsBets[0] && data.smsBets[0].smsString) {
          smsText = data.smsBets[0].smsString;
        } else {
          throw new Error('No SMS string returned');
        }
      }

      const smsUrl = `sms:79079?body=${encodeURIComponent(smsText)}`;
      const opened = window.open(smsUrl, '_blank');
      if (!opened || opened.closed || typeof opened.closed === 'undefined') {
        const errorReport = {
          status: 'error',
          message: 'Unable to open SMS composer (popup blocked or not supported)',
          timestamp: new Date().toISOString()
        };
        this.setState(prev => {
          const reportsNext = { ...(prev.reports || {}), [predId]: errorReport };
          this.persistReports(reportsNext);
          return { reports: reportsNext };
        });
        alert('Unable to open SMS composer on this device/browser. SMS not sent.');
        return;
      }

      const successReport = {
        status: 'success',
        message: `Opened composer for 79079 — body: ${smsText}`,
        timestamp: new Date().toISOString()
      };
      this.setState(prev => {
        const reportsNext = { ...(prev.reports || {}), [predId]: successReport };
        this.persistReports(reportsNext);
        return { reports: reportsNext };
      });

    } catch (err) {
      const errMsg = err && err.message ? err.message : String(err);
      const errorReport = {
        status: 'error',
        message: errMsg,
        timestamp: new Date().toISOString()
      };
      this.setState(prev => {
        const reportsNext = { ...(prev.reports || {}), [predId]: errorReport };
        this.persistReports(reportsNext);
        return { reports: reportsNext };
      });
      console.error('Failed to send SMS to 79079 for prediction', predId, err);
      alert('Failed to prepare SMS for 79079: ' + errMsg);
    }
  };

  handleProgrammaticSend = async (prediction) => {
    if (!prediction) return;
    const predId = prediction.predictionId || prediction._localId;
    if (!predId) return;
    if (this.state.sendingMap && this.state.sendingMap[predId]) return;

    this.setState(prev => ({ sendingMap: { ...(prev.sendingMap || {}), [predId]: true } }));
    try {
      let smsText = prediction.smsString;
      if (!smsText) {
        const resp = await fetch('http://localhost:5000/api/createSmsBets', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({ predictions: [prediction], stake: 1 })
        });
        if (!resp.ok) throw new Error('Failed to create SMS string');
        const data = await resp.json();
        if (data && data.smsBets && data.smsBets[0] && data.smsBets[0].smsString) {
          smsText = data.smsBets[0].smsString;
        } else {
          throw new Error('No SMS string returned');
        }
      }

      const bridgeUrl = 'http://localhost:4567/send-sms';
      const payload = { to: '79079', message: smsText };
      const sendResp = await fetch(bridgeUrl, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(payload)
      });
      if (!sendResp.ok) throw new Error('Bridge server error: ' + sendResp.status);
      const sendData = await sendResp.json();
      console.log('Bridge server response:', sendData);

      const successReport = {
        status: 'success',
        message: `Sent via device to 79079: ${smsText}`,
        timestamp: new Date().toISOString()
      };
      this.setState(prev => {
        const reportsNext = { ...(prev.reports || {}), [predId]: successReport };
        this.persistReports(reportsNext);
        const visitedNext = { ...(prev.visitedMap || {}), [predId]: true };
        this.persistVisited(visitedNext);
        return { reports: reportsNext, visitedMap: visitedNext };
      });
      alert('SMS sent via device bridge!');
    } catch (err) {
      const errMsg = err && err.message ? err.message : String(err);
      const errorReport = {
        status: 'error',
        message: errMsg,
        timestamp: new Date().toISOString()
      };
      this.setState(prev => {
        const reportsNext = { ...(prev.reports || {}), [predId]: errorReport };
        this.persistReports(reportsNext);
        return { reports: reportsNext };
      });
      console.error('Failed to send SMS via device for prediction', predId, err);
      alert('Failed to send SMS via device: ' + errMsg);
    } finally {
      this.setState(prev => ({ sendingMap: { ...(prev.sendingMap || {}), [predId]: false } }));
    }
  };

  handleConfirmSuccess = async (prediction) => {
    if (!prediction) return;
    const predId = prediction.predictionId || prediction._localId;
    if (!predId) return;

    const userInput = prompt('Enter the SportPesa confirmation SMS (or leave blank if no SMS):');
    if (userInput === null) return;

    const parsed = this.parseSportPesaMessage(userInput);
    let finalMsg = '';
    if (parsed.success) {
      finalMsg = `Confirmed: amount=${parsed.amount || 'N/A'}, possibleWin=${parsed.possibleWin || 'N/A'}, balance=${parsed.balance || 'N/A'}`;
    } else {
      finalMsg = `Marked as confirmed (no SMS or invalid SMS)`;
    }

    const confirmReport = {
      status: 'confirmed',
      message: finalMsg,
      timestamp: new Date().toISOString(),
      smsRaw: userInput || ''
    };
    this.setState(prev => {
      const reportsNext = { ...(prev.reports || {}), [predId]: confirmReport };
      this.persistReports(reportsNext);
      const confirmedNext = { ...(prev.confirmedMap || {}), [predId]: true };
      this.persistConfirmed(confirmedNext);
      return { reports: reportsNext, confirmedMap: confirmedNext };
    });

    try {
      await fetch('http://localhost:5000/api/confirmPrediction', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ predictionId: predId })
      });
    } catch (e) {
      console.warn('Could not save confirmation to server', e);
    }

    this.fetchPredictions();
  };

  handleReportError = async (prediction) => {
    if (!prediction) return;
    const predId = prediction.predictionId || prediction._localId;
    if (!predId) return;

    const userInput = prompt('Enter error details or SMS from SportPesa:');
    if (userInput === null) return;

    const errorReport = {
      status: 'failed',
      message: userInput || 'User reported error',
      timestamp: new Date().toISOString()
    };
    this.setState(prev => {
      const reportsNext = { ...(prev.reports || {}), [predId]: errorReport };
      this.persistReports(reportsNext);
      return { reports: reportsNext };
    });

    try {
      await fetch('http://localhost:5000/api/markFailedPrediction', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ predictionId: predId, error: userInput || 'User reported error' })
      });
    } catch (e) {
      console.warn('Could not save failed prediction to server', e);
    }
  };

  renderPredictionStatus = (prediction) => {
    const id = prediction.predictionId || prediction._localId;
    const report = this.state.reports && this.state.reports[id];
    if (!report || !report.status) return null;
    const isConfirmed = report.status === 'confirmed';
    const isFailed = report.status === 'failed';
    const isSuccess = report.status === 'success';
    const isError = report.status === 'error';
    let color = '#333';
    let text = '';
    if (isConfirmed) { color = '#0a7'; text = 'Confirmed ✓'; }
    else if (isFailed) { color = '#c00'; text = 'Failed ✗'; }
    else if (isSuccess) { color = '#0a7'; text = 'Success'; }
    else if (isError) { color = '#c00'; text = 'Error'; }
    return (
      <div style={{ marginTop: 4, fontSize: 12, color }}>
        <strong>{text}</strong> — {report.message}
      </div>
    );
  };

  emergencyStopAutobet = () => {
    this.setState({ autobetRunning: false, autobetStats: null });
    alert('AutoBet stopped by user.');
  };

  // Run prediction pipeline
  runPredictionPipeline = async () => {
    this.setState({ runningPipeline: true, pipelineStatus: 'Starting pipeline...' });
    
    try {
      // Step 1: Save/Fetch sports and leagues data
      this.setState({ pipelineStatus: 'Step 1/6: Fetching sports data...' });
      await fetch('http://localhost:5000/api/saveData');
      
      // Step 2: Get game IDs
      this.setState({ pipelineStatus: 'Step 2/6: Fetching game IDs...' });
      await fetch('http://localhost:5000/api/getGameIds');
      
      // Step 3: Fetch all game data
      this.setState({ pipelineStatus: 'Step 3/6: Fetching all game data...' });
      await fetch('http://localhost:5000/api/allGameData');
      
      // Step 4: Filter and save odds
      this.setState({ pipelineStatus: 'Step 4/6: Filtering odds (< 2.7)...' });
      await fetch('http://localhost:5000/api/saveodds');
      
      // Step 5: Create today's predictions
      this.setState({ pipelineStatus: 'Step 5/6: Creating predictions...' });
      const response = await fetch('http://localhost:5000/api/lettherebetodayspredictions');
      const data = await response.json();
      
      // Step 6: Refresh the predictions list
      this.setState({ pipelineStatus: 'Step 6/6: Refreshing predictions...' });
      await this.fetchPredictions();
      
      this.setState({ 
        runningPipeline: false, 
        pipelineStatus: `✓ Pipeline completed! ${data.count || 0} predictions created.` 
      });
      
      setTimeout(() => {
        this.setState({ pipelineStatus: null });
      }, 5000);
    } catch (error) {
      console.error('Pipeline error:', error);
      this.setState({ 
        runningPipeline: false, 
        pipelineStatus: `✗ Pipeline failed: ${error.message}` 
      });
      
      setTimeout(() => {
        this.setState({ pipelineStatus: null });
      }, 10000);
    }
  };

  // Quick refresh predictions
  handleRefreshPredictions = async () => {
    this.setState({ pipelineStatus: 'Refreshing predictions...' });
    try {
      await this.fetchPredictions();
      this.setState({ pipelineStatus: '✓ Predictions refreshed!' });
      setTimeout(() => {
        this.setState({ pipelineStatus: null });
      }, 2000);
    } catch (error) {
      this.setState({ pipelineStatus: `✗ Refresh failed: ${error.message}` });
      setTimeout(() => {
        this.setState({ pipelineStatus: null });
      }, 5000);
    }
  };

  runAutoBetAll = async (options = {}) => {
    const delayMs = options.delayMs || 1200;
    const targetSuccesses = this.state.targetSuccessfulBets || 20;
    const predictions = this.state.predictions || [];
    const unconfirmedPreds = predictions.filter(p => {
      const id = p.predictionId || p._localId;
      return !this.state.confirmedMap[id];
    });

    if (unconfirmedPreds.length === 0) {
      alert('No unconfirmed predictions to auto-bet.');
      return;
    }

    this.setState({ autobetRunning: true, autobetStats: { total: unconfirmedPreds.length, remaining: unconfirmedPreds.length, successes: 0, failures: 0, target: targetSuccesses, placedStakeTotal: 0, balanceRemaining: undefined } });

    let successes = 0;
    let failures = 0;
    let placedStakeTotal = 0;
    let balanceRemaining = undefined;

    for (let i = 0; i < unconfirmedPreds.length; i++) {
      if (!this.state.autobetRunning) {
        alert('AutoBet stopped by user.');
        break;
      }
      if (successes >= targetSuccesses) {
        alert(`AutoBet reached target of ${targetSuccesses} successful bets.`);
        break;
      }
      const pred = unconfirmedPreds[i];
      const predId = pred.predictionId || pred._localId;

      try {
        let smsText = pred.smsString;
        if (!smsText) {
          console.log('Creating SMS for prediction:', pred.predictionId || pred._localId);
          
          // Ensure prediction has required structure
          if (!pred.selectedOdds || !Array.isArray(pred.selectedOdds) || pred.selectedOdds.length === 0) {
            throw new Error('Prediction missing selectedOdds array');
          }
          
          const resp = await fetch('http://localhost:5000/api/createSmsBets', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ predictions: [pred], stake: 1 })
          });
          
          if (!resp.ok) {
            const errText = await resp.text();
            console.error('SMS creation failed:', resp.status, errText);
            throw new Error(`Failed to create SMS string: ${resp.status} ${errText}`);
          }
          
          const data = await resp.json();
          console.log('SMS creation response:', data);
          
          if (data && data.smsBets && data.smsBets[0] && data.smsBets[0].smsString) {
            smsText = data.smsBets[0].smsString;
          } else if (data && data.errors && data.errors.length > 0) {
            throw new Error(`SMS creation errors: ${JSON.stringify(data.errors)}`);
          } else {
            throw new Error('No SMS string returned from server');
          }
        }

        const bridgeUrl = 'http://localhost:4567/send-sms';
        const payload = { to: '79079', message: smsText };
        const sendResp = await fetch(bridgeUrl, {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify(payload)
        });
        if (!sendResp.ok) throw new Error('Bridge server error: ' + sendResp.status);
        const sendData = await sendResp.json();
        console.log('AutoBet bridge response:', sendData);

        const successReport = {
          status: 'success',
          message: `AutoBet sent via device to 79079: ${smsText}`,
          timestamp: new Date().toISOString()
        };
        this.setState(prev => {
          const reportsNext = { ...(prev.reports || {}), [predId]: successReport };
          this.persistReports(reportsNext);
          const visitedNext = { ...(prev.visitedMap || {}), [predId]: true };
          this.persistVisited(visitedNext);
          return { reports: reportsNext, visitedMap: visitedNext };
        });

        const stake = this.parseStakeFromSms(smsText);
        placedStakeTotal += stake;

        successes++;
        const currentSuccesses = successes;
        const currentFailures = failures;
        const currentPlacedStake = placedStakeTotal;
        const currentBalanceRemaining = balanceRemaining;
        this.setState(prev => ({
          autobetStats: {
            ...prev.autobetStats,
            successes: currentSuccesses,
            failures: currentFailures,
            remaining: unconfirmedPreds.length - (i + 1),
            placedStakeTotal: currentPlacedStake,
            balanceRemaining: currentBalanceRemaining
          }
        }));
      } catch (err) {
        const errMsg = err && err.message ? err.message : String(err);
        const errorReport = {
          status: 'error',
          message: `AutoBet error: ${errMsg}`,
          timestamp: new Date().toISOString()
        };
        this.setState(prev => {
          const reportsNext = { ...(prev.reports || {}), [predId]: errorReport };
          this.persistReports(reportsNext);
          return { reports: reportsNext };
        });
        console.error('AutoBet failed for prediction', predId, err);
        failures++;
        const currentSuccesses = successes;
        const currentFailures = failures;
        const currentPlacedStake = placedStakeTotal;
        const currentBalanceRemaining = balanceRemaining;
        this.setState(prev => ({
          autobetStats: {
            ...prev.autobetStats,
            successes: currentSuccesses,
            failures: currentFailures,
            remaining: unconfirmedPreds.length - (i + 1),
            placedStakeTotal: currentPlacedStake,
            balanceRemaining: currentBalanceRemaining
          }
        }));
      }

      if (i < unconfirmedPreds.length - 1) {
        await new Promise(resolve => setTimeout(resolve, delayMs));
      }
    }

    this.setState({ autobetRunning: false });
    alert(`AutoBet completed: ${successes} successes, ${failures} failures.`);
  };

  handleExportReports = () => {
    const reports = this.state.reports || {};
    const json = JSON.stringify(reports, null, 2);
    const blob = new Blob([json], { type: 'application/json' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `prediction-reports-${Date.now()}.json`;
    a.click();
    URL.revokeObjectURL(url);
  };

  handleUploadReports = async () => {
    const reports = this.state.reports || {};
    try {
      const resp = await fetch('http://localhost:5000/api/predictionReports', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(reports)
      });
      if (!resp.ok) throw new Error('Failed to upload reports');
      alert('Reports uploaded to server successfully.');
    } catch (err) {
      console.error('Failed to upload reports', err);
      alert('Failed to upload reports: ' + (err && err.message ? err.message : String(err)));
    }
  };

  renderReportsPanel = () => {
    const reports = this.state.reports || {};
    const keys = reports
      ? Object.keys(reports).filter(k => reports[k] && typeof reports[k] === 'object' && reports[k].status)
      : [];
    if (!keys.length) return null;
    return (
      <div style={{ marginTop: 20, padding: 12, border: '1px solid #ddd', borderRadius: 6 }}>
        <h4 style={{ marginTop: 0 }}>Prediction Copy Reports (last per prediction)</h4>
        <ul style={{ paddingLeft: 18 }}>
          {keys.map(k => {
            const r = reports[k];
            return (
              <li key={k} style={{ marginBottom: 8 }}>
                <strong>{k}</strong> — {r.status === 'success' ? <span style={{ color: '#0a7' }}>Success</span> : <span style={{ color: '#c00' }}>Error</span>}
                <div style={{ fontSize: 12, color: '#444' }}>{r.message}</div>
                <div style={{ fontSize: 11, color: '#666' }}>{new Date(r.timestamp).toLocaleString()}</div>
              </li>
            );
          })}
        </ul>
        <button onClick={this.handleExportReports}>Export Reports</button>
        <button onClick={this.handleUploadReports} style={{ marginLeft: 8 }}>Upload Reports to Server</button>
      </div>
    );
  };

  render() {
    return (
      <React.Fragment>
        {this.state.selectedEvent && (
          <Modal
            title={this.state.selectedEvent.title}
            canCancel
            canConfirm
            onCancel={() => this.setState({ selectedEvent: null })}
            onConfirm={this.bookEventHandler}
            confirmText={this.context.token ? 'Book' : 'Confirm'}
          >
            <h1>{this.state.selectedEvent.name}</h1>
            <h2>
              ${this.state.selectedEvent.price} -{' '}
              {new Date(this.state.selectedEvent.date).toLocaleDateString()}
            </h2>
            <p>{this.state.selectedEvent.description}</p>
          </Modal>
        )}

        <div>
          <EventList
            products={this.state.products}
            authUserId={this.context.userId}
            onViewDetail={this.showDetailHandler}
          />
        </div>

        <div style={{ marginTop: 24 }}>
          <h2>Today's Predictions</h2>
          <div style={{ marginBottom: 12, backgroundColor: '#d1ecf1', padding: '12px', borderRadius: '6px', border: '1px solid #bee5eb' }}>
            <strong>ℹ️ Notice:</strong> These are <strong>standard predictions</strong> based on our analysis.
          </div>
          
          {/* Pipeline Control Buttons */}
          <div style={{ marginBottom: 16, padding: '16px', backgroundColor: '#f8f9fa', borderRadius: '8px', border: '1px solid #dee2e6' }}>
            <h3 style={{ marginTop: 0, marginBottom: 12, fontSize: 18 }}>Prediction Pipeline</h3>
            <div style={{ display: 'flex', gap: 12, flexWrap: 'wrap', alignItems: 'center' }}>
              <button 
                onClick={this.runPredictionPipeline} 
                disabled={this.state.runningPipeline}
                style={{ 
                  padding: '10px 20px', 
                  fontSize: 16, 
                  fontWeight: 'bold',
                  background: this.state.runningPipeline ? '#6c757d' : '#007bff', 
                  color: '#fff', 
                  border: 'none', 
                  borderRadius: 6,
                  cursor: this.state.runningPipeline ? 'not-allowed' : 'pointer'
                }}
              >
                {this.state.runningPipeline ? '⏳ Running...' : '▶️ Run Full Pipeline'}
              </button>
              
              <button 
                onClick={this.handleRefreshPredictions}
                disabled={this.state.runningPipeline}
                style={{ 
                  padding: '10px 20px', 
                  fontSize: 16, 
                  background: '#17a2b8', 
                  color: '#fff', 
                  border: 'none', 
                  borderRadius: 6,
                  cursor: this.state.runningPipeline ? 'not-allowed' : 'pointer'
                }}
              >
                🔄 Refresh List
              </button>
              
              {this.state.pipelineStatus && (
                <div style={{ 
                  padding: '8px 16px', 
                  backgroundColor: this.state.pipelineStatus.startsWith('✓') ? '#d4edda' : this.state.pipelineStatus.startsWith('✗') ? '#f8d7da' : '#fff3cd',
                  color: this.state.pipelineStatus.startsWith('✓') ? '#155724' : this.state.pipelineStatus.startsWith('✗') ? '#721c24' : '#856404',
                  borderRadius: 6,
                  fontSize: 14,
                  fontWeight: 'bold',
                  border: `1px solid ${this.state.pipelineStatus.startsWith('✓') ? '#c3e6cb' : this.state.pipelineStatus.startsWith('✗') ? '#f5c6cb' : '#ffeaa7'}`
                }}>
                  {this.state.pipelineStatus}
                </div>
              )}
            </div>
            <div style={{ marginTop: 10, fontSize: 13, color: '#6c757d' }}>
              <strong>Pipeline steps:</strong> Fetch sports data → Get game IDs → Fetch game details → Filter odds → Create predictions
            </div>
          </div>
          
          {/* AutoBet Section */}
          <div style={{ marginBottom: 12 }}>
            <button onClick={() => this.runAutoBetAll({ delayMs: 1200 })} style={{ padding: '12px 20px', fontSize: 16, background: '#28a745', color: '#fff', border: 'none', borderRadius: 6 }} disabled={this.state.autobetRunning}>
              Auto-Bet All (send unconfirmed)
            </button>
            <button onClick={this.emergencyStopAutobet} style={{ padding: '12px 16px', marginLeft: 8, backgroundColor: '#dc3545', color: 'white', border: 'none', borderRadius: 4 }}>🛑 EMERGENCY STOP</button>
            {this.state.autobetStats && (
              <div style={{ marginTop: 8, fontSize: 13 }}>
                <strong>AutoBet:</strong> successes {this.state.autobetStats.successes}/{this.state.autobetStats.target || 'N/A'}, failures {this.state.autobetStats.failures}, remaining {this.state.autobetStats.remaining}
                {this.state.autobetStats.placedStakeTotal && ` | Stake: ${this.state.autobetStats.placedStakeTotal}`}
                {this.state.autobetStats.balanceRemaining && ` | Balance: KES ${this.state.autobetStats.balanceRemaining}`}
              </div>
            )}
          </div>
          {(console.log('Render check: predictions.length =', this.state.predictions.length), this.state.predictions.length === 0) ? (
            <p>No predictions available.</p>
          ) : (
            <ol>
              {this.state.predictions.map((pred, idx) => {
                const id = pred.predictionId || pred._localId || `local-${idx}`;
                const copying = !!this.state.copyingMap[id];
                const visited = !!this.state.visitedMap[id];
                const uniqueKey = `${id}-${idx}`;
                return (
                  <li key={uniqueKey} style={{ marginBottom: 12, backgroundColor: pred.status === 'error' ? '#ffeaea' : '#f8f9fa', padding: '10px', borderRadius: '6px', border: pred.status === 'error' ? '2px solid #c00' : '2px solid #28a745' }}>
                    <div style={{ display: 'flex', alignItems: 'center', gap: 12 }}>
                      <div style={{ flex: 1 }}>
                        <strong style={{ color: pred.status === 'error' ? '#c00' : '#28a745' }}>⚽ {pred.predictionName || `Prediction ${id}`}</strong>
                        <div style={{ fontSize: 13, color: '#555' }}>
                          {pred.numGames ? `${pred.numGames} game(s)` : ''}
                          {pred.totalOdds ? ` — Odds: ${pred.totalOdds}` : ''}
                          {pred.price ? ` — Price: KES ${pred.price}` : ''}
                        </div>
                        {pred.status === 'error' && (
                          <div style={{ color: '#c00', fontWeight: 'bold', marginTop: 6 }}>
                            Error: {pred.error}
                          </div>
                        )}
                        {pred.status === 'success' && pred.smsString && (
                          <div style={{ color: '#0a7', fontWeight: 'bold', marginTop: 6 }}>
                            SMS: {pred.smsString}
                          </div>
                        )}
                      </div>
                      <div>
                        <button onClick={() => this.handleBuyClick(pred, idx)} style={{ marginRight: 8, backgroundColor: pred.status === 'error' ? '#c00' : '#28a745', color: 'white', border: 'none', padding: '6px 12px', borderRadius: '4px' }}>
                          View Details
                        </button>
                        <button
                          onClick={() => this.handleCopySms(pred)}
                          disabled={copying || pred.status === 'error'}
                          style={visited ? { color: '#551A8B', textDecoration: 'underline' } : {}}
                        >
                          {copying ? (
                            <span>
                              Copying… <span style={{ fontSize: 12 }}>(⏳)</span>
                            </span>
                          ) : (visited ? 'Copy SMS (visited)' : 'Copy SMS')}
                        </button>
                        <button onClick={() => this.handlePreviewSms(pred)} style={{ marginLeft: 8 }} disabled={pred.status === 'error'}>
                          Preview SMS
                        </button>
                        <button
                          onClick={() => this.handleSendTo79079(pred)}
                          style={{ marginLeft: 8 }}
                          disabled={pred.status === 'error'}
                        >
                          Send to 79079
                        </button>
                        <button
                          onClick={() => this.handleProgrammaticSend(pred)}
                          style={{ marginLeft: 8 }}
                          disabled={pred.status === 'error'}
                        >
                          Send via device
                        </button>
                        <button onClick={() => this.handleConfirmSuccess(pred)} style={{ marginLeft: 8 }}>
                          Confirm Success
                        </button>
                        <button onClick={() => this.handleReportError(pred)} style={{ marginLeft: 8 }}>
                          Report Error
                        </button>
                        {this.renderPredictionStatus(pred)}
                      </div>
                    </div>
                  </li>
                );
              })}
            </ol>
          )}
        </div>
        {this.renderReportsPanel()}
      </React.Fragment>
    );
  }
}

// Wrapper component to provide navigate function to class component
const BetlistWithNavigation = () => {
  const navigate = useNavigate();
  return <Betlist navigate={navigate} />;
};

export default BetlistWithNavigation;
