import React, { useState, useEffect, useContext, useCallback, useRef } from 'react';
import { useNavigate, Link } from 'react-router-dom'; // Added Link for navigation
import { AuthContext } from '../context/auth-context';
import gameOddsService from '../services/gameOddsService';
// import Modal from '../components/Modal/Modal'; // Assuming this exists from Betlist
import './Events.css'; // Reuse styles if applicable

const AdminPanel = () => {
  const navigate = useNavigate();
  const countdownIntervalRef = useRef(null);
  const [state, setState] = useState({
    loading: {},
    messages: [], // Array of { type: 'success'|'error', text: string }
    logLines: [], // lines from server terminal/log output
    tailing: false,
    predictionsCount: 0,
    oddsCount: 0,
    repeatedOdds: null,
    gameOdds: null,
    gameOddsPage: 1,
    gameOddsFilters: {
      timeCategory: 'all', // 'all', 'past', 'upcoming2h', 'upcoming24h', 'future', 'invalid'
      searchTerm: '',
      sortBy: 'timestamp',
      sortOrder: 'desc'
    },
    gameOddsViewMode: 'table', // 'table' or 'competitors'
    gameOddsDashboard: null,
    selectedGameDetails: null,
    lastRefresh: null,
    nextRefresh: null,
    refreshCountdown: null,
    // Value Bets Dashboard
    valueBetsDashboard: null,
    valueBetsLastRefresh: null,
    valueBetsPredictions: [],
    // admin toggles
    valueBetsCleanDryRun: (() => {
      try { const v = localStorage.getItem('valueBetsCleanDryRun'); return v === null ? true : v === 'true'; } catch (e) { return true; }
    })(), // default to dry-run for safety
    // When true, the cleaner will attempt to alter the created prediction (instead of just deleting duplicates)
    valueBetsCleanAlterCreated: (() => {
      try { const v = localStorage.getItem('valueBetsCleanAlterCreated'); return v === 'true'; } catch (e) { return false; }
    })(),
    // Modal state for preview/confirmation
    valueBetsCleanPreview: null, // holds dry-run JSON result
    valueBetsCleanPreviewVisible: false,
    valueBetsCleanConfirmVisible: false,
  });

  const { loading, messages, predictionsCount, oddsCount, repeatedOdds, gameOdds, gameOddsFilters, gameOddsViewMode, gameOddsDashboard, selectedGameDetails, valueBetsDashboard, valueBetsPredictions } = state;
  const { token: authToken } = useContext(AuthContext);
  const token = authToken || localStorage.getItem('token'); // fallback if needed

  const addMessage = (type, text) => {
    setState(prev => ({ ...prev, messages: [...prev.messages, { type, text }] }));
    setTimeout(() => setState(prev => ({ ...prev, messages: prev.messages.slice(1) })), 5000);
  };

  // Fetch logs once. Mock implementation since we're not using backend.
  const fetchLogs = useCallback(async () => {
    try {
      // Mock log data for demo purposes
      const mockLogs = [
        'Game odds service initialized',
        'Loaded predictions data successfully',
        'Dashboard data ready',
        'Admin panel loaded'
      ];
      setState(prev => ({ ...prev, logLines: mockLogs }));
      return true;
    } catch (err) {
      addMessage('error', `Failed to fetch logs: ${err.message}`);
      return false;
    }
  }, []);

  // Start/stop tailing (polling) logs when tailing is true
  useEffect(() => {
    let timer = null;
    if (state.tailing) {
      // immediately fetch then poll
      fetchLogs();
      timer = setInterval(() => {
        fetchLogs();
      }, 2000);
    }
    return () => { if (timer) clearInterval(timer); };
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [state.tailing]); // Only depend on tailing state, not fetchLogs function

  // Simple stats fetch function (stable via useCallback)
  const fetchStats = useCallback(async () => {
    try {
      // Get stats from our service instead of backend
      const data = await gameOddsService.getDashboardData();
      setState(prev => ({ 
        ...prev, 
        predictionsCount: data.totalGamesInDb || 0,
        oddsCount: data.totalGamesInDb || 0 
      }));
    } catch (err) {
      addMessage('error', 'Failed to fetch stats: ' + err.message);
    }
  }, []);

  // Simple auth check on mount (DISABLED FOR TESTING - re-enable when adding auth)
  useEffect(() => {
    // if (!token) {
    //   navigate('/login'); // Redirect if no token
    //   return;
    // }
    fetchStats();
  }, [navigate, token, fetchStats]);

  // Handle URL parameters for search functionality from navbar
  useEffect(() => {
    const urlParams = new URLSearchParams(window.location.search);
    const searchTerm = urlParams.get('search');
    const section = urlParams.get('section');
    const searchType = urlParams.get('type'); // 'league' or default to 'team'
    
    if (searchTerm && section === 'gameodds') {
      // Update game odds search filter
      setState(prev => ({
        ...prev,
        gameOddsFilters: {
          ...prev.gameOddsFilters,
          searchTerm: searchTerm
        },
        currentTab: 'gameodds' // Switch to game odds tab
      }));
      
      // Trigger search after a short delay to ensure component is ready
      setTimeout(() => {
        fetchGameOdds(1, { ...gameOddsFilters, searchTerm });
      }, 100);
      
      // Clean up URL parameters after handling
      const cleanUrl = window.location.pathname;
      window.history.replaceState({}, document.title, cleanUrl);
      
      const searchTypeLabel = searchType === 'league' ? 'leagues' : 'teams';
      addMessage('info', `Searching for ${searchTypeLabel}: "${searchTerm}"`);
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, []); // Only run once on mount


  const executeEndpoint = async (endpoint, method = 'GET', body = null, label) => {
    const key = label.replace(/\s+/g, '_').toLowerCase();
    if (loading[key]) return;
    setState(prev => ({ ...prev, loading: { ...prev.loading, [key]: true } }));

    try {
      // Mock implementation - no backend calls
      addMessage('success', `Mock: ${label} completed successfully`);
      if (endpoint === '/api/repeated-odds') {
        // Mock repeated odds data
        setState(prev => ({ ...prev, repeatedOdds: {} }));
      }
      fetchStats(); // Refresh stats after mutations
    } catch (err) {
      addMessage('error', `${label} failed: ${err.message}`);
    } finally {
      setState(prev => ({ ...prev, loading: { ...prev.loading, [key]: false } }));
    }
  };

  const handleDeleteRepeatedBet = (predictionId) => {
    executeEndpoint('/api/delete_repeated_bets', 'POST', { predictionId }, `Delete repeated bets from ${predictionId}`);
  };

  const handleLoadUsers = async () => {
    try {
      // Mock user data
      const mockUsers = [
        { uid: 'user1', email: 'admin@example.com', customClaims: { admin: true } },
        { uid: 'user2', email: 'user@example.com', customClaims: {} }
      ];
      addMessage('success', `Loaded ${mockUsers.length} users (mock data)`);
      setState(prev => ({ ...prev, users: mockUsers }));
    } catch (err) {
      addMessage('error', err.message);
    }
  };

  const handlePromote = async (uid) => {
    try {
      addMessage('success', `Mock: Promoted ${uid}`);
      // refresh users list
      handleLoadUsers();
    } catch (err) {
      addMessage('error', 'Promote failed: ' + err.message);
    }
  };

  // Value Bets Management - Automated 5-minute workflow
  const handleValueBetsWorkflow = async () => {
    const key = 'value_bets_workflow';
    if (loading[key]) return;
    setState(prev => ({ ...prev, loading: { ...prev.loading, [key]: true } }));

    try {
      addMessage('info', '🎯 Starting Value Bets Workflow...');
      
      // Step 1: Check for repeated odds
      addMessage('info', 'Step 1/3: Checking for repeated odds...');
      const repeatedResponse = await fetch('http://localhost:5000/api/repeated-odds-valuebets', {
        method: 'GET',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': token ? `Bearer ${token}` : undefined
        }
      });

      if (!repeatedResponse.ok) {
        throw new Error(`Failed to check repeated odds: ${repeatedResponse.status}`);
      }

      const repeatedData = await repeatedResponse.json();
      const repeatedCount = repeatedData ? Object.keys(repeatedData).length : 0;
      
      if (repeatedCount > 0) {
        addMessage('warning', `Found ${repeatedCount} repeated odds. Cleaning...`);
        
  // Clean repeated odds (respect dry-run and alter-created toggles)
  const dryRunParam = state.valueBetsCleanDryRun ? '?dryRun=true' : '';
  const alterParam = state.valueBetsCleanAlterCreated ? (dryRunParam ? '&alterCreated=true' : '?alterCreated=true') : '';
  const cleanResponse = await fetch(`http://localhost:5000/api/clean-repeated-odds-valuebets${dryRunParam}${alterParam}`, {
          method: 'GET',
          headers: {
            'Content-Type': 'application/json',
            'Authorization': token ? `Bearer ${token}` : undefined
          }
        });

        if (!cleanResponse.ok) {
          throw new Error(`Failed to clean repeated odds: ${cleanResponse.status}`);
        }

        const cleanData = await cleanResponse.json();
        addMessage('success', `✅ ${cleanData.message || 'Repeated odds cleaned'}`);
      } else {
        addMessage('success', '✅ No repeated odds found');
      }

      // Step 2: Remove outdated predictions (wait 5 minutes as requested)
      addMessage('info', 'Step 2/3: Waiting 5 minutes before removing outdated predictions...');
      
      // Show countdown
      let countdown = 300; // 5 minutes in seconds
      const countdownInterval = setInterval(() => {
        countdown--;
        if (countdown <= 0) {
          clearInterval(countdownInterval);
        } else {
          const minutes = Math.floor(countdown / 60);
          const seconds = countdown % 60;
          addMessage('info', `⏱️ Removing outdated predictions in ${minutes}:${seconds.toString().padStart(2, '0')}...`);
        }
      }, 60000); // Update every minute

      // Wait 5 minutes
      await new Promise(resolve => setTimeout(resolve, 300000)); // 300000ms = 5 minutes
      clearInterval(countdownInterval);

      addMessage('info', 'Removing outdated value bets predictions...');
      const outdatedResponse = await fetch('http://localhost:5000/api/moveStartedValueBets', {
        method: 'GET',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': token ? `Bearer ${token}` : undefined
        }
      });

      if (!outdatedResponse.ok) {
        throw new Error(`Failed to remove outdated predictions: ${outdatedResponse.status}`);
      }

      const outdatedData = await outdatedResponse.json();
      addMessage('success', `✅ ${outdatedData.message || 'Outdated predictions removed'}`);

      // Step 3: Create bets from remaining value bets predictions
      addMessage('info', 'Step 3/3: Fetching value bets predictions to create bets...');
      const predictionsResponse = await fetch('http://localhost:5000/api/getValueBetsPredictions', {
        method: 'GET',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': token ? `Bearer ${token}` : undefined
        }
      });

      if (!predictionsResponse.ok) {
        throw new Error(`Failed to fetch value bets predictions: ${predictionsResponse.status}`);
      }

      const predictionsData = await predictionsResponse.json();
      const predictions = predictionsData.predictions || [];
      
      if (predictions.length === 0) {
        addMessage('warning', '⚠️ No value bets predictions available to create bets');
        addMessage('success', '🎉 Value Bets Workflow completed!');
        return;
      }

      addMessage('info', `Creating SMS bets for ${predictions.length} predictions...`);
      const smsBetsResponse = await fetch('http://localhost:5000/api/createSmsBets', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': token ? `Bearer ${token}` : undefined
        },
        body: JSON.stringify({ predictions, stake: 1 })
      });

      if (!smsBetsResponse.ok) {
        throw new Error(`Failed to create SMS bets: ${smsBetsResponse.status}`);
      }

      const smsBetsData = await smsBetsResponse.json();
      const successCount = (smsBetsData.smsBets || []).length;
      const errorCount = (smsBetsData.errors || []).length;

      addMessage('success', `✅ Created ${successCount} SMS bets successfully`);
      if (errorCount > 0) {
        addMessage('warning', `⚠️ ${errorCount} bets had errors`);
      }

      addMessage('success', '🎉 Value Bets Workflow completed successfully!');
      
    } catch (err) {
      console.error('Value Bets Workflow error:', err);
      addMessage('error', `❌ Workflow failed: ${err.message}`);
    } finally {
      setState(prev => ({ ...prev, loading: { ...prev.loading, [key]: false } }));
    }
  };

  // Value Bets Dashboard Functions
  const fetchValueBetsDashboard = useCallback(async () => {
    const key = 'fetch_value_bets_dashboard';
    if (loading[key]) return;
    setState(prev => ({ ...prev, loading: { ...prev.loading, [key]: true } }));

    try {
      console.log('Fetching value bets dashboard...');
      
      // Fetch predictions
      const response = await fetch('http://localhost:5000/api/getValueBetsPredictions', {
        method: 'GET',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': token ? `Bearer ${token}` : undefined
        }
      });

      if (!response.ok) {
        throw new Error(`HTTP ${response.status}: ${response.statusText}`);
      }

      const data = await response.json();
      const predictions = data.predictions || [];
      
      // Calculate statistics
      const totalPredictions = predictions.length;
      let totalGames = 0;
      let totalStake = 0;
      let totalPotentialWin = 0;
      const predictionsByOddsRange = { low: 0, medium: 0, high: 0, veryHigh: 0 };
      const predictionsByGameCount = { '2': 0, '3': 0, '4': 0, '5+': 0 };
      
      predictions.forEach(pred => {
        const games = pred.games || pred.selectedOdds || [];
        const gameCount = games.length;
        const totalOdds = parseFloat(pred.totalOdds) || 0;
        const price = parseFloat(pred.price) || 10;
        const potentialWin = parseFloat(pred.potentialWin) || (price * totalOdds);
        
        totalGames += gameCount;
        totalStake += price;
        totalPotentialWin += potentialWin;
        
        // Categorize by odds range
        if (totalOdds < 10) predictionsByOddsRange.low++;
        else if (totalOdds < 50) predictionsByOddsRange.medium++;
        else if (totalOdds < 100) predictionsByOddsRange.high++;
        else predictionsByOddsRange.veryHigh++;
        
        // Categorize by game count
        if (gameCount === 2) predictionsByGameCount['2']++;
        else if (gameCount === 3) predictionsByGameCount['3']++;
        else if (gameCount === 4) predictionsByGameCount['4']++;
        else predictionsByGameCount['5+']++;
      });

      const dashboard = {
        totalPredictions,
        totalGames,
        totalStake: totalStake.toFixed(2),
        totalPotentialWin: totalPotentialWin.toFixed(2),
        avgOdds: totalPredictions > 0 ? (predictions.reduce((sum, p) => sum + (parseFloat(p.totalOdds) || 0), 0) / totalPredictions).toFixed(2) : 0,
        avgGamesPerPrediction: totalPredictions > 0 ? (totalGames / totalPredictions).toFixed(1) : 0,
        predictionsByOddsRange,
        predictionsByGameCount
      };
      
      setState(prev => ({ 
        ...prev, 
        valueBetsDashboard: dashboard,
        valueBetsPredictions: predictions,
        valueBetsLastRefresh: new Date().toLocaleTimeString()
      }));
      
      addMessage('success', `Value Bets Dashboard loaded: ${totalPredictions} predictions`);
    } catch (err) {
      console.error('Error fetching value bets dashboard:', err);
      addMessage('error', `Failed to fetch value bets dashboard: ${err.message}`);
      setState(prev => ({ ...prev, valueBetsDashboard: null }));
    } finally {
      setState(prev => ({ ...prev, loading: { ...prev.loading, [key]: false } }));
    }
  }, [loading, token]);

  // Game Odds Management Functions
  const fetchGameOdds = useCallback(async (page = 1, filters = gameOddsFilters) => {
    const key = 'fetch_game_odds';
    if (loading[key]) return;
    setState(prev => ({ ...prev, loading: { ...prev.loading, [key]: true } }));

    try {
      console.log('Fetching game odds with params:', { page, filters });
      const data = await gameOddsService.getFilteredGames({
        page: page,
        limit: 20,
        timeCategory: filters.timeCategory,
        search: filters.searchTerm,
        sortBy: filters.sortBy,
        sortOrder: filters.sortOrder
      });
      
      console.log('Received game odds data:', data);
      
      // Validate data structure
      if (!data || typeof data !== 'object') {
        throw new Error('Invalid data received from service');
      }
      
      if (!data.games || !Array.isArray(data.games)) {
        throw new Error('Games data is not an array');
      }
      
      setState(prev => ({ 
        ...prev, 
        gameOdds: data,
        gameOddsPage: page,
        gameOddsFilters: filters
      }));
      addMessage('success', `Loaded ${data.totalGames || 0} games (page ${page})`);
    } catch (err) {
      console.error('Error fetching game odds:', err);
      addMessage('error', `Failed to fetch game odds: ${err.message}`);
      // Reset gameOdds to a safe state
      setState(prev => ({ ...prev, gameOdds: null }));
    } finally {
      setState(prev => ({ ...prev, loading: { ...prev.loading, [key]: false } }));
    }
  }, [loading, gameOddsFilters]);

  const fetchGameOddsDashboard = useCallback(async () => {
    const key = 'fetch_dashboard';
    if (loading[key]) return;
    setState(prev => ({ ...prev, loading: { ...prev.loading, [key]: true } }));

    try {
      console.log('Fetching dashboard data...');
      const data = await gameOddsService.getDashboardData();
      console.log('Received dashboard data:', data);
      
      if (!data || typeof data !== 'object') {
        throw new Error('Invalid dashboard data received');
      }
      
      setState(prev => ({ ...prev, gameOddsDashboard: data }));
      addMessage('success', 'Dashboard data loaded');
    } catch (err) {
      console.error('Error fetching dashboard:', err);
      addMessage('error', `Failed to fetch dashboard: ${err.message}`);
      setState(prev => ({ ...prev, gameOddsDashboard: null }));
    } finally {
      setState(prev => ({ ...prev, loading: { ...prev.loading, [key]: false } }));
    }
  }, [loading]);

  const fetchGameDetails = async (gameId) => {
    const key = 'fetch_game_details';
    if (loading[key]) return;
    setState(prev => ({ ...prev, loading: { ...prev.loading, [key]: true } }));

    try {
      const data = await gameOddsService.getGameDetails(gameId);
      
      setState(prev => ({ ...prev, selectedGameDetails: data }));
      addMessage('success', `Game details loaded for ${gameId}`);
    } catch (err) {
      addMessage('error', `Failed to fetch game details: ${err.message}`);
    } finally {
      setState(prev => ({ ...prev, loading: { ...prev.loading, [key]: false } }));
    }
  };

  const cleanupExpiredGames = async () => {
    const key = 'cleanup_expired';
    if (loading[key]) return;
    setState(prev => ({ ...prev, loading: { ...prev.loading, [key]: true } }));

    try {
      const data = await gameOddsService.cleanup();
      
      addMessage('success', `Cleanup completed: ${data.message}`);
      // Refresh data after cleanup
      fetchGameOdds(1, gameOddsFilters);
      fetchGameOddsDashboard();
    } catch (err) {
      addMessage('error', `Cleanup failed: ${err.message}`);
    } finally {
      setState(prev => ({ ...prev, loading: { ...prev.loading, [key]: false } }));
    }
  };

  // Save odds to backend endpoint
  const saveOdds = async () => {
    const key = 'save_odds';
    if (loading[key]) return;
    setState(prev => ({ ...prev, loading: { ...prev.loading, [key]: true } }));

    try {
      addMessage('info', 'Starting odds save process...');
      
      // Call the existing backend endpoint
      const response = await fetch('http://localhost:5000/api/saveodds', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': token ? `Bearer ${token}` : undefined
        },
        body: JSON.stringify({
          overwriteDuplicates: true // Enable overwriting duplicates as requested
        })
      });

      if (!response.ok) {
        throw new Error(`HTTP ${response.status}: ${response.statusText}`);
      }

      const data = await response.json();
      
      // Log full response for debugging
      console.log('Save odds response:', data);
      
      // Show detailed results
      addMessage('success', `Save odds completed successfully!`);
      if (data.message) {
        addMessage('info', `Result: ${data.message}`);
      }
      if (data.saved !== undefined) {
        addMessage('info', `Records saved: ${data.saved}`);
      }
      if (data.overwritten !== undefined) {
        addMessage('info', `Duplicates overwritten: ${data.overwritten}`);
      }
      if (data.total !== undefined) {
        addMessage('info', `Total processed: ${data.total}`);
      }
      if (data.skipped !== undefined) {
        addMessage('info', `Records skipped: ${data.skipped}`);
      }
      if (data.errors && data.errors.length > 0) {
        addMessage('warning', `Errors encountered: ${data.errors.length}`);
        data.errors.slice(0, 3).forEach(error => {
          addMessage('error', `Error: ${error}`);
        });
        if (data.errors.length > 3) {
          addMessage('info', `... and ${data.errors.length - 3} more errors`);
        }
      }
      
      // Show any additional properties in the response
      Object.keys(data).forEach(key => {
        if (!['message', 'saved', 'overwritten', 'total', 'skipped', 'errors'].includes(key)) {
          addMessage('info', `${key}: ${JSON.stringify(data[key])}`);
        }
      });
      
      // Refresh data after saving
      fetchGameOdds(1, gameOddsFilters);
      fetchGameOddsDashboard();
      
    } catch (err) {
      // Enhanced error handling
      if (err.message.includes('fetch')) {
        addMessage('error', `Cannot connect to backend server at localhost:5000. Please ensure the server is running.`);
      } else if (err.message.includes('404')) {
        addMessage('error', `API endpoint /api/saveodds not found. Please check if the endpoint exists.`);
      } else if (err.message.includes('500')) {
        addMessage('error', `Server error occurred while saving odds. Check server logs.`);
      } else {
        addMessage('error', `Save odds failed: ${err.message}`);
      }
      console.error('Save odds error details:', err);
    } finally {
      setState(prev => ({ ...prev, loading: { ...prev.loading, [key]: false } }));
    }
  };

  // Test database connection
  const testDatabaseConnection = async () => {
    const key = 'test_db';
    if (loading[key]) return;
    setState(prev => ({ ...prev, loading: { ...prev.loading, [key]: true } }));

    const endpointsToTest = [
      'http://localhost:5000/api/getodds',
      'http://localhost:5000/api/odds', 
      'http://localhost:5000/api/predictions',
      'http://localhost:5000/api/games'
    ];

    try {
      addMessage('info', 'Testing multiple database endpoints...');
      let successfulEndpoint = null;

      for (const endpoint of endpointsToTest) {
        try {
          addMessage('info', `Testing: ${endpoint}`);
          
          const response = await fetch(endpoint, {
            method: 'GET',
            headers: {
              'Content-Type': 'application/json',
              'Authorization': token ? `Bearer ${token}` : undefined
            }
          });

          if (response.ok) {
            const data = await response.json();
            addMessage('success', `✅ ${endpoint} - SUCCESS!`);
            addMessage('info', `Data type: ${typeof data}, Records: ${data && typeof data === 'object' ? Object.keys(data).length : 'Unknown'}`);
            successfulEndpoint = endpoint;
            break;
          } else {
            addMessage('warning', `❌ ${endpoint} - HTTP ${response.status}: ${response.statusText}`);
          }
        } catch (err) {
          addMessage('warning', `❌ ${endpoint} - ${err.message}`);
        }
      }

      if (successfulEndpoint) {
        addMessage('success', `Database connection successful with: ${successfulEndpoint}`);
        
        // Force refresh the game odds with new data
        await gameOddsService.forceRefreshFromDatabase();
        fetchGameOdds(1, gameOddsFilters);
        fetchGameOddsDashboard();
      } else {
        addMessage('error', 'All endpoints failed. Please check:');
        addMessage('error', '1. Is your backend server running on localhost:5000?');
        addMessage('error', '2. Do any of these endpoints exist on your backend?');
        addMessage('error', '3. Check CORS settings if using different ports');
      }
      
    } catch (err) {
      addMessage('error', `Database test failed: ${err.message}`);
      console.error('Database test error:', err);
    } finally {
      setState(prev => ({ ...prev, loading: { ...prev.loading, [key]: false } }));
    }
  };

  const handleGameOddsFilterChange = (newFilters) => {
    const updatedFilters = { ...gameOddsFilters, ...newFilters };
    setState(prev => ({ ...prev, gameOddsFilters: updatedFilters }));
    fetchGameOdds(1, updatedFilters);
  };

  // Load dashboard data on mount
  useEffect(() => {
    fetchGameOddsDashboard();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, []); // Run only once on mount

  // Auto-refresh functionality - refresh data every 5 minutes
  useEffect(() => {
    const refreshData = async () => {
      console.log('Refreshing admin panel data...');
      const now = new Date();
      const nextRefreshTime = new Date(now.getTime() + 5 * 60 * 1000);
      
      setState(prev => ({ 
        ...prev, 
        lastRefresh: now.toLocaleTimeString(),
        nextRefresh: nextRefreshTime.toLocaleTimeString(),
        refreshCountdown: 300 // 5 minutes in seconds
      }));
      
      try {
        await fetchGameOddsDashboard();
        // Use current filters from state via callback to avoid closure issues
        setState(prevState => {
          fetchGameOdds(1, prevState.gameOddsFilters);
          return prevState;
        });
        addMessage('success', `Data refreshed at ${now.toLocaleTimeString()}`);
      } catch (error) {
        addMessage('error', 'Auto-refresh failed');
      }
    };
    
    // Initial load when page loads
    refreshData();

    // Set up auto-refresh every 5 minutes (300,000 milliseconds)
    const refreshInterval = setInterval(refreshData, 5 * 60 * 1000);

    // Cleanup interval on component unmount
    return () => {
      clearInterval(refreshInterval);
    };
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, []); // Empty dependency array to run only once

  // Auto-refresh Value Bets Dashboard every 5 minutes
  useEffect(() => {
    const refreshValueBets = async () => {
      try {
        console.log(`[${new Date().toISOString()}] Auto-refreshing Value Bets Dashboard...`);
        await fetchValueBetsDashboard();
        addMessage('success', 'Value Bets Dashboard auto-refreshed');
      } catch (error) {
        console.error('Value Bets auto-refresh failed:', error);
        addMessage('error', 'Value Bets auto-refresh failed');
      }
    };

    // Initial load
    refreshValueBets();

    // Set up auto-refresh every 5 minutes (300,000 milliseconds)
    const valueBetsRefreshInterval = setInterval(refreshValueBets, 5 * 60 * 1000);

    // Cleanup interval on component unmount
    return () => {
      clearInterval(valueBetsRefreshInterval);
    };
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, []); // Empty dependency array to run only once

  // Auto-run checkOutdatedPredictions every 3 minutes
  useEffect(() => {
    const checkOutdatedPredictions = async () => {
      try {
        console.log(`[${new Date().toISOString()}] Auto-running checkOutdatedPredictions...`);
        
        const response = await fetch('http://localhost:5000/api/checkOutdatedPredictions', {
          method: 'GET',
          headers: {
            'Content-Type': 'application/json',
            'Authorization': token ? `Bearer ${token}` : undefined
          }
        });

        if (!response.ok) {
          throw new Error(`HTTP ${response.status}: ${response.statusText}`);
        }

        const data = await response.json();
        console.log('CheckOutdatedPredictions result:', data);
        
        if (data.status === 'ok') {
          addMessage('success', `Auto-check outdated predictions completed successfully`);
        } else if (data.status === 'partial_failure') {
          addMessage('warning', `Auto-check outdated predictions: ${data.message}`);
        } else {
          addMessage('error', `Auto-check outdated predictions: ${data.message}`);
        }
        
      } catch (error) {
        console.error('Auto checkOutdatedPredictions failed:', error);
        addMessage('error', `Auto-check outdated predictions failed: ${error.message}`);
      }
    };

    // Run immediately on mount
    checkOutdatedPredictions();

    // Then run every 3 minutes (180,000 milliseconds)
    const outdatedCheckInterval = setInterval(checkOutdatedPredictions, 3 * 60 * 1000);

    // Cleanup interval on component unmount
    return () => {
      clearInterval(outdatedCheckInterval);
    };
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, []); // Empty dependency array to run only once

  // Countdown timer for next refresh - optimized to prevent memory leaks
  useEffect(() => {
    // Clear any existing countdown
    if (countdownIntervalRef.current) {
      clearInterval(countdownIntervalRef.current);
      countdownIntervalRef.current = null;
    }

    if (state.refreshCountdown === null) return;

    countdownIntervalRef.current = setInterval(() => {
      setState(prev => {
        if (prev.refreshCountdown <= 1) {
          return { ...prev, refreshCountdown: null };
        }
        return { ...prev, refreshCountdown: prev.refreshCountdown - 1 };
      });
    }, 1000); // Update every second

    return () => {
      if (countdownIntervalRef.current) {
        clearInterval(countdownIntervalRef.current);
        countdownIntervalRef.current = null;
      }
    };
    // eslint-disable-next-line react-hooks/exhaustive-deps  
  }, [state.refreshCountdown !== null]); // Only re-run when countdown starts/stops

  // Helper function to format countdown
  const formatCountdown = (seconds) => {
    if (!seconds) return '';
    const minutes = Math.floor(seconds / 60);
    const remainingSeconds = seconds % 60;
    return `${minutes}:${remainingSeconds.toString().padStart(2, '0')}`;
  };

  const renderMessages = () => (
    <div style={{ position: 'fixed', top: 10, right: 10, zIndex: 1000 }}>
      {messages.map((msg, i) => (
        <div key={i} style={{ padding: 10, marginBottom: 5, background: msg.type === 'success' ? '#d4edda' : '#f8d7da', borderRadius: 4, color: msg.type === 'success' ? '#155724' : '#721c24' }}>
          {msg.text}
        </div>
      ))}
    </div>
  );

  const renderRepeatedOddsSection = () => (
    <div style={{ marginTop: 20, padding: 12, border: '1px solid #ddd', borderRadius: 6 }}>
      <div style={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', marginBottom: 10 }}>
        <h3>Repeated Odds</h3>
        <Link to="/repeated-odds" style={{ textDecoration: 'none', color: '#007bff' }}>View Full Repeated Odds Page →</Link>
      </div>
      {repeatedOdds ? (
        <div>
          <p><em>Preview (full details on dedicated page):</em></p>
          {Object.entries(repeatedOdds).map(([gameId, odds]) => (
            <div key={gameId} style={{ marginBottom: 10, padding: 8, background: '#f8f9fa', borderRadius: 4 }}>
              <strong>Game ID: {gameId}</strong>
              <ul style={{ margin: 5, paddingLeft: 20 }}>
                {odds.map((item, i) => (
                  <li key={i} style={{ marginBottom: 4 }}>
                    {item.odd?.selectionName} ({item.odd?.odds}) - Repeated in: {item.repeatedInPredictions.join(', ')}
                    {item.repeatedInPredictions.map(pid => (
                      <button key={pid} onClick={() => handleDeleteRepeatedBet(pid)} style={{ marginLeft: 8, fontSize: 12, padding: 2 }}>Clean {pid}</button>
                    ))}
                  </li>
                ))}
              </ul>
            </div>
          ))}
        </div>
      ) : (
        <button onClick={() => executeEndpoint('/api/repeated-odds', 'GET', null, 'Find repeated odds')}>Find Repeated Odds</button>
      )}
    </div>
  );

  const renderValueBetsDashboard = () => (
    <div style={{ marginTop: 20, padding: 12, border: '1px solid #ddd', borderRadius: 6, background: '#f0f8ff' }}>
      <div style={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', marginBottom: 15 }}>
        <h3>🎯 Value Bets Dashboard</h3>
        <div style={{ 
          padding: '8px 12px', 
          backgroundColor: '#d4edda', 
          border: '1px solid #c3e6cb', 
          borderRadius: 4, 
          fontSize: 12,
          color: '#155724'
        }}>
          📊 Data Source: Firebase valuebetspredictions | 
          🔄 Auto-refresh: Every 5 minutes | 
          🎲 SMS Stake: 1 KSH per bet
        </div>
        <div style={{ display: 'flex', gap: 10, alignItems: 'center' }}>
          {state.valueBetsLastRefresh && (
            <div style={{ fontSize: 12, color: '#666', textAlign: 'right' }}>
              <div>Last: {state.valueBetsLastRefresh}</div>
            </div>
          )}
          <button 
            onClick={fetchValueBetsDashboard} 
            disabled={loading.fetch_value_bets_dashboard}
            style={{
              padding: '8px 16px',
              borderRadius: '4px',
              border: 'none',
              backgroundColor: '#007bff',
              color: 'white',
              cursor: loading.fetch_value_bets_dashboard ? 'not-allowed' : 'pointer',
              opacity: loading.fetch_value_bets_dashboard ? 0.6 : 1
            }}
          >
            {loading.fetch_value_bets_dashboard ? 'Loading...' : 'Manual Refresh'}
          </button>
          <button 
            onClick={handleValueBetsWorkflow} 
            disabled={loading.value_bets_workflow}
            style={{
              padding: '8px 16px',
              borderRadius: '4px',
              border: 'none',
              backgroundColor: loading.value_bets_workflow ? '#6c757d' : '#28a745',
              color: 'white',
              cursor: loading.value_bets_workflow ? 'not-allowed' : 'pointer',
              fontWeight: 'bold'
            }}
            title="Automated workflow: Check repeated odds → Clean duplicates → Wait 5 minutes → Remove outdated predictions → Create SMS bets"
          >
            {loading.value_bets_workflow ? '⏳ Running...' : '🎯 Run Workflow (5min)'}
          </button>
        </div>
      </div>
      
      {valueBetsDashboard ? (
        <div>
          {/* Statistics Cards */}
          <div style={{ display: 'grid', gridTemplateColumns: 'repeat(auto-fit, minmax(180px, 1fr))', gap: 15, marginBottom: 20 }}>
            <div style={{ padding: 10, background: '#fff', borderRadius: 4, border: '1px solid #ddd' }}>
              <h4 style={{ margin: '0 0 10px 0', color: '#333' }}>Total Predictions</h4>
              <div style={{ fontSize: 24, fontWeight: 'bold', color: '#007bff' }}>
                {valueBetsDashboard.totalPredictions || 0}
              </div>
              <div style={{ fontSize: 12, color: '#666', marginTop: 5 }}>
                Active value bets
              </div>
            </div>
            
            <div style={{ padding: 10, background: '#fff', borderRadius: 4, border: '1px solid #ddd' }}>
              <h4 style={{ margin: '0 0 10px 0', color: '#333' }}>Total Games</h4>
              <div style={{ fontSize: 24, fontWeight: 'bold', color: '#28a745' }}>
                {valueBetsDashboard.totalGames || 0}
              </div>
              <div style={{ fontSize: 12, color: '#666', marginTop: 5 }}>
                Across all predictions
              </div>
            </div>
            
            <div style={{ padding: 10, background: '#fff', borderRadius: 4, border: '1px solid #ddd' }}>
              <h4 style={{ margin: '0 0 10px 0', color: '#333' }}>Avg Games/Pred</h4>
              <div style={{ fontSize: 24, fontWeight: 'bold', color: '#17a2b8' }}>
                {valueBetsDashboard.avgGamesPerPrediction || 0}
              </div>
              <div style={{ fontSize: 12, color: '#666', marginTop: 5 }}>
                Average multibet size
              </div>
            </div>
            
            <div style={{ padding: 10, background: '#fff', borderRadius: 4, border: '1px solid #ddd' }}>
              <h4 style={{ margin: '0 0 10px 0', color: '#333' }}>Avg Total Odds</h4>
              <div style={{ fontSize: 24, fontWeight: 'bold', color: '#ffc107' }}>
                {valueBetsDashboard.avgOdds || 0}
              </div>
              <div style={{ fontSize: 12, color: '#666', marginTop: 5 }}>
                Average accumulator odds
              </div>
            </div>
            
            <div style={{ padding: 10, background: '#fff', borderRadius: 4, border: '1px solid #ddd' }}>
              <h4 style={{ margin: '0 0 10px 0', color: '#333' }}>Total Stake</h4>
              <div style={{ fontSize: 24, fontWeight: 'bold', color: '#dc3545' }}>
                KSH {valueBetsDashboard.totalStake || 0}
              </div>
              <div style={{ fontSize: 12, color: '#666', marginTop: 5 }}>
                Sum of all prediction prices
              </div>
            </div>
            
            <div style={{ padding: 10, background: '#fff', borderRadius: 4, border: '1px solid #ddd' }}>
              <h4 style={{ margin: '0 0 10px 0', color: '#333' }}>Potential Win</h4>
              <div style={{ fontSize: 24, fontWeight: 'bold', color: '#28a745' }}>
                KSH {valueBetsDashboard.totalPotentialWin || 0}
              </div>
              <div style={{ fontSize: 12, color: '#666', marginTop: 5 }}>
                If all predictions win
              </div>
            </div>
          </div>
          {/* Dry-run and alter-created toggles */}
          <div style={{ display: 'flex', gap: 12, alignItems: 'center', marginLeft: 16 }}>
            <label style={{ display: 'flex', alignItems: 'center', gap: 6, fontSize: 13 }}>
              <input type="checkbox" checked={state.valueBetsCleanDryRun} onChange={(e) => { const v = e.target.checked; localStorage.setItem('valueBetsCleanDryRun', String(v)); setState(prev => ({ ...prev, valueBetsCleanDryRun: v })); }} />
              Dry Run (Preview)
            </label>
            <label style={{ display: 'flex', alignItems: 'center', gap: 6, fontSize: 13 }}>
              <input type="checkbox" checked={state.valueBetsCleanAlterCreated} onChange={(e) => { const v = e.target.checked; localStorage.setItem('valueBetsCleanAlterCreated', String(v)); setState(prev => ({ ...prev, valueBetsCleanAlterCreated: v })); }} />
              Alter Created Prediction
            </label>
          </div>

          {/* Odds Range Distribution */}
          <div style={{ marginBottom: 20, padding: 15, background: '#fff', borderRadius: 4, border: '1px solid #ddd' }}>
            <h5 style={{ marginTop: 0, color: '#333' }}>Predictions by Odds Range:</h5>
            <div style={{ display: 'flex', gap: 10, flexWrap: 'wrap' }}>
              <div style={{ padding: '8px 12px', borderRadius: 4, background: '#d4edda', border: '1px solid #c3e6cb' }}>
                <strong>Low (&lt;10)</strong>: {valueBetsDashboard.predictionsByOddsRange.low || 0}
              </div>
              <div style={{ padding: '8px 12px', borderRadius: 4, background: '#fff3cd', border: '1px solid #ffeeba' }}>
                <strong>Medium (10-50)</strong>: {valueBetsDashboard.predictionsByOddsRange.medium || 0}
              </div>
              <div style={{ padding: '8px 12px', borderRadius: 4, background: '#f8d7da', border: '1px solid #f5c6cb' }}>
                <strong>High (50-100)</strong>: {valueBetsDashboard.predictionsByOddsRange.high || 0}
              </div>
              <div style={{ padding: '8px 12px', borderRadius: 4, background: '#e2e3e5', border: '1px solid #d6d8db' }}>
                <strong>Very High (100+)</strong>: {valueBetsDashboard.predictionsByOddsRange.veryHigh || 0}
              </div>
            </div>
          </div>

          {/* Game Count Distribution */}
          <div style={{ marginBottom: 20, padding: 15, background: '#fff', borderRadius: 4, border: '1px solid #ddd' }}>
            <h5 style={{ marginTop: 0, color: '#333' }}>Predictions by Game Count:</h5>
            <div style={{ display: 'flex', gap: 10, flexWrap: 'wrap' }}>
              <div style={{ padding: '8px 12px', borderRadius: 4, background: '#cfe2ff', border: '1px solid #b6d4fe' }}>
                <strong>2 Games</strong>: {valueBetsDashboard.predictionsByGameCount['2'] || 0}
              </div>
              <div style={{ padding: '8px 12px', borderRadius: 4, background: '#cfe2ff', border: '1px solid #b6d4fe' }}>
                <strong>3 Games</strong>: {valueBetsDashboard.predictionsByGameCount['3'] || 0}
              </div>
              <div style={{ padding: '8px 12px', borderRadius: 4, background: '#cfe2ff', border: '1px solid #b6d4fe' }}>
                <strong>4 Games</strong>: {valueBetsDashboard.predictionsByGameCount['4'] || 0}
              </div>
              <div style={{ padding: '8px 12px', borderRadius: 4, background: '#cfe2ff', border: '1px solid #b6d4fe' }}>
                <strong>5+ Games</strong>: {valueBetsDashboard.predictionsByGameCount['5+'] || 0}
              </div>
            </div>
          </div>

          {/* Quick Actions */}
          <div style={{ marginBottom: 20, padding: 15, background: '#fff', borderRadius: 4, border: '1px solid #ddd' }}>
            <h5 style={{ marginTop: 0, color: '#333' }}>Quick Actions:</h5>
            <div style={{ display: 'flex', gap: 10, flexWrap: 'wrap' }}>
              <button
                onClick={async () => {
                  const key = 'create_value_bets';
                  if (loading[key]) return;
                  
                  const count = prompt('How many predictions to create?', '100');
                  if (!count) return;
                  
                  setState(prev => ({ ...prev, loading: { ...prev.loading, [key]: true } }));
                  try {
                    addMessage('info', `Creating ${count} value bet predictions with unique keys...`);
                    const response = await fetch(`http://localhost:5000/api/createValueBetsPredictions?count=${count}&minGames=2&maxGames=6`);
                    const data = await response.json();
                    
                    if (data.success) {
                      addMessage('success', data.message);
                      addMessage('info', `Games available: ${data.gamesAvailable}, Filtered: ${data.gamesFiltered}`);
                      fetchValueBetsDashboard();
                    } else {
                      addMessage('error', data.error || 'Failed to create predictions');
                    }
                  } catch (err) {
                    addMessage('error', err.message);
                  } finally {
                    setState(prev => ({ ...prev, loading: { ...prev.loading, [key]: false } }));
                  }
                }}
                disabled={loading.create_value_bets}
                style={{ 
                  padding: '8px 16px', 
                  borderRadius: 4, 
                  border: '1px solid #17a2b8', 
                  background: '#17a2b8', 
                  color: 'white', 
                  cursor: 'pointer',
                  fontWeight: 'bold'
                }}
                title="Create new value bet predictions ensuring no game selection is repeated (unique keys)"
              >
                {loading.create_value_bets ? 'Creating...' : '✨ Create Predictions (Unique)'}
              </button>
              
              <button
                onClick={async () => {
                  const key = 'check_repeated_odds_vb';
                  if (loading[key]) return;
                  setState(prev => ({ ...prev, loading: { ...prev.loading, [key]: true } }));
                  try {
                    const response = await fetch('http://localhost:5000/api/repeated-odds-valuebets');
                    const data = await response.json();
                    const count = data ? Object.keys(data).length : 0;
                    if (count > 0) {
                      addMessage('warning', `Found ${count} repeated odds`);
                      setState(prev => ({ ...prev, repeatedOdds: data }));
                    } else {
                      addMessage('success', 'No repeated odds found');
                    }
                  } catch (err) {
                    addMessage('error', err.message);
                  } finally {
                    setState(prev => ({ ...prev, loading: { ...prev.loading, [key]: false } }));
                  }
                }}
                disabled={loading.check_repeated_odds_vb}
                style={{ padding: '8px 16px', borderRadius: 4, border: '1px solid #007bff', background: '#007bff', color: 'white', cursor: 'pointer' }}
              >
                {loading.check_repeated_odds_vb ? 'Checking...' : '🔍 Check Repeated Odds'}
              </button>
              
              <button
                onClick={async () => {
                  const key = 'clean_repeated_odds_vb';
                  if (loading[key]) return;
                  setState(prev => ({ ...prev, loading: { ...prev.loading, [key]: true } }));
                  try {
                    // Respect dry-run toggle when invoked from quick actions
                    const dryRunParam = state.valueBetsCleanDryRun ? '?dryRun=true' : '';
                    const alterParam = state.valueBetsCleanAlterCreated ? (dryRunParam ? '&alterCreated=true' : '?alterCreated=true') : '';
                    const url = `http://localhost:5000/api/clean-repeated-odds-valuebets${dryRunParam}${alterParam}`;
                    const response = await fetch(url);
                    const data = await response.json();

                    if (state.valueBetsCleanDryRun) {
                      // Show preview modal with returned changes
                      setState(prev => ({ ...prev, valueBetsCleanPreview: data, valueBetsCleanPreviewVisible: true }));
                    } else {
                      // Non-dry-run executed — show confirmation modal with summary
                      setState(prev => ({ ...prev, valueBetsCleanPreview: data, valueBetsCleanConfirmVisible: true }));
                      addMessage('success', data.message || 'Clean executed');
                      fetchValueBetsDashboard();
                    }
                  } catch (err) {
                    addMessage('error', err.message);
                  } finally {
                    setState(prev => ({ ...prev, loading: { ...prev.loading, [key]: false } }));
                  }
                }}
                disabled={loading.clean_repeated_odds_vb}
                style={{ padding: '8px 16px', borderRadius: 4, border: '1px solid #ffc107', background: '#ffc107', color: '#000', cursor: 'pointer' }}
              >
                {loading.clean_repeated_odds_vb ? 'Cleaning...' : '🧹 Clean Repeated Odds'}
              </button>
              
              <button
                onClick={async () => {
                  const key = 'move_outdated_vb';
                  if (loading[key]) return;
                  setState(prev => ({ ...prev, loading: { ...prev.loading, [key]: true } }));
                  try {
                    const response = await fetch('http://localhost:5000/api/moveStartedValueBets');
                    const data = await response.json();
                    addMessage('success', data.message);
                    fetchValueBetsDashboard();
                  } catch (err) {
                    addMessage('error', err.message);
                  } finally {
                    setState(prev => ({ ...prev, loading: { ...prev.loading, [key]: false } }));
                  }
                }}
                disabled={loading.move_outdated_vb}
                style={{ padding: '8px 16px', borderRadius: 4, border: '1px solid #dc3545', background: '#dc3545', color: 'white', cursor: 'pointer' }}
              >
                {loading.move_outdated_vb ? 'Removing...' : '🗑️ Remove Outdated Games'}
              </button>
              
              <Link 
                to="/valuebetspredictions" 
                style={{ 
                  padding: '8px 16px', 
                  borderRadius: 4, 
                  border: '1px solid #28a745', 
                  background: '#28a745', 
                  color: 'white', 
                  textDecoration: 'none',
                  display: 'inline-block'
                }}
              >
                📋 View All Predictions
              </Link>
            </div>
          </div>

          {/* Predictions Table */}
          {valueBetsPredictions && valueBetsPredictions.length > 0 && (
            <div style={{ marginTop: 20 }}>
              <h5 style={{ color: '#333' }}>Recent Predictions (Top 10):</h5>
              <div style={{ overflowX: 'auto' }}>
                <table style={{ width: '100%', borderCollapse: 'collapse', fontSize: 12, background: '#fff' }}>
                  <thead>
                    <tr style={{ background: '#f8f9fa' }}>
                      <th style={{ padding: 8, border: '1px solid #ddd', textAlign: 'left' }}>ID</th>
                      <th style={{ padding: 8, border: '1px solid #ddd', textAlign: 'left' }}>Name</th>
                      <th style={{ padding: 8, border: '1px solid #ddd', textAlign: 'center' }}>Games</th>
                      <th style={{ padding: 8, border: '1px solid #ddd', textAlign: 'right' }}>Total Odds</th>
                      <th style={{ padding: 8, border: '1px solid #ddd', textAlign: 'right' }}>Price (KSH)</th>
                      <th style={{ padding: 8, border: '1px solid #ddd', textAlign: 'right' }}>Potential Win</th>
                      <th style={{ padding: 8, border: '1px solid #ddd', textAlign: 'left' }}>Created</th>
                    </tr>
                  </thead>
                  <tbody>
                    {valueBetsPredictions.slice(0, 10).map((pred, idx) => {
                      const games = pred.games || pred.selectedOdds || [];
                      return (
                        <tr key={idx} style={{ borderBottom: '1px solid #ddd' }}>
                          <td style={{ padding: 8, border: '1px solid #ddd' }}>{pred.id || pred.predictionId || '-'}</td>
                          <td style={{ padding: 8, border: '1px solid #ddd' }}>{pred.predictionName || pred.name || '-'}</td>
                          <td style={{ padding: 8, border: '1px solid #ddd', textAlign: 'center' }}>{games.length}</td>
                          <td style={{ padding: 8, border: '1px solid #ddd', textAlign: 'right', fontWeight: 'bold', color: '#007bff' }}>
                            {parseFloat(pred.totalOdds || 0).toFixed(2)}
                          </td>
                          <td style={{ padding: 8, border: '1px solid #ddd', textAlign: 'right' }}>
                            {parseFloat(pred.price || 0).toFixed(2)}
                          </td>
                          <td style={{ padding: 8, border: '1px solid #ddd', textAlign: 'right', fontWeight: 'bold', color: '#28a745' }}>
                            {parseFloat(pred.potentialWin || 0).toFixed(2)}
                          </td>
                          <td style={{ padding: 8, border: '1px solid #ddd', fontSize: 11 }}>
                            {pred.createdAt ? new Date(pred.createdAt).toLocaleString() : '-'}
                          </td>
                        </tr>
                      );
                    })}
                  </tbody>
                </table>
              </div>
            </div>
          )}
        </div>
      ) : (
        <div style={{ padding: 20, textAlign: 'center', color: '#666' }}>
          <p>Click "Manual Refresh" to load Value Bets data</p>
        </div>
      )}
    </div>
  );

  const renderGameOddsDashboard = () => (
    <div style={{ marginTop: 20, padding: 12, border: '1px solid #ddd', borderRadius: 6, background: '#f8f9fa' }}>
      <div style={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', marginBottom: 15 }}>
        <h3>Game Odds Dashboard</h3>
        <div style={{ 
          padding: '8px 12px', 
          backgroundColor: '#e3f2fd', 
          border: '1px solid #90caf9', 
          borderRadius: 4, 
          marginBottom: 15,
          fontSize: 12,
          color: '#1565c0'
        }}>
          📊 Data Source: Backend Database (http://localhost:5000/api/getodds) | 
          💾 Fallback: Local preds_dump.json | 
          🔄 Auto-refresh: Every 5 minutes | 
          ⏰ Shows: Live and upcoming games only
        </div>
        <div style={{ display: 'flex', gap: 10, alignItems: 'center' }}>
          {state.lastRefresh && (
            <div style={{ fontSize: 12, color: '#666', textAlign: 'right' }}>
              <div>Last: {state.lastRefresh}</div>
              <div style={{ display: 'flex', alignItems: 'center', gap: 5 }}>
                <span>Next in:</span>
                {state.refreshCountdown ? (
                  <span style={{ 
                    fontWeight: 'bold', 
                    color: state.refreshCountdown < 60 ? '#dc3545' : '#007bff',
                    fontFamily: 'monospace'
                  }}>
                    {formatCountdown(state.refreshCountdown)}
                  </span>
                ) : (
                  <span>{state.nextRefresh}</span>
                )}
              </div>
            </div>
          )}
          <button onClick={fetchGameOddsDashboard} disabled={loading.fetch_dashboard}>
            {loading.fetch_dashboard ? 'Loading...' : 'Manual Refresh'}
          </button>
          <button 
            onClick={saveOdds} 
            disabled={loading.save_odds}
            style={{ 
              marginLeft: 10, 
              backgroundColor: '#28a745', 
              color: 'white', 
              border: 'none', 
              padding: '8px 16px', 
              borderRadius: '4px',
              cursor: loading.save_odds ? 'not-allowed' : 'pointer',
              opacity: loading.save_odds ? 0.6 : 1
            }}
            title="Saves current odds data to backend database (overwrite duplicates)"
          >
            {loading.save_odds ? 'Saving...' : 'Save Odds to DB'}
          </button>
          <div style={{ 
            fontSize: '12px', 
            color: '#666', 
            marginTop: '5px',
            fontStyle: 'italic' 
          }}>
            Calls POST http://localhost:5000/api/saveodds with duplicate overwrite enabled
          </div>
        </div>
      </div>
      
      {gameOddsDashboard ? (
        <div>
          <div style={{ display: 'grid', gridTemplateColumns: 'repeat(auto-fit, minmax(200px, 1fr))', gap: 15, marginBottom: 20 }}>
            <div style={{ padding: 10, background: '#fff', borderRadius: 4, border: '1px solid #ddd' }}>
              <h4 style={{ margin: '0 0 10px 0', color: '#333' }}>Total Games</h4>
              <div style={{ fontSize: 24, fontWeight: 'bold', color: '#007bff' }}>
                {gameOddsDashboard.totalGamesInDb || 0}
              </div>
              <div style={{ fontSize: 12, color: '#666', marginTop: 5 }}>
                Live and upcoming games only
              </div>
            </div>
            
            <div style={{ padding: 10, background: '#fff', borderRadius: 4, border: '1px solid #ddd' }}>
              <h4 style={{ margin: '0 0 10px 0', color: '#333' }}>Live Games 🔴</h4>
              <div style={{ fontSize: 24, fontWeight: 'bold', color: '#dc3545' }}>
                {gameOddsDashboard.live || 0}
              </div>
              <div style={{ fontSize: 12, color: '#666', marginTop: 5 }}>
                Games started &lt; 2hrs ago
              </div>
            </div>
            
            <div style={{ padding: 10, background: '#fff', borderRadius: 4, border: '1px solid #ddd' }}>
              <h4 style={{ margin: '0 0 10px 0', color: '#333' }}>Upcoming (2h)</h4>
              <div style={{ fontSize: 24, fontWeight: 'bold', color: '#28a745' }}>
                {gameOddsDashboard.upcomingIn2Hours || 0}
              </div>
              <div style={{ fontSize: 12, color: '#666', marginTop: 5 }}>
                Games starting within 2 hours
              </div>
            </div>
            
            <div style={{ padding: 10, background: '#fff', borderRadius: 4, border: '1px solid #ddd' }}>
              <h4 style={{ margin: '0 0 10px 0', color: '#333' }}>Upcoming (24h)</h4>
              <div style={{ fontSize: 24, fontWeight: 'bold', color: '#ffc107' }}>
                {gameOddsDashboard.upcomingIn24Hours || 0}
              </div>
              <div style={{ fontSize: 12, color: '#666', marginTop: 5 }}>
                Games starting within 24 hours
              </div>
            </div>
            
            <div style={{ padding: 10, background: '#fff', borderRadius: 4, border: '1px solid #ddd' }}>
              <h4 style={{ margin: '0 0 10px 0', color: '#333' }}>Future Games</h4>
              <div style={{ fontSize: 24, fontWeight: 'bold', color: '#17a2b8' }}>
                {gameOddsDashboard.futureGames || 0}
              </div>
              <div style={{ fontSize: 12, color: '#666', marginTop: 5 }}>
                Games beyond 24 hours
              </div>
            </div>
            
            {/* Invalid Time section removed as requested */}
          </div>

          {/* Quick Filter Buttons with Counts */}
          <div style={{ marginBottom: 15 }}>
            <h5 style={{ marginBottom: 10, color: '#333' }}>Quick Filters:</h5>
            <div style={{ display: 'flex', gap: 8, flexWrap: 'wrap' }}>
              <button 
                onClick={() => handleGameOddsFilterChange({ timeCategory: 'upcoming2h' })}
                style={{ 
                  padding: '6px 12px', 
                  borderRadius: 4, 
                  border: gameOddsFilters.timeCategory === 'upcoming2h' ? '2px solid #28a745' : '1px solid #28a745',
                  background: gameOddsFilters.timeCategory === 'upcoming2h' ? '#28a745' : 'white',
                  color: gameOddsFilters.timeCategory === 'upcoming2h' ? 'white' : '#28a745',
                  cursor: 'pointer',
                  fontSize: 12,
                  fontWeight: 'bold'
                }}
              >
                Urgent (2h) • {gameOddsDashboard.upcomingIn2Hours || 0}
              </button>
              
              <button 
                onClick={() => handleGameOddsFilterChange({ timeCategory: 'live' })}
                style={{ 
                  padding: '6px 12px', 
                  borderRadius: 4, 
                  border: gameOddsFilters.timeCategory === 'live' ? '2px solid #dc3545' : '1px solid #dc3545',
                  background: gameOddsFilters.timeCategory === 'live' ? '#dc3545' : 'white',
                  color: gameOddsFilters.timeCategory === 'live' ? 'white' : '#dc3545',
                  cursor: 'pointer',
                  fontSize: 12,
                  fontWeight: 'bold'
                }}
              >
                🔴 Live • {gameOddsDashboard.live || 0}
              </button>
              
              <button 
                onClick={() => handleGameOddsFilterChange({ timeCategory: 'upcoming24h' })}
                style={{ 
                  padding: '6px 12px', 
                  borderRadius: 4, 
                  border: gameOddsFilters.timeCategory === 'upcoming24h' ? '2px solid #ffc107' : '1px solid #ffc107',
                  background: gameOddsFilters.timeCategory === 'upcoming24h' ? '#ffc107' : 'white',
                  color: gameOddsFilters.timeCategory === 'upcoming24h' ? 'white' : '#ffc107',
                  cursor: 'pointer',
                  fontSize: 12,
                  fontWeight: 'bold'
                }}
              >
                Today (24h) • {gameOddsDashboard.upcomingIn24Hours || 0}
              </button>
              
              <button 
                onClick={() => handleGameOddsFilterChange({ timeCategory: 'future' })}
                style={{ 
                  padding: '6px 12px', 
                  borderRadius: 4, 
                  border: gameOddsFilters.timeCategory === 'future' ? '2px solid #17a2b8' : '1px solid #17a2b8',
                  background: gameOddsFilters.timeCategory === 'future' ? '#17a2b8' : 'white',
                  color: gameOddsFilters.timeCategory === 'future' ? 'white' : '#17a2b8',
                  cursor: 'pointer',
                  fontSize: 12,
                  fontWeight: 'bold'
                }}
              >
                Future • {gameOddsDashboard.futureGames || 0}
              </button>
              
              {(gameOddsDashboard.invalidTimestamp || 0) > 0 && (
                <button 
                  onClick={() => handleGameOddsFilterChange({ timeCategory: 'invalid' })}
                  style={{ 
                    padding: '6px 12px', 
                    borderRadius: 4, 
                    border: gameOddsFilters.timeCategory === 'invalid' ? '2px solid #dc3545' : '1px solid #dc3545',
                    background: gameOddsFilters.timeCategory === 'invalid' ? '#dc3545' : 'white',
                    color: gameOddsFilters.timeCategory === 'invalid' ? 'white' : '#dc3545',
                    cursor: 'pointer',
                    fontSize: 12,
                    fontWeight: 'bold'
                  }}
                >
                  Invalid • {gameOddsDashboard.invalidTimestamp || 0}
                </button>
              )}
              
              <button 
                onClick={() => handleGameOddsFilterChange({ timeCategory: 'all' })}
                style={{ 
                  padding: '6px 12px', 
                  borderRadius: 4, 
                  border: gameOddsFilters.timeCategory === 'all' ? '2px solid #007bff' : '1px solid #007bff',
                  background: gameOddsFilters.timeCategory === 'all' ? '#007bff' : 'white',
                  color: gameOddsFilters.timeCategory === 'all' ? 'white' : '#007bff',
                  cursor: 'pointer',
                  fontSize: 12,
                  fontWeight: 'bold'
                }}
              >
                All • {gameOddsDashboard.totalGamesInDb || 0}
              </button>
            </div>
          </div>
        </div>
      ) : (
        <p>Click "Refresh Dashboard" to load statistics</p>
      )}
      
      <div style={{ marginTop: 15, display: 'flex', gap: 10, flexWrap: 'wrap' }}>
        <button onClick={cleanupExpiredGames} disabled={loading.cleanup_expired} style={{ background: '#dc3545', color: 'white', border: 'none', padding: '8px 16px', borderRadius: 4 }}>
          {loading.cleanup_expired ? 'Cleaning...' : 'Cleanup Expired Games'}
        </button>
        <button onClick={() => fetchGameOdds(1, gameOddsFilters)} disabled={loading.fetch_game_odds} style={{ background: '#007bff', color: 'white', border: 'none', padding: '8px 16px', borderRadius: 4 }}>
          {loading.fetch_game_odds ? 'Loading...' : 'Load Game Odds'}
        </button>
        <button 
          onClick={saveOdds} 
          disabled={loading.save_odds}
          style={{ 
            backgroundColor: '#28a745', 
            color: 'white', 
            border: 'none', 
            padding: '8px 16px', 
            borderRadius: 4,
            cursor: loading.save_odds ? 'not-allowed' : 'pointer',
            opacity: loading.save_odds ? 0.6 : 1
          }}
          title="Save odds to backend database with duplicate overwrite"
        >
          {loading.save_odds ? 'Saving Odds...' : 'Save Odds to Backend'}
        </button>
        <button 
          onClick={testDatabaseConnection} 
          disabled={loading.test_db}
          style={{ 
            backgroundColor: '#17a2b8', 
            color: 'white', 
            border: 'none', 
            padding: '8px 16px', 
            borderRadius: 4,
            cursor: loading.test_db ? 'not-allowed' : 'pointer',
            opacity: loading.test_db ? 0.6 : 1
          }}
          title="Test connection to database and reload data"
        >
          {loading.test_db ? 'Testing...' : 'Test DB Connection'}
        </button>
      </div>
    </div>
  );

  const renderGameOddsFilters = () => (
    <div style={{ display: 'flex', gap: 10, flexWrap: 'wrap', marginBottom: 15, padding: 10, background: '#f8f9fa', borderRadius: 4 }}>
      <div style={{ display: 'flex', gap: 10, alignItems: 'center' }}>
        <label style={{ fontWeight: 'bold', fontSize: 14 }}>View:</label>
        <select 
          value={gameOddsViewMode} 
          onChange={(e) => setState(prev => ({ ...prev, gameOddsViewMode: e.target.value }))}
          style={{ padding: 5, borderRadius: 4, border: '1px solid #ddd', fontWeight: 'bold' }}
        >
          <option value="table">Table View</option>
          <option value="competitors">Competitors List</option>
        </select>
      </div>
      
      <div style={{ display: 'flex', gap: 10, alignItems: 'center' }}>
        <label style={{ fontWeight: 'bold', fontSize: 14 }}>Time Filter:</label>
        <select 
          value={gameOddsFilters.timeCategory} 
          onChange={(e) => handleGameOddsFilterChange({ timeCategory: e.target.value })}
          style={{ padding: 5, borderRadius: 4, border: '1px solid #ddd', minWidth: 180 }}
        >
          <option value="all">
            All Games {gameOddsDashboard ? `(${gameOddsDashboard.totalGamesInDb || 0})` : ''}
          </option>
          <option value="upcoming2h">
            Upcoming 2h {gameOddsDashboard ? `(${gameOddsDashboard.upcomingIn2Hours || 0})` : ''}
          </option>
          <option value="live">
            🔴 Live Games {gameOddsDashboard ? `(${gameOddsDashboard.live || 0})` : ''}
          </option>
          <option value="upcoming24h">
            Upcoming 24h {gameOddsDashboard ? `(${gameOddsDashboard.upcomingIn24Hours || 0})` : ''}
          </option>
          <option value="future">
            Future Games {gameOddsDashboard ? `(${gameOddsDashboard.futureGames || 0})` : ''}
          </option>
        </select>
      </div>
      
      <input 
        type="text" 
        placeholder="Search teams, leagues..." 
        value={gameOddsFilters.searchTerm}
        onChange={(e) => handleGameOddsFilterChange({ searchTerm: e.target.value })}
        style={{ padding: 5, borderRadius: 4, border: '1px solid #ddd', minWidth: 200 }}
      />
      
      <select 
        value={gameOddsFilters.sortBy} 
        onChange={(e) => handleGameOddsFilterChange({ sortBy: e.target.value })}
        style={{ padding: 5, borderRadius: 4, border: '1px solid #ddd' }}
      >
        <option value="timestamp">Sort by Time</option>
        <option value="homeTeam">Sort by Home Team</option>
        <option value="awayTeam">Sort by Away Team</option>
        <option value="league">Sort by League</option>
      </select>
      
      <select 
        value={gameOddsFilters.sortOrder} 
        onChange={(e) => handleGameOddsFilterChange({ sortOrder: e.target.value })}
        style={{ padding: 5, borderRadius: 4, border: '1px solid #ddd' }}
      >
        <option value="desc">Descending</option>
        <option value="asc">Ascending</option>
      </select>
    </div>
  );

  const renderGameOddsTable = () => {
    if (!gameOdds || !gameOdds.games || !Array.isArray(gameOdds.games)) {
      return (
        <div style={{ marginTop: 10, padding: 20, textAlign: 'center', color: '#666' }}>
          <p>No game data loaded. Click "Load Game Odds" to fetch data.</p>
        </div>
      );
    }
    
    const games = gameOdds.games || [];
    const currentPage = gameOdds.currentPage || 1;
    const totalPages = gameOdds.totalPages || 1;
    const totalGames = gameOdds.totalGames || 0;
    
    if (games.length === 0) {
      return (
        <div style={{ marginTop: 10, padding: 20, textAlign: 'center', color: '#666' }}>
          <p>No games found for the current filters.</p>
        </div>
      );
    }
    
    return (
      <div style={{ marginTop: 10 }}>
        <div style={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', marginBottom: 10 }}>
          <span>Showing {games.length} of {totalGames} games (Page {currentPage} of {totalPages})</span>
          <div style={{ display: 'flex', gap: 5 }}>
            <button 
              onClick={() => fetchGameOdds(currentPage - 1, gameOddsFilters)} 
              disabled={currentPage <= 1 || loading.fetch_game_odds}
              style={{ padding: '5px 10px', borderRadius: 4, border: '1px solid #ddd' }}
            >
              Previous
            </button>
            <span style={{ padding: '5px 10px' }}>Page {currentPage}</span>
            <button 
              onClick={() => fetchGameOdds(currentPage + 1, gameOddsFilters)} 
              disabled={currentPage >= totalPages || loading.fetch_game_odds}
              style={{ padding: '5px 10px', borderRadius: 4, border: '1px solid #ddd' }}
            >
              Next
            </button>
          </div>
        </div>
        
        <div style={{ overflowX: 'auto' }}>
          <table style={{ width: '100%', borderCollapse: 'collapse', background: 'white' }}>
            <thead>
              <tr style={{ background: '#f8f9fa' }}>
                <th style={{ padding: 10, border: '1px solid #ddd', textAlign: 'left' }}>Game Time</th>
                <th style={{ padding: 10, border: '1px solid #ddd', textAlign: 'left' }}>Competitors</th>
                <th style={{ padding: 10, border: '1px solid #ddd', textAlign: 'left' }}>League</th>
                <th style={{ padding: 10, border: '1px solid #ddd', textAlign: 'left' }}>Odds Count</th>
                <th style={{ padding: 10, border: '1px solid #ddd', textAlign: 'left' }}>Status</th>
                <th style={{ padding: 10, border: '1px solid #ddd', textAlign: 'left' }}>Actions</th>
              </tr>
            </thead>
            <tbody>
              {games.map((game, index) => (
                <tr key={game?.gameId || index} style={{ borderBottom: '1px solid #ddd' }}>
                  <td style={{ padding: 10, border: '1px solid #ddd' }}>
                    {game?.timestamp ? new Date(game.timestamp).toLocaleString() : 'Invalid'}
                  </td>
                  <td style={{ padding: 10, border: '1px solid #ddd' }}>
                    <div style={{ display: 'flex', flexDirection: 'column', gap: 5 }}>
                      <div style={{ display: 'flex', alignItems: 'center', gap: 8 }}>
                        <span style={{ 
                          padding: '2px 8px', 
                          background: '#e3f2fd', 
                          borderRadius: 4, 
                          fontSize: 12, 
                          fontWeight: 'bold',
                          color: '#1976d2'
                        }}>
                          HOME
                        </span>
                        <span style={{ fontWeight: 'bold', color: '#333' }}>{game?.homeTeam || 'Unknown'}</span>
                      </div>
                      <div style={{ display: 'flex', alignItems: 'center', gap: 8 }}>
                        <span style={{ 
                          padding: '2px 8px', 
                          background: '#fce4ec', 
                          borderRadius: 4, 
                          fontSize: 12, 
                          fontWeight: 'bold',
                          color: '#c2185b'
                        }}>
                          AWAY
                        </span>
                        <span style={{ fontWeight: 'bold', color: '#333' }}>{game?.awayTeam || 'Unknown'}</span>
                      </div>
                      <div style={{ fontSize: 11, color: '#666', marginTop: 2 }}>
                        Game ID: {game?.gameId || 'Unknown'}
                      </div>
                    </div>
                  </td>
                  <td style={{ padding: 10, border: '1px solid #ddd' }}>
                    <div style={{ fontWeight: '500' }}>{game?.league || 'Unknown'}</div>
                  </td>
                  <td style={{ padding: 10, border: '1px solid #ddd' }}>
                    <span style={{ 
                      padding: '4px 8px',
                      background: '#f0f0f0',
                      borderRadius: 4,
                      fontWeight: 'bold',
                      color: '#333'
                    }}>
                      {game?.oddsCount || 0}
                    </span>
                  </td>
                  <td style={{ padding: 10, border: '1px solid #ddd' }}>
                    <span style={{ 
                      padding: '4px 8px', 
                      borderRadius: 4, 
                      fontSize: 12,
                      fontWeight: 'bold',
                      background: game?.timeCategory === 'past' ? '#6c757d' : 
                                 game?.timeCategory === 'upcoming2h' ? '#28a745' :
                                 game?.timeCategory === 'upcoming24h' ? '#ffc107' :
                                 game?.timeCategory === 'future' ? '#17a2b8' : '#dc3545',
                      color: 'white'
                    }}>
                      {game?.timeCategory === 'past' ? 'Past' :
                       game?.timeCategory === 'upcoming2h' ? 'Upcoming 2h' :
                       game?.timeCategory === 'upcoming24h' ? 'Upcoming 24h' :
                       game?.timeCategory === 'future' ? 'Future' : 'Invalid'}
                    </span>
                  </td>
                  <td style={{ padding: 10, border: '1px solid #ddd' }}>
                    <button 
                      onClick={() => fetchGameDetails(game?.gameId)}
                      disabled={loading.fetch_game_details || !game?.gameId}
                      style={{ 
                        padding: '6px 12px', 
                        fontSize: 12, 
                        borderRadius: 4, 
                        border: '1px solid #007bff', 
                        background: '#007bff', 
                        color: 'white',
                        cursor: 'pointer',
                        fontWeight: 'bold',
                        opacity: !game?.gameId ? 0.5 : 1
                      }}
                    >
                      View Details
                    </button>
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>
      </div>
    );
  };

  const renderCompetitorsList = () => {
    if (!gameOdds || !gameOdds.games || !Array.isArray(gameOdds.games)) {
      return (
        <div style={{ marginTop: 10, padding: 20, textAlign: 'center', color: '#666' }}>
          <p>No game data loaded. Click "Load Game Odds" to fetch data.</p>
        </div>
      );
    }
    
    const games = gameOdds.games || [];
    const currentPage = gameOdds.currentPage || 1;
    const totalPages = gameOdds.totalPages || 1;
    const totalGames = gameOdds.totalGames || 0;
    
    if (games.length === 0) {
      return (
        <div style={{ marginTop: 10, padding: 20, textAlign: 'center', color: '#666' }}>
          <p>No games found for the current filters.</p>
        </div>
      );
    }
    
    return (
      <div style={{ marginTop: 10 }}>
        <div style={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', marginBottom: 10 }}>
          <span>Showing {games.length} of {totalGames} games (Page {currentPage} of {totalPages})</span>
          <div style={{ display: 'flex', gap: 5 }}>
            <button 
              onClick={() => fetchGameOdds(currentPage - 1, gameOddsFilters)} 
              disabled={currentPage <= 1 || loading.fetch_game_odds}
              style={{ padding: '5px 10px', borderRadius: 4, border: '1px solid #ddd' }}
            >
              Previous
            </button>
            <span style={{ padding: '5px 10px' }}>Page {currentPage}</span>
            <button 
              onClick={() => fetchGameOdds(currentPage + 1, gameOddsFilters)} 
              disabled={currentPage >= totalPages || loading.fetch_game_odds}
              style={{ padding: '5px 10px', borderRadius: 4, border: '1px solid #ddd' }}
            >
              Next
            </button>
          </div>
        </div>
        
        <div style={{ display: 'grid', gridTemplateColumns: 'repeat(auto-fill, minmax(350px, 1fr))', gap: 15 }}>
          {games.map((game, index) => (
            <div key={game?.gameId || index} style={{ 
              background: 'white', 
              border: '1px solid #ddd', 
              borderRadius: 8, 
              padding: 15,
              boxShadow: '0 2px 4px rgba(0,0,0,0.1)'
            }}>
              {/* Game Header */}
              <div style={{ 
                display: 'flex', 
                justifyContent: 'space-between', 
                alignItems: 'center', 
                marginBottom: 12,
                paddingBottom: 8,
                borderBottom: '1px solid #eee'
              }}>
                <div style={{ fontSize: 12, color: '#666' }}>
                  {game?.timestamp ? new Date(game.timestamp).toLocaleString() : 'Invalid Time'}
                </div>
                <span style={{ 
                  padding: '3px 8px', 
                  borderRadius: 4, 
                  fontSize: 11,
                  fontWeight: 'bold',
                  background: game?.timeCategory === 'past' ? '#6c757d' : 
                             game?.timeCategory === 'upcoming2h' ? '#28a745' :
                             game?.timeCategory === 'upcoming24h' ? '#ffc107' :
                             game?.timeCategory === 'future' ? '#17a2b8' : '#dc3545',
                  color: 'white'
                }}>
                  {game?.timeCategory === 'past' ? 'Past' :
                   game?.timeCategory === 'upcoming2h' ? 'Upcoming 2h' :
                   game?.timeCategory === 'upcoming24h' ? 'Upcoming 24h' :
                   game?.timeCategory === 'future' ? 'Future' : 'Invalid'}
                </span>
              </div>

              {/* Competitors */}
              <div style={{ marginBottom: 15 }}>
                <h4 style={{ margin: '0 0 10px 0', fontSize: 14, color: '#333' }}>Competitors</h4>
                
                {/* Home Team */}
                <div style={{ 
                  display: 'flex', 
                  alignItems: 'center', 
                  gap: 10, 
                  marginBottom: 8,
                  padding: 8,
                  background: '#e3f2fd',
                  borderRadius: 6,
                  border: '1px solid #bbdefb'
                }}>
                  <div style={{ 
                    padding: '4px 8px', 
                    background: '#1976d2', 
                    color: 'white',
                    borderRadius: 4, 
                    fontSize: 11, 
                    fontWeight: 'bold'
                  }}>
                    HOME
                  </div>
                  <div style={{ fontWeight: 'bold', fontSize: 15, color: '#1976d2' }}>
                    {game?.homeTeam || 'Unknown Team'}
                  </div>
                </div>

                {/* VS Divider */}
                <div style={{ 
                  textAlign: 'center', 
                  margin: '8px 0',
                  fontSize: 12,
                  fontWeight: 'bold',
                  color: '#666'
                }}>
                  VS
                </div>

                {/* Away Team */}
                <div style={{ 
                  display: 'flex', 
                  alignItems: 'center', 
                  gap: 10,
                  padding: 8,
                  background: '#fce4ec',
                  borderRadius: 6,
                  border: '1px solid #f8bbd9'
                }}>
                  <div style={{ 
                    padding: '4px 8px', 
                    background: '#c2185b', 
                    color: 'white',
                    borderRadius: 4, 
                    fontSize: 11, 
                    fontWeight: 'bold'
                  }}>
                    AWAY
                  </div>
                  <div style={{ fontWeight: 'bold', fontSize: 15, color: '#c2185b' }}>
                    {game?.awayTeam || 'Unknown Team'}
                  </div>
                </div>
              </div>

              {/* Game Info */}
              <div style={{ marginBottom: 15 }}>
                <div style={{ 
                  display: 'flex', 
                  justifyContent: 'space-between', 
                  alignItems: 'center',
                  marginBottom: 8
                }}>
                  <span style={{ fontSize: 12, color: '#666' }}>League:</span>
                  <span style={{ fontWeight: 'bold', fontSize: 13 }}>{game?.league || 'Unknown'}</span>
                </div>
                <div style={{ 
                  display: 'flex', 
                  justifyContent: 'space-between', 
                  alignItems: 'center',
                  marginBottom: 8
                }}>
                  <span style={{ fontSize: 12, color: '#666' }}>Available Odds:</span>
                  <span style={{ 
                    padding: '2px 8px',
                    background: '#f0f0f0',
                    borderRadius: 4,
                    fontWeight: 'bold',
                    fontSize: 12
                  }}>
                    {game?.oddsCount || 0}
                  </span>
                </div>
                <div style={{ 
                  display: 'flex', 
                  justifyContent: 'space-between', 
                  alignItems: 'center'
                }}>
                  <span style={{ fontSize: 12, color: '#666' }}>Game ID:</span>
                  <span style={{ fontSize: 11, fontFamily: 'monospace', color: '#666' }}>{game?.gameId || 'Unknown'}</span>
                </div>
              </div>

              {/* Action Button */}
              <button 
                onClick={() => fetchGameDetails(game?.gameId)}
                disabled={loading.fetch_game_details || !game?.gameId}
                style={{ 
                  width: '100%',
                  padding: '8px 12px', 
                  fontSize: 13, 
                  borderRadius: 6, 
                  border: '1px solid #007bff', 
                  background: '#007bff', 
                  color: 'white',
                  cursor: 'pointer',
                  fontWeight: 'bold',
                  opacity: !game?.gameId ? 0.5 : 1
                }}
              >
                View Game Details & Odds
              </button>
            </div>
          ))}
        </div>
      </div>
    );
  };

  const renderGameOddsSection = () => (
    <div style={{ marginTop: 20, padding: 12, border: '1px solid #ddd', borderRadius: 6 }}>
      <h3>Game Odds Management</h3>
      
      {renderGameOddsFilters()}
      {gameOddsViewMode === 'table' ? renderGameOddsTable() : renderCompetitorsList()}
      
      {selectedGameDetails && selectedGameDetails.gameInfo && (
        <div style={{ marginTop: 20, padding: 15, background: '#f8f9fa', borderRadius: 4, border: '1px solid #ddd' }}>
          <div style={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', marginBottom: 10 }}>
            <h4>Game Details: {selectedGameDetails.gameInfo.homeTeam || 'Unknown'} vs {selectedGameDetails.gameInfo.awayTeam || 'Unknown'}</h4>
            <button onClick={() => setState(prev => ({ ...prev, selectedGameDetails: null }))} style={{ background: 'none', border: '1px solid #666', borderRadius: 4, padding: '2px 8px' }}>×</button>
          </div>
          
          <div style={{ display: 'grid', gridTemplateColumns: 'repeat(auto-fit, minmax(200px, 1fr))', gap: 10, marginBottom: 15 }}>
            <div><strong>League:</strong> {selectedGameDetails.gameInfo.league || 'Unknown'}</div>
            <div><strong>Time:</strong> {selectedGameDetails.gameInfo.timestamp ? new Date(selectedGameDetails.gameInfo.timestamp).toLocaleString() : 'Invalid'}</div>
            <div><strong>Total Odds:</strong> {selectedGameDetails.totalOdds || 0}</div>
            <div><strong>Status:</strong> {selectedGameDetails.timeCategory || 'Unknown'}</div>
          </div>
          
          <div style={{ maxHeight: 300, overflowY: 'auto' }}>
            <h5>Odds Details:</h5>
            {selectedGameDetails.odds && selectedGameDetails.odds.length > 0 ? (
              selectedGameDetails.odds.map((odd, index) => (
                <div key={index} style={{ padding: 8, marginBottom: 5, background: 'white', borderRadius: 4, border: '1px solid #ddd' }}>
                  <div style={{ display: 'flex', justifyContent: 'space-between' }}>
                    <span><strong>{odd.selectionName || 'Unknown Selection'}</strong></span>
                    <span style={{ fontWeight: 'bold', color: '#007bff' }}>{odd.odds || 'N/A'}</span>
                  </div>
                  <div style={{ fontSize: 12, color: '#666' }}>
                    Market: {odd.marketName || 'Unknown'} | Type: {odd.periodType || 'Unknown'}
                  </div>
                </div>
              ))
            ) : (
              <p style={{ color: '#666', fontStyle: 'italic' }}>No odds available for this game.</p>
            )}
          </div>
        </div>
      )}
    </div>
  );

  // if (!isAuthenticated) return <div>Loading...</div>; // Disabled for testing

  return (
    <div style={{ padding: 20 }}>
      <h1>Admin Panel</h1>
      <div style={{ display: 'grid', gridTemplateColumns: 'repeat(auto-fit, minmax(250px, 1fr))', gap: 10, marginBottom: 20 }}>
        {/* Data Fetching & Saving */}
        <button onClick={() => executeEndpoint('/api/saveData', 'GET', null, 'Save Countries & Leagues')} disabled={loading.save_countries_leagues}>
          {loading.save_countries_leagues ? 'Loading...' : 'Save Countries & Leagues'}
        </button>
        <button onClick={() => executeEndpoint('/api/getData', 'GET', null, 'Get & Save Games')} disabled={loading.get_save_games}>
          {loading.get_save_games ? 'Loading...' : 'Get & Save Games'}
        </button>
        <button onClick={() => executeEndpoint('/api/getGameIds', 'GET', null, 'Get Game IDs')} disabled={loading.get_game_ids}>
          {loading.get_game_ids ? 'Loading...' : 'Get Game IDs'}
        </button>
        <button onClick={() => executeEndpoint('/api/allGameData', 'GET', null, 'Save All Game Data')} disabled={loading.save_all_game_data}>
          {loading.save_all_game_data ? 'Loading...' : 'Save All Game Data'}
        </button>
        <button onClick={() => executeEndpoint('/api/saveodds', 'GET', null, 'Filter & Save Odds <2.7')} disabled={loading.save_odds}>
          {loading.save_odds ? 'Loading...' : 'Filter & Save Odds <2.7'}
        </button>

        {/* Predictions Creation & Viewing */}
        <button onClick={() => executeEndpoint('/api/createPredictions', 'GET', null, 'Create 100 Predictions')} disabled={loading.create_predictions}>
          {loading.create_predictions ? 'Loading...' : 'Create 100 Predictions'}
        </button>
        <button onClick={() => executeEndpoint('/api/lettherebetodayspredictions', 'GET', null, 'Create Today\'s Predictions (24h)')} disabled={loading.create_todays_24h}>
          {loading.create_todays_24h ? 'Loading...' : 'Create Today\'s Predictions (24h)'}
        </button>
        <button onClick={() => executeEndpoint('/api/createTodaysPredictionz', 'GET', null, 'Create Today\'s Predictions (Exact)')} disabled={loading.create_todays_exact}>
          {loading.create_todays_exact ? 'Loading...' : 'Create Today\'s Predictions (Exact)'}
        </button>
        <button onClick={() => executeEndpoint('/api/createTodaysPredictions', 'GET', null, 'Create SMS Bets for Today')} disabled={loading.create_sms_today}>
          {loading.create_sms_today ? 'Loading...' : 'Create SMS Bets for Today'}
        </button>

        {/* Cleaning & Maintenance */}
        <button onClick={() => executeEndpoint('/api/clean-repeated-odds', 'GET', null, 'Clean Repeated Odds')} disabled={loading.clean_repeated_odds}>
          {loading.clean_repeated_odds ? 'Loading...' : 'Clean Repeated Odds'}
        </button>
        <button onClick={() => executeEndpoint('/api/checkOutdatedPredictions', 'GET', null, 'Check & Move Outdated Predictions')} disabled={loading.check_outdated}>
          {loading.check_outdated ? 'Loading...' : 'Check & Move Outdated Predictions'}
        </button>
        <button onClick={() => executeEndpoint('/api/cleanStartedGames', 'GET', null, 'Clean Started Games')} disabled={loading.clean_started_games}>
          {loading.clean_started_games ? 'Loading...' : 'Clean Started Games'}
        </button>
        
        {/* Value Bets Management */}
        <button 
          onClick={handleValueBetsWorkflow} 
          disabled={loading.value_bets_workflow}
          style={{
            backgroundColor: loading.value_bets_workflow ? '#6c757d' : '#28a745',
            color: 'white',
            fontWeight: 'bold',
            border: 'none',
            padding: '10px 15px',
            borderRadius: '4px',
            cursor: loading.value_bets_workflow ? 'not-allowed' : 'pointer'
          }}
          title="Automated workflow: Check repeated odds → Clean duplicates → Wait 5 minutes → Remove outdated predictions → Create SMS bets (stake 1 KSH)"
        >
          {loading.value_bets_workflow ? '⏳ Running Workflow...' : '🎯 Value Bets Workflow (5min)'}
        </button>

        <button onClick={() => executeEndpoint('/api/evaluateFirstPrediction', 'GET', null, 'Evaluate First Prediction')} disabled={loading.evaluate_first}>
          {loading.evaluate_first ? 'Loading...' : 'Evaluate First Prediction'}
        </button>

        {/* Other */}
        <button onClick={() => executeEndpoint('/api/mpesaPayment', 'POST', { phoneNumber: '+254700000000', amount: 100, accountReference: 'TEST', transactionDesc: 'Admin Test' }, 'Test Mpesa Payment')} disabled={loading.test_mpesa}>
          {loading.test_mpesa ? 'Loading...' : 'Test Mpesa Payment'}
        </button>
        <button onClick={() => executeEndpoint('/api/createSmsBets', 'POST', { predictions: [], stake: 1 }, 'Create SMS Bets (Manual)')} disabled={loading.create_sms_manual}>
          {loading.create_sms_manual ? 'Loading...' : 'Create SMS Bets (Manual)'}
        </button>
      </div>

      {/* Stats with Navigation Links */}
      <div style={{ marginBottom: 20, padding: 10, background: '#f0f0f0', borderRadius: 4 }}>
        <h3>Stats</h3>
        <p>Predictions: {predictionsCount} <Link to="/betlist" style={{ fontSize: 14, color: '#007bff', textDecoration: 'underline' }}>View All Predictions →</Link></p>
        <p>Odds Count: {oddsCount}</p>
        <button onClick={fetchStats} style={{ marginTop: 10 }}>Refresh Stats</button>
        <Link to="/todayspredictions" style={{ marginLeft: 10, textDecoration: 'none', color: '#007bff' }}>View Today's Predictions →</Link>
      </div>
      {renderValueBetsDashboard()}
      {renderRepeatedOddsSection()}
      {renderGameOddsDashboard()}
      {renderGameOddsSection()}

      {/* Live logs viewer */}
      <div style={{ marginTop: 20, padding: 12, border: '1px solid #ddd', borderRadius: 6, background: '#fff' }}>
        <div style={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center' }}>
          <h3 style={{ margin: 0 }}>Server Logs / Task Output</h3>
          <div style={{ display: 'flex', gap: 8, alignItems: 'center' }}>
            <button onClick={() => setState(prev => ({ ...prev, tailing: !prev.tailing }))}>
              {state.tailing ? 'Stop Tail' : 'Start Tail'}
            </button>
            <button onClick={() => fetchLogs()}>Refresh</button>
            <button onClick={() => setState(prev => ({ ...prev, logLines: [] }))}>Clear</button>
          </div>
        </div>

        <div style={{ marginTop: 10, display: 'flex', gap: 12 }}>
          <div style={{ flex: 1 }}>
            <div style={{ height: 300, overflow: 'auto', background: '#0b0b0b', color: '#e6e6e6', padding: 8, borderRadius: 4, fontFamily: 'monospace', fontSize: 12 }}>
              {state.logLines && state.logLines.length ? state.logLines.map((l, i) => (
                <div key={i} style={{ whiteSpace: 'pre-wrap' }}>{l}</div>
              )) : <div style={{ opacity: 0.6 }}>No logs loaded. Click Refresh or Start Tail to begin.</div>}
            </div>
          </div>

          <div style={{ width: 220 }}>
            <div style={{ marginBottom: 8 }}><strong>Task Status</strong></div>
            <div style={{ marginBottom: 8 }}>
              {Object.keys(loading).some(k => loading[k]) ? (
                <div style={{ color: '#155724', background: '#d4edda', padding: 8, borderRadius: 4 }}>A task appears to be running</div>
              ) : (
                <div style={{ color: '#856404', background: '#fff3cd', padding: 8, borderRadius: 4 }}>No active tasks</div>
              )}
            </div>
            <div>
              <div style={{ marginBottom: 6 }}><strong>Controls</strong></div>
              <button onClick={() => executeEndpoint('/api/checkOutdatedPredictions', 'GET', null, 'Check & Move Outdated Predictions')} disabled={loading.check_outdated}>Run Check Outdated</button>
            </div>
          </div>
        </div>
      </div>

      {/* User management (visible to admins) */}
      <div style={{ marginTop: 20 }}>
        <h3>Users</h3>
        <div style={{ marginBottom: 8 }}>
          <button onClick={handleLoadUsers}>Load Users</button>
        </div>
        <div>
          {state.users && state.users.length ? (
            <ul>
              {state.users.map(u => (
                <li key={u.uid} style={{ marginBottom: 6 }}>
                  <strong>{u.displayName || u.email}</strong> ({u.email}) - Role: {u.claims?.role || 'user'}
                  <button style={{ marginLeft: 8 }} onClick={() => handlePromote(u.uid)}>Promote</button>
                </li>
              ))}
            </ul>
          ) : <div>No users loaded</div>}
        </div>
      </div>
      {renderMessages()}
      {/* Dry-run preview modal */}
      {state.valueBetsCleanPreviewVisible && (
        <div style={{ position: 'fixed', left: 0, top: 0, right: 0, bottom: 0, background: 'rgba(0,0,0,0.5)', display: 'flex', alignItems: 'center', justifyContent: 'center', zIndex: 2000 }}>
          <div style={{ width: '80%', maxHeight: '80%', overflow: 'auto', background: 'white', padding: 20, borderRadius: 8 }}>
            <h3>Dry-run Preview — Clean Repeated Odds</h3>
            <p style={{ color: '#666' }}>This is a preview of changes that would be applied. No data was modified.</p>
            <pre style={{ whiteSpace: 'pre-wrap', fontSize: 12, background: '#f8f9fa', padding: 12, borderRadius: 6, maxHeight: '60vh', overflow: 'auto' }}>{JSON.stringify(state.valueBetsCleanPreview || {}, null, 2)}</pre>
            <div style={{ marginTop: 12, display: 'flex', gap: 8, justifyContent: 'flex-end' }}>
              <button onClick={() => setState(prev => ({ ...prev, valueBetsCleanPreviewVisible: false, valueBetsCleanPreview: null }))} style={{ padding: '8px 12px' }}>Close</button>
              <button onClick={async () => {
                // Apply the real run: call endpoint without dryRun param but keep alterCreated if set
                try {
                  const alterParam = state.valueBetsCleanAlterCreated ? '?alterCreated=true' : '';
                  const resp = await fetch(`http://localhost:5000/api/clean-repeated-odds-valuebets${alterParam}`);
                  const json = await resp.json();
                  setState(prev => ({ ...prev, valueBetsCleanPreviewVisible: false, valueBetsCleanPreview: json }));
                  addMessage('success', json.message || 'Clean executed');
                  fetchValueBetsDashboard();
                } catch (err) { addMessage('error', err.message); }
              }} style={{ padding: '8px 12px', background: '#dc3545', color: 'white', border: 'none', borderRadius: 4 }}>Apply Changes</button>
            </div>
          </div>
        </div>
      )}

      {/* Confirmation modal for executed clean (summary) */}
      {state.valueBetsCleanConfirmVisible && (
        <div style={{ position: 'fixed', left: 0, top: 0, right: 0, bottom: 0, background: 'rgba(0,0,0,0.5)', display: 'flex', alignItems: 'center', justifyContent: 'center', zIndex: 2000 }}>
          <div style={{ width: '60%', background: 'white', padding: 20, borderRadius: 8 }}>
            <h3>Clean Result</h3>
            <p style={{ color: '#666' }}>Summary of the performed clean operation:</p>
            <pre style={{ whiteSpace: 'pre-wrap', fontSize: 12, background: '#f8f9fa', padding: 12, borderRadius: 6 }}>{JSON.stringify(state.valueBetsCleanPreview || {}, null, 2)}</pre>
            <div style={{ marginTop: 12, display: 'flex', gap: 8, justifyContent: 'flex-end' }}>
              <button onClick={() => setState(prev => ({ ...prev, valueBetsCleanConfirmVisible: false, valueBetsCleanPreview: null }))} style={{ padding: '8px 12px' }}>Close</button>
            </div>
          </div>
        </div>
      )}

    </div>
  );
};

export default AdminPanel;