import React, { useEffect, useState, useContext } from 'react';
import { AuthContext } from '../context/auth-context';

const AdminOrders = () => {
  const { user } = useContext(AuthContext);
  const [orders, setOrders] = useState([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);

  useEffect(() => {
    const load = async () => {
      if (!user) return setLoading(false);
      try {
        const token = await user.getIdToken();
        const res = await fetch('http://localhost:5000/api/admin/orders', { headers: { Authorization: `Bearer ${token}` } });
        if (!res.ok) throw new Error('Failed to fetch admin orders');
        const json = await res.json();
        setOrders(json.orders || []);
      } catch (e) {
        setError(e.message || 'Error');
      } finally {
        setLoading(false);
      }
    };
    load();
  }, [user]);

  if (loading) return <div>Loading orders...</div>;
  if (!user) return <div>Please log in as admin to see all orders.</div>;
  if (error) return <div style={{ color: 'red' }}>{error}</div>;

  return (
    <div style={{ padding: 16 }}>
      <h2>All Orders (Admin)</h2>
      {orders.length === 0 ? (
        <p>No orders found.</p>
      ) : (
        <div>
          {orders.map(o => (
            <div key={o.orderId} style={{ border: '1px solid #ddd', padding: 12, marginBottom: 8 }}>
              <div><strong>Order ID:</strong> <a href={`/orders/${o.orderId}`}>{o.orderId}</a></div>
              <div><strong>User:</strong> {o.uid}</div>
              <div><strong>Created:</strong> {new Date(o.createdAt).toLocaleString()}</div>
              <div><strong>Amount:</strong> Ksh. {o.amount}.00</div>
              <div><strong>Status:</strong> {o.status}</div>
              <div><strong>Prediction:</strong> {o.prediction && o.prediction.predictionName}</div>
              <div style={{ marginTop: 8 }}>
                <button onClick={async () => {
                  try {
                    const token = await user.getIdToken();
                    await fetch(`http://localhost:5000/api/orders/${encodeURIComponent(o.orderId)}/status`, { method: 'POST', headers: { 'Content-Type': 'application/json', Authorization: `Bearer ${token}` }, body: JSON.stringify({ status: 'paid' }) });
                    // refresh list
                    const newRes = await fetch('http://localhost:5000/api/admin/orders', { headers: { Authorization: `Bearer ${token}` } });
                    const json = await newRes.json();
                    setOrders(json.orders || []);
                  } catch (e) {
                    alert('Failed to update status');
                  }
                }} style={{ marginRight: 8 }}>Mark Paid</button>

                <button onClick={async () => {
                  try {
                    const token = await user.getIdToken();
                    await fetch(`http://localhost:5000/api/orders/${encodeURIComponent(o.orderId)}/status`, { method: 'POST', headers: { 'Content-Type': 'application/json', Authorization: `Bearer ${token}` }, body: JSON.stringify({ status: 'failed' }) });
                    const newRes = await fetch('http://localhost:5000/api/admin/orders', { headers: { Authorization: `Bearer ${token}` } });
                    const json = await newRes.json();
                    setOrders(json.orders || []);
                  } catch (e) {
                    alert('Failed to update status');
                  }
                }}>Mark Failed</button>
              </div>
            </div>
          ))}
        </div>
      )}
    </div>
  );
};

export default AdminOrders;
