import { useEffect, useState } from 'react';

function isHtmlResponse(res) {
  const contentType = res.headers.get('content-type') || '';
  return contentType.includes('text/html');
}

export function useRepeatedOdds() {
  const [data, setData] = useState(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);

  useEffect(() => {
    setLoading(true);

    const backendBase = process.env.REACT_APP_API_BASE || 'http://localhost:5000';
    const url = `${backendBase.replace(/\/$/, '')}/api/repeated-odds`;

    const tryFetch = (fetchUrl) =>
      fetch(fetchUrl, { credentials: 'include' })
        .then(async (res) => {
          if (!res.ok) {
            const text = await res.text().catch(() => '');
            throw new Error(`Failed to fetch repeated odds: ${res.status} ${res.statusText} ${text}`);
          }

          if (isHtmlResponse(res)) {
            const text = await res.text().catch(() => '');
            throw new Error(`Expected JSON but received HTML: ${text.slice(0, 500)}`);
          }

          return res.json();
        });

    tryFetch(url)
      .catch((primaryErr) => {
        // If the primary backend fetch fails (network, CORS, server down, etc.),
        // try the relative path (dev-server proxy). Preserve both errors so
        // the UI can show a helpful diagnostic instead of only the HTML index.
        if (url.startsWith('http://localhost:5000') || url.startsWith('https://')) {
          console.warn('Primary repeated-odds fetch failed:', primaryErr);
          return tryFetch('/api/repeated-odds').catch((fallbackErr) => {
            throw new Error(`Primary fetch failed: ${primaryErr.message}; fallback failed: ${fallbackErr.message}`);
          });
        }
        throw primaryErr;
      })
      .then((json) => {
        setData(json);
        setLoading(false);
      })
      .catch((err) => {
        setError(err);
        setLoading(false);
      });
  }, []);

  return { data, loading, error };
}

