import React, { useContext } from 'react';
import jsPDF from 'jspdf';
import '../ProductReceipt/ProductReceipt.css';
import { AuthContext } from '../../context/auth-context';

const ProductReceipt = ({ product, customer, paymentReference }) => {
  const { user } = useContext(AuthContext);
  const productPrice = product?.price || 0;

  const defaultCustomer = customer || {
    name: user?.displayName || (user?.email || '').split('@')[0],
    contact: user?.email || 'customer@example.com'
  };

  const downloadPDF = () => {
    try {
      const doc = new jsPDF();
      const lineHeight = 10;
      let yPosition = 20;

      // Header
      doc.setFontSize(18);
      doc.text('PRODUCT RECEIPT', 20, yPosition);
      yPosition += lineHeight * 1.5;

      // Receipt Details
      doc.setFontSize(10);
      doc.text(`Receipt ID: ${paymentReference || 'N/A'}`, 20, yPosition);
      yPosition += lineHeight;
      doc.text(`Date: ${new Date().toLocaleDateString()}`, 20, yPosition);
      yPosition += lineHeight;
      doc.text(`Time: ${new Date().toLocaleTimeString()}`, 20, yPosition);
      yPosition += lineHeight * 1.5;

      // Customer Details
      doc.setFontSize(12);
      doc.text('CUSTOMER DETAILS', 20, yPosition);
      yPosition += lineHeight;
      doc.setFontSize(10);
      doc.text(`Name: ${defaultCustomer.name}`, 20, yPosition);
      yPosition += lineHeight;
      doc.text(`Contact: ${defaultCustomer.contact}`, 20, yPosition);
      yPosition += lineHeight * 1.5;

      // Product Details
      doc.setFontSize(12);
      doc.text('PRODUCT DETAILS', 20, yPosition);
      yPosition += lineHeight;
      doc.setFontSize(10);
      doc.text(`Product Name: ${product?.predictionName || product?.name || 'N/A'}`, 20, yPosition);
      yPosition += lineHeight;
      doc.text(`Category: ${product?.category || 'N/A'}`, 20, yPosition);
      yPosition += lineHeight;
      doc.text(`Brand: ${product?.brand || 'N/A'}`, 20, yPosition);
      yPosition += lineHeight;
      if (product?.description) {
        const descLines = doc.splitTextToSize(`Description: ${product.description}`, 170);
        doc.text(descLines, 20, yPosition);
        yPosition += lineHeight * descLines.length;
      }
      yPosition += lineHeight;

      // Amount
      doc.setFontSize(12);
      doc.text('AMOUNT', 20, yPosition);
      yPosition += lineHeight;
      doc.setFontSize(14);
      doc.setFont(undefined, 'bold');
      doc.text(`KES ${productPrice.toFixed(2)}`, 20, yPosition);
      doc.setFont(undefined, 'normal');
      yPosition += lineHeight * 1.5;

      // Payment Status
      doc.setFontSize(12);
      doc.text('PAYMENT STATUS', 20, yPosition);
      yPosition += lineHeight;
      doc.setFontSize(10);
      doc.text('Paid', 20, yPosition);
      yPosition += lineHeight * 2;

      // Footer
      doc.setFontSize(9);
      doc.text('Thank you for your purchase!', 20, yPosition);
      yPosition += lineHeight;
      doc.text('For support, contact: support@predictai.com', 20, yPosition);

      // Save PDF
      doc.save(`product-receipt-${paymentReference || 'download'}.pdf`);
    } catch (error) {
      console.error('Error generating PDF:', error);
      alert('Failed to generate PDF');
    }
  };

  return (
    <div className="product-receipt-container">
      <h1>Product Receipt</h1>
      <p className="divider">=======</p>

      <div className="receipt-section">
        <h3>Receipt Details</h3>
        <p className="divider">-----------------</p>
        <p><strong>Receipt ID:</strong> {paymentReference || 'N/A'}</p>
        <p><strong>Date:</strong> {new Date().toLocaleDateString()}</p>
        <p><strong>Time:</strong> {new Date().toLocaleTimeString()}</p>
      </div>

      <div className="receipt-section">
        <h3>Customer Details</h3>
        <p className="divider">-----------------</p>
        <p><strong>Name:</strong> {defaultCustomer.name}</p>
        <p><strong>Contact:</strong> {defaultCustomer.contact}</p>
      </div>

      <div className="receipt-section">
        <h3>Product Details</h3>
        <p className="divider">-----------------</p>
        <p><strong>Product Name:</strong> {product?.predictionName || product?.name || 'N/A'}</p>
        <p><strong>Category:</strong> {product?.category || 'N/A'}</p>
        <p><strong>Brand:</strong> {product?.brand || 'N/A'}</p>
        {product?.description && (
          <p><strong>Description:</strong> {product.description}</p>
        )}
        {product?.image && (
          <div style={{ marginTop: '10px' }}>
            <img 
              src={product.image} 
              alt={product.name || 'Product'} 
              style={{ maxWidth: '200px', maxHeight: '200px', objectFit: 'cover', borderRadius: '4px' }} 
            />
          </div>
        )}
      </div>

      <div className="receipt-section">
        <h3>Amount</h3>
        <p className="divider">---------</p>
        <p style={{ fontSize: '1.5rem', fontWeight: 'bold', color: '#007bff' }}>
          KES {productPrice.toFixed(2)}
        </p>
      </div>

      <div className="receipt-section">
        <h3>Payment Status</h3>
        <p className="divider">---------------</p>
        <p style={{ fontSize: '1.1rem', fontWeight: 'bold', color: '#28a745' }}>✓ Paid</p>
      </div>

      <button onClick={downloadPDF} className="btn-receipt btn-download">
        Download PDF Receipt
      </button>
    </div>
  );
};

export default ProductReceipt;
