import React, { useState, useContext } from 'react';
import { AuthContext } from '../context/auth-context';
import axios from 'axios';
import { loadStripe } from '@stripe/stripe-js';
import { useNavigate } from 'react-router-dom';
import {
    Elements,
    CardElement,
    useStripe,
    useElements
} from '@stripe/react-stripe-js';

// import { useAuth } from '../contexts/AuthContext'; // optional: use if available

const stripePromise = loadStripe(process.env.REACT_APP_STRIPE_PUBLISHABLE_KEY || 'pk_test_replace_me');

const CardForm = ({ amount, onPaid, setError, setLoading, prediction, onPaymentSuccess }) => {
    const { user } = useContext(AuthContext);
    const stripe = useStripe();
    const elements = useElements();
    const navigate = useNavigate();

    const handleCardPay = async (e) => {
        e.preventDefault();
        if (!stripe || !elements) return;
        setError(null);
        setLoading(true);
        try {
            // Create payment intent on server
            const resp = await axios.post('http://localhost:5000/api/payments/create-card-payment', {
                amount,
                userId: user?.uid || 'guest'
            });
            const clientSecret = resp.data.clientSecret;
            const cardElement = elements.getElement(CardElement);
            const result = await stripe.confirmCardPayment(clientSecret, {
                payment_method: { card: cardElement }
            });
            if (result.error) {
                setError(result.error.message);
                setLoading(false);
            } else {
                if (result.paymentIntent && result.paymentIntent.status === 'succeeded') {
                    // Call the payment success callback with prediction and reference
                    if (onPaymentSuccess) {
                        onPaymentSuccess(result.paymentIntent.id);
                    } else {
                        // Fallback: navigate to prediction details if callback not provided
                        if (prediction) {
                            navigate('/prediction-details', {
                                state: {
                                    prediction: prediction,
                                    fromPayment: true,
                                    paymentReference: result.paymentIntent.id,
                                    showInvoice: true
                                }
                            });
                        }
                        onPaid();
                    }
                } else {
                    setError('Payment not completed.');
                    setLoading(false);
                }
            }
        } catch (err) {
            setError(err.response?.data?.message || err.message || 'Card payment failed');
            setLoading(false);
        }
    };

    return (
        <form onSubmit={handleCardPay}>
            <div className="mb-4">
                <label className="block text-sm font-medium mb-1">Card details</label>
                <div className="p-2 border rounded">
                    <CardElement />
                </div>
            </div>
            <button type="submit" className="w-full bg-blue-600 text-white py-2 rounded">Pay with Card</button>
        </form>
    );
};

const PaymentModal = ({ isOpen, onClose, amount, onSuccess, prediction }) => {
    const [phoneNumber, setPhoneNumber] = useState('');
    const [loading, setLoading] = useState(false);
    const [error, setError] = useState(null);
    const [paymentStatus, setPaymentStatus] = useState(null);
    const [method, setMethod] = useState('mpesa');
    const [paybillFallback, setPaybillFallback] = useState(null);
    const [mpesaCode, setMpesaCode] = useState('');
    const { user } = useContext(AuthContext);
    const navigate = useNavigate();

    const initiatePayment = async (e) => {
        e.preventDefault();
        setLoading(true);
        setError(null);
        setPaybillFallback(null);

        try {
            // Use AuthContext user if available
            const userId = user?.uid || 'guest';
            // Normalize phone: convert leading 0 to 254 and strip +
            let phone = (phoneNumber || '').toString().trim();
            if (phone.startsWith('+')) phone = phone.slice(1);
            if (phone.startsWith('0')) phone = '254' + phone.slice(1);

            const response = await axios.post('http://localhost:5000/api/stkpush', {
                phone: phone,
                accountNumber: userId,
                amount: amount
            });

            // Store PayBill fallback option if provided
            if (response.data.paybillFallback) {
                setPaybillFallback(response.data.paybillFallback);
            }

            if (response.data.success) {
                // If backend returned an internal reference we can poll, otherwise show success
                if (response.data && response.data.reference) {
                    // Use Promise with timeout: show PayBill after 30 seconds if no response
                    pollPaymentStatus(response.data.reference, 30000); // 30 second timeout
                    setLoading(true);
                } else {
                    setPaymentStatus('initiated');
                    setLoading(false);
                }
            } else {
                // STK failed but PayBill fallback is available
                setError(response.data.message);
                setLoading(false);
                setPaymentStatus('paybill_fallback');
            }
        } catch (error) {
            const errorMessage = error.response?.data?.message || error.response?.data?.error || 'Failed to initiate payment';
            setError(errorMessage);
            
            // Store PayBill fallback from error response if available
            if (error.response?.data?.paybillFallback) {
                setPaybillFallback(error.response.data.paybillFallback);
                setPaymentStatus('paybill_fallback');
            }
            setLoading(false);
        }
    };

    const pollPaymentStatus = async (reference, timeoutMs = 30000) => {
        console.log('[PaymentModal] Starting poll for ref:', reference, 'Timeout:', timeoutMs, 'prediction:', prediction);
        
        let pollInterval;
        let timeoutReached = false;
        
        const pollPromise = new Promise((resolve) => {
            pollInterval = setInterval(async () => {
                if (timeoutReached) return; // Stop polling if timeout reached
                
                try {
                    console.log('[PaymentModal] Polling status...');
                    const response = await axios.get(`http://localhost:5000/api/payments/status/${reference}`);
                    const { payment } = response.data;
                    console.log('[PaymentModal] Status:', payment.status);

                    if (payment.status === 'completed') {
                        clearInterval(pollInterval);
                        timeoutReached = true;
                        console.log('[PaymentModal] Completed! Redirecting...');
                        setPaymentStatus('completed');
                        
                        // Call the onSuccess callback first (for any additional handling)
                        if (onSuccess) onSuccess();
                        
                        // Redirect to prediction-details with invoice after successful payment
                        setTimeout(() => {
                            console.log('[PaymentModal] Navigate - prediction:', prediction);
                            if (prediction) {
                                navigate('/prediction-details', {
                                    state: {
                                        prediction: prediction,
                                        fromPayment: true,
                                        paymentReference: reference,
                                        showInvoice: true
                                    }
                                });
                            } else {
                                console.error('[PaymentModal] prediction is undefined!');
                            }
                            onClose(); // Close the modal
                        }, 1500); // Give slight delay for visual feedback
                        resolve('completed');
                    } else if (payment.status === 'failed') {
                        clearInterval(pollInterval);
                        timeoutReached = true;
                        setPaymentStatus('failed');
                        setError(payment.failureReason || 'Payment failed');
                        resolve('failed');
                    }
                } catch (error) {
                    console.error('[PaymentModal] Polling error:', error);
                }
            }, 3000); // Poll every 3 seconds
        });

        // Timeout promise - resolves after specified time
        const timeoutPromise = new Promise((resolve) => {
            setTimeout(() => {
                clearInterval(pollInterval);
                timeoutReached = true;
                console.log('[PaymentModal] Timeout reached! Showing PayBill fallback...');
                resolve('timeout');
            }, timeoutMs);
        });

        // Race: whichever completes first
        const result = await Promise.race([pollPromise, timeoutPromise]);
        
        // If timeout reached and no payment detected, show PayBill fallback
        if (result === 'timeout') {
            console.log('[PaymentModal] Payment took too long. Switching to PayBill manual mode...');
            setError('STK push is taking longer than expected. Please pay manually using M-Pesa PayBill below:');
            setPaymentStatus('paybill_fallback');
            setLoading(false);
        }
    };

    if (!isOpen) return null;

    return (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center p-4">
            <div className="bg-white rounded-lg p-6 max-w-md w-full">
                <h2 className="text-xl font-bold mb-4">Complete Payment</h2>
                <p className="mb-4">Amount to pay: KES {amount}</p>

                {!loading && !paymentStatus && (
                    <div>
                        <div className="flex gap-2 mb-4">
                            <button className={`px-3 py-1 rounded ${method==='mpesa' ? 'bg-green-500 text-white' : 'bg-gray-200'}`} onClick={() => setMethod('mpesa')}>M-Pesa</button>
                            <button className={`px-3 py-1 rounded ${method==='card' ? 'bg-blue-500 text-white' : 'bg-gray-200'}`} onClick={() => setMethod('card')}>Card</button>
                        </div>

                        {method === 'mpesa' && (
                            <form onSubmit={initiatePayment}>
                                <div className="mb-4">
                                    <label className="block text-sm font-medium mb-1">M-Pesa Phone Number</label>
                                    <input type="tel" value={phoneNumber} onChange={(e)=>setPhoneNumber(e.target.value)} placeholder="254712345678" className="w-full p-2 border rounded" required />
                                </div>
                                {error && (<p className="text-red-500 text-sm mb-4">{error}</p>)}
                                <button type="submit" className="w-full bg-green-500 text-white py-2 rounded hover:bg-green-600">Pay with M-Pesa</button>
                            </form>
                        )}

                        {method === 'card' && (
                            <Elements stripe={stripePromise}>
                                <CardForm 
                                    amount={amount} 
                                    onPaid={() => onSuccess()} 
                                    setError={setError} 
                                    setLoading={setLoading}
                                    prediction={prediction}
                                    onPaymentSuccess={(paymentId) => {
                                        // Handle successful card payment with redirect
                                        if (onSuccess) onSuccess();
                                        setTimeout(() => {
                                            if (prediction) {
                                                navigate('/prediction-details', {
                                                    state: {
                                                        prediction: prediction,
                                                        fromPayment: true,
                                                        paymentReference: paymentId,
                                                        showInvoice: true
                                                    }
                                                });
                                            }
                                            onClose();
                                        }, 1500);
                                    }}
                                />
                            </Elements>
                        )}
                    </div>
                )}

                {loading && !paymentStatus && (
                    <div className="text-center">
                        {paymentStatus === 'initiated' ? (
                            <div>
                                <p className="mb-2">Payment initiated. Check your phone for the M-Pesa prompt.</p>
                                <p className="text-sm text-gray-600">If you don't receive a prompt within 2 minutes, try retrying or contact support.</p>
                                <div className="flex gap-2 justify-center mt-3">
                                    <button onClick={initiatePayment} className="px-3 py-1 bg-yellow-500 text-white rounded">Retry</button>
                                    <a href="mailto:support@example.com" className="px-3 py-1 bg-gray-200 rounded">Contact Support</a>
                                </div>
                            </div>
                        ) : (
                            <div>
                                <p className="mb-2">Check your phone for M-Pesa prompt</p>
                                <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-gray-900 mx-auto"></div>
                            </div>
                        )}
                    </div>
                )}

                {paymentStatus === 'completed' && (
                    <div className="text-center text-green-500">
                        <p>Payment completed successfully!</p>
                        <p>Redirecting to invoice...</p>
                        <button
                            onClick={() => {
                                if (prediction) {
                                    navigate('/prediction-details', {
                                        state: {
                                            prediction: prediction,
                                            fromPayment: true,
                                            showInvoice: true
                                        }
                                    });
                                }
                                onClose();
                            }}
                            className="mt-4 bg-green-500 text-white py-2 px-4 rounded"
                        >
                            View Invoice
                        </button>
                    </div>
                )}

                {!loading && (
                    <button
                        onClick={onClose}
                        className="mt-4 text-gray-500 hover:text-gray-700"
                    >
                        Cancel
                    </button>
                )}

                {paymentStatus === 'paybill_fallback' && paybillFallback && (
                    <div className="mt-6 p-4 bg-blue-50 border-l-4 border-blue-500 rounded">
                        <h3 className="font-bold text-blue-900 mb-3">📱 Pay Manually via M-Pesa PayBill</h3>
                        
                        <div className="bg-white p-3 rounded mb-4">
                            <p className="text-sm font-semibold mb-2">Payment Details:</p>
                            <div className="text-sm space-y-1">
                                <p><span className="font-semibold">PayBill Number:</span> {paybillFallback.paybill}</p>
                                <p><span className="font-semibold">Account Reference:</span> {paybillFallback.accountReference}</p>
                                <p><span className="font-semibold">Amount:</span> KES {paybillFallback.amount}</p>
                            </div>
                        </div>

                        <div className="bg-white p-3 rounded mb-4">
                            <p className="text-sm font-semibold mb-3">Step-by-Step Instructions:</p>
                            <ol className="text-sm space-y-2 ml-4">
                                {paybillFallback.stepByStep && paybillFallback.stepByStep.map((step, index) => (
                                    <li key={index} className="list-decimal">
                                        {step}
                                    </li>
                                ))}
                            </ol>
                        </div>

                        <div className="bg-yellow-50 p-3 rounded mb-4 border border-yellow-200">
                            <p className="text-sm"><strong>⚠️ Important:</strong> After completing the M-Pesa payment, you will receive a confirmation message with a code. Enter that code below to verify your payment.</p>
                        </div>

                        <div className="mb-4">
                            <label className="block text-sm font-semibold mb-2">M-Pesa Confirmation Code:</label>
                            <input 
                                type="text" 
                                value={mpesaCode} 
                                onChange={(e) => setMpesaCode(e.target.value.toUpperCase())} 
                                placeholder="e.g., SAP19BLV5E"
                                maxLength="10"
                                className="w-full p-2 border border-gray-300 rounded text-center font-mono text-lg"
                            />
                            <p className="text-xs text-gray-600 mt-1">The code appears in your M-Pesa confirmation message</p>
                        </div>

                        <button 
                            onClick={async () => {
                                if (!mpesaCode) {
                                    setError('Please enter the M-Pesa confirmation code');
                                    return;
                                }
                                // Call verification endpoint
                                try {
                                    const response = await axios.post('http://localhost:5000/api/paybill/verify', {
                                        reference: paybillFallback.reference || 'manual-paybill',
                                        amount: paybillFallback.amount,
                                        mpesaCode: mpesaCode
                                    });
                                    if (response.data.success) {
                                        setPaymentStatus('completed');
                                        if (onSuccess) onSuccess();
                                    } else {
                                        setError('Code verification failed: ' + response.data.message);
                                    }
                                } catch (err) {
                                    setError('Verification failed: ' + (err.response?.data?.message || err.message));
                                }
                            }}
                            className="w-full bg-green-500 text-white py-2 rounded hover:bg-green-600 font-semibold"
                        >
                            Verify Payment
                        </button>

                        <div className="mt-3 flex gap-2">
                            <button 
                                onClick={() => setPaymentStatus(null)}
                                className="flex-1 px-3 py-1 bg-gray-300 text-gray-700 rounded hover:bg-gray-400 text-sm"
                            >
                                Back
                            </button>
                            <button 
                                onClick={() => {
                                    setMpesaCode('');
                                    setPaymentStatus(null);
                                    setPaybillFallback(null);
                                    setError(null);
                                }}
                                className="flex-1 px-3 py-1 bg-red-300 text-red-700 rounded hover:bg-red-400 text-sm"
                            >
                                Cancel
                            </button>
                        </div>
                    </div>
                )}
            </div>
        </div>
    );
};

export default PaymentModal;