import React, { useState, useEffect, useContext, useCallback } from 'react';
import { useNavigate } from 'react-router-dom';
import { AuthContext } from '../context/auth-context';
import axios from 'axios';
import jsPDF from 'jspdf';
import './PaymentHistory.css';

const PaymentHistory = () => {
  const { user } = useContext(AuthContext);
  const navigate = useNavigate();
  const [payments, setPayments] = useState([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);
  const [filter, setFilter] = useState('all'); // all, completed, pending, failed
  const [sortBy, setSortBy] = useState('date-desc'); // date-desc, date-asc, amount-desc, amount-asc

  const fetchPaymentHistory = useCallback(async () => {
    if (!user) {
      setLoading(false);
      return;
    }

    try {
      setLoading(true);
      const response = await axios.get(
        `http://localhost:5000/api/payments/status/${user.uid}`,
        {
          headers: {
            Authorization: `Bearer ${await user.getIdToken()}`
          }
        }
      );

      // If the endpoint returns payment object, convert to array
      const paymentData = response.data.payment || response.data.payments || [];
      const paymentsArray = Array.isArray(paymentData) ? paymentData : [paymentData];
      setPayments(paymentsArray);
      setError(null);
    } catch (err) {
      console.error('Error fetching payment history:', err);
      setError('Failed to load payment history. Please try again later.');
      setPayments([]);
    } finally {
      setLoading(false);
    }
  }, [user]);

  useEffect(() => {
    fetchPaymentHistory();
  }, [fetchPaymentHistory]);

  // Filter payments based on status
  const filterPayments = () => {
    let filtered = [...payments];

    if (filter !== 'all') {
      filtered = filtered.filter(p => p.status === filter);
    }

    // Sort payments
    if (sortBy === 'date-desc') {
      filtered.sort((a, b) => new Date(b.createdAt) - new Date(a.createdAt));
    } else if (sortBy === 'date-asc') {
      filtered.sort((a, b) => new Date(a.createdAt) - new Date(b.createdAt));
    } else if (sortBy === 'amount-desc') {
      filtered.sort((a, b) => (b.amount || 0) - (a.amount || 0));
    } else if (sortBy === 'amount-asc') {
      filtered.sort((a, b) => (a.amount || 0) - (b.amount || 0));
    }

    return filtered;
  };

  const viewInvoice = (payment) => {
    // Build a prediction object from payment data for display on prediction-details page
    const prediction = {
      predictionId: payment.reference,
      predictionName: `Invoice #${payment.reference}`,
      amount: payment.amount,
      status: payment.status,
      createdAt: payment.createdAt,
      price: payment.amount,
      selectedOdds: [], // Payment history doesn't store odds, but we can show in invoice
      numGames: 0,
      totalOdds: 1,
      paymentMethod: payment.provider === 'stripe' ? 'Card' : 'M-Pesa'
    };

    // Navigate to prediction-details page with invoice display
    navigate('/prediction-details', {
      state: {
        prediction: prediction,
        fromPayment: true,
        paymentReference: payment.reference,
        showInvoice: true,
        paymentData: payment // Pass full payment data for invoice display
      }
    });
  };

  const downloadInvoicePDF = (payment) => {
    try {
      const doc = new jsPDF();
      const pageWidth = doc.internal.pageSize.getWidth();
      const pageHeight = doc.internal.pageSize.getHeight();

      // Header
      doc.setFontSize(20);
      doc.text('INVOICE', pageWidth / 2, 20, { align: 'center' });

      // Company info
      doc.setFontSize(10);
      doc.text('Predict AI', 20, 35);
      doc.text('Payment Receipt', 20, 42);

      // Invoice details
      doc.setFontSize(9);
      const invoiceY = 55;
      doc.text(`Invoice #: ${payment.reference || 'N/A'}`, 20, invoiceY);
      doc.text(`Date: ${new Date(payment.createdAt).toLocaleDateString()}`, 20, invoiceY + 7);
      doc.text(`Status: ${payment.status?.toUpperCase()}`, 20, invoiceY + 14);

      // Customer info
      doc.text('BILL TO:', 20, invoiceY + 28);
      doc.text(`User ID: ${payment.userId || 'N/A'}`, 20, invoiceY + 35);
      doc.text(`Phone: ${payment.phoneNumber || payment.rawPhone || 'N/A'}`, 20, invoiceY + 42);

      // Items table
      const tableY = invoiceY + 56;
      doc.setFillColor(200, 200, 200);
      doc.rect(20, tableY, pageWidth - 40, 8, 'F');
      doc.text('Description', 25, tableY + 6);
      doc.text('Amount (KES)', pageWidth - 35, tableY + 6, { align: 'right' });

      // Item
      doc.setFillColor(255, 255, 255);
      doc.rect(20, tableY + 8, pageWidth - 40, 8);
      doc.text('Prediction Access Fee', 25, tableY + 14);
      doc.text(`${payment.amount || 0}`, pageWidth - 25, tableY + 14, { align: 'right' });

      // Total
      doc.setFontSize(11);
      doc.setFont(undefined, 'bold');
      doc.rect(20, tableY + 16, pageWidth - 40, 10, 'F');
      doc.text('TOTAL:', 25, tableY + 24);
      doc.text(`KES ${payment.amount || 0}`, pageWidth - 25, tableY + 24, { align: 'right' });

      // Footer
      doc.setFontSize(8);
      doc.setFont(undefined, 'normal');
      const footerY = pageHeight - 20;
      doc.text('Thank you for your payment!', pageWidth / 2, footerY, { align: 'center' });
      doc.text('For support, contact: support@predictai.com', pageWidth / 2, footerY + 7, { align: 'center' });

      // Save PDF
      const fileName = `invoice-${payment.reference || 'payment'}-${new Date().getTime()}.pdf`;
      doc.save(fileName);
    } catch (err) {
      console.error('Error generating PDF:', err);
      alert('Failed to generate invoice PDF');
    }
  };

  const getStatusBadgeClass = (status) => {
    switch (status) {
      case 'completed':
        return 'badge-success';
      case 'pending':
        return 'badge-warning';
      case 'failed':
        return 'badge-danger';
      default:
        return 'badge-secondary';
    }
  };

  const getPaymentMethodDisplay = (payment) => {
    if (payment.provider === 'stripe') return 'Card (Stripe)';
    if (payment.phoneNumber || payment.rawPhone) return 'M-Pesa';
    if (payment.checkoutRequestId) return 'M-Pesa STK';
    return 'Unknown';
  };

  const filteredPayments = filterPayments();

  if (!user) {
    return (
      <div className="payment-history-container">
        <div className="alert alert-info">
          Please log in to view your payment history.
        </div>
      </div>
    );
  }

  return (
    <div className="payment-history-container">
      <div className="payment-history-header">
        <h1>📊 Payment History & Invoices</h1>
        <p>View and manage all your past payments</p>
      </div>

      {error && (
        <div className="alert alert-danger">
          {error}
          <button className="btn-close" onClick={() => setError(null)}>×</button>
        </div>
      )}

      <div className="filters-section">
        <div className="filter-group">
          <label>Filter by Status:</label>
          <select value={filter} onChange={(e) => setFilter(e.target.value)}>
            <option value="all">All Payments</option>
            <option value="completed">✓ Completed</option>
            <option value="pending">⏳ Pending</option>
            <option value="failed">✗ Failed</option>
          </select>
        </div>

        <div className="filter-group">
          <label>Sort by:</label>
          <select value={sortBy} onChange={(e) => setSortBy(e.target.value)}>
            <option value="date-desc">Newest First</option>
            <option value="date-asc">Oldest First</option>
            <option value="amount-desc">Highest Amount</option>
            <option value="amount-asc">Lowest Amount</option>
          </select>
        </div>

        <button className="btn-refresh" onClick={fetchPaymentHistory}>
          🔄 Refresh
        </button>
      </div>

      {loading ? (
        <div className="loading-spinner">
          <p>Loading your payments...</p>
        </div>
      ) : filteredPayments.length === 0 ? (
        <div className="empty-state">
          <p>📭 No payments found</p>
          <small>Your payment history will appear here once you make a purchase.</small>
        </div>
      ) : (
        <div className="payments-table-wrapper">
          <table className="payments-table">
            <thead>
              <tr>
                <th>Date</th>
                <th>Reference</th>
                <th>Amount (KES)</th>
                <th>Method</th>
                <th>Status</th>
                <th>Actions</th>
              </tr>
            </thead>
            <tbody>
              {filteredPayments.map((payment) => (
                <tr key={payment.reference || payment.paymentIntentId}>
                  <td>{new Date(payment.createdAt).toLocaleDateString()} {new Date(payment.createdAt).toLocaleTimeString()}</td>
                  <td className="reference-cell">
                    <code>{payment.reference?.substring(0, 20)}...</code>
                  </td>
                  <td className="amount-cell">
                    <strong>KES {payment.amount || 0}</strong>
                  </td>
                  <td>{getPaymentMethodDisplay(payment)}</td>
                  <td>
                    <span className={`badge ${getStatusBadgeClass(payment.status)}`}>
                      {payment.status?.charAt(0).toUpperCase() + payment.status?.slice(1)}
                    </span>
                  </td>
                  <td className="actions-cell">
                    <button
                      className="btn-action btn-download"
                      onClick={() => downloadInvoicePDF(payment)}
                      title="Download Invoice PDF"
                    >
                      📥 Download
                    </button>
                    {payment.status === 'completed' && (
                      <button
                        className="btn-action btn-view"
                        onClick={() => viewInvoice(payment)}
                        title="View Details"
                      >
                        👁️ View
                      </button>
                    )}
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>
      )}

      <div className="payment-history-footer">
        <p>Total Payments: <strong>{payments.length}</strong></p>
        <p>Total Spent: <strong>KES {payments.reduce((sum, p) => sum + (p.amount || 0), 0)}</strong></p>
      </div>
    </div>
  );
};

export default PaymentHistory;
