import React, { useState } from 'react';

const PaymentForm = () => {
  const [formData, setFormData] = useState({
    phone: '',
    accountNumber: '',
    amount: ''
  });
  const [message, setMessage] = useState('');
  const [isLoading, setIsLoading] = useState(false);

  const handleChange = (e) => {
    const { id, value } = e.target;
    setFormData(prevData => ({
      ...prevData,
      [id]: value
    }));
  };

  const validateForm = () => {
    if (!formData.phone) return 'Phone number is required';
    if (!formData.phone.match(/^(?:254|\+254|0)?([71](?:(?:0[0-8])|(?:[12][0-9])|(?:9[0-9])|(?:4[0-3])|(?:4[5-9])|(?:5[7-9])|(?:6[8-9])[0-9])[0-9]{6})$/)) {
      return 'Please enter a valid Safaricom number';
    }
    if (!formData.accountNumber) return 'Account number is required';
    if (!formData.amount) return 'Amount is required';
    if (formData.amount < 1) return 'Amount must be greater than 0';
    return null;
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    
    // Form validation
    const validationError = validateForm();
    if (validationError) {
      setMessage(validationError);
      return;
    }

    setIsLoading(true);
    setMessage('Processing payment...');
    
    try {
      // Format phone number to include country code if needed
      let phoneNumber = formData.phone;
      if (phoneNumber.startsWith('0')) {
        phoneNumber = '254' + phoneNumber.slice(1);
      }

      const response = await fetch("http://localhost:5000/api/stkpush", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify({
          phone: phoneNumber,
          accountNumber: formData.accountNumber,
          amount: parseFloat(formData.amount)
        }),
      });

      const data = await response.json();
      
      if (!response.ok) {
        throw new Error(data.message || 'Payment request failed');
      }

      setMessage('Payment request sent successfully! Check your phone for the STK push.');
      // Clear form after successful submission
      setFormData({ phone: '', accountNumber: '', amount: '' });
      
    } catch (error) {
      console.error('Error:', error);
      setMessage(error.message || 'Payment failed! Please try again.');
    } finally {
      setIsLoading(false);
    }
  };

  return (
    <div style={styles.container}>
      <form onSubmit={handleSubmit} style={styles.form}>
        <h1 style={styles.title}>M-Pesa Payment</h1>
        
        {message && (
          <div style={{
            ...styles.message,
            backgroundColor: message.includes('failed') || message.includes('valid') ? '#ffebee' : '#e8f5e9',
            color: message.includes('failed') || message.includes('valid') ? '#c62828' : '#2e7d32'
          }}>
            {message}
          </div>
        )}

        <div style={styles.formGroup}>
          <label htmlFor="phone" style={styles.label}>Phone Number</label>
          <input
            type="tel"
            id="phone"
            placeholder="e.g., 0712345678"
            value={formData.phone}
            onChange={handleChange}
            style={styles.input}
            required
          />
        </div>

        <div style={styles.formGroup}>
          <label htmlFor="accountNumber" style={styles.label}>Account Number</label>
          <input
            type="text"
            id="accountNumber"
            placeholder="Enter account number"
            value={formData.accountNumber}
            onChange={handleChange}
            style={styles.input}
            required
          />
        </div>

        <div style={styles.formGroup}>
          <label htmlFor="amount" style={styles.label}>Amount (KES)</label>
          <input
            type="number"
            id="amount"
            placeholder="Enter amount"
            value={formData.amount}
            onChange={handleChange}
            min="1"
            style={styles.input}
            required
          />
        </div>

        <button 
          type="submit" 
          style={styles.button} 
          disabled={isLoading}
        >
          {isLoading ? 'Processing...' : 'Pay Now'}
        </button>
      </form>
    </div>
  );
};

const styles = {
  container: {
    display: 'flex',
    justifyContent: 'center',
    alignItems: 'center',
    minHeight: '100vh',
    backgroundColor: '#f5f5f5',
    padding: '20px',
  },
  form: {
    width: '100%',
    maxWidth: '400px',
    backgroundColor: '#ffffff',
    padding: '2rem',
    borderRadius: '10px',
    boxShadow: '0 4px 6px rgba(0, 0, 0, 0.1)',
  },
  title: {
    textAlign: 'center',
    color: '#333',
    marginBottom: '1.5rem',
    fontSize: '1.8rem',
    fontWeight: '600',
  },
  formGroup: {
    marginBottom: '1.5rem',
  },
  label: {
    display: 'block',
    marginBottom: '0.5rem',
    color: '#555',
    fontSize: '0.9rem',
    fontWeight: '500',
  },
  input: {
    width: '100%',
    padding: '0.75rem',
    fontSize: '1rem',
    border: '1px solid #ddd',
    borderRadius: '4px',
    transition: 'border-color 0.2s ease',
    outline: 'none',
    '&:focus': {
      borderColor: '#4CAF50',
    }
  },
  button: {
    width: '100%',
    padding: '0.75rem',
    fontSize: '1rem',
    fontWeight: '600',
    color: '#fff',
    backgroundColor: '#4CAF50',
    border: 'none',
    borderRadius: '4px',
    cursor: 'pointer',
    transition: 'background-color 0.2s ease',
    '&:hover': {
      backgroundColor: '#43A047',
    },
    '&:disabled': {
      backgroundColor: '#9E9E9E',
      cursor: 'not-allowed',
    }
  },
  message: {
    padding: '0.75rem',
    marginBottom: '1rem',
    borderRadius: '4px',
    textAlign: 'center',
    fontSize: '0.9rem',
  }
};

export default PaymentForm;