import React, { useState, useEffect, useContext } from 'react';
import jsPDF from 'jspdf';
import '../Invoice/Invoice.css';
import { AuthContext } from '../../context/auth-context';
import { UI_BASE, CONTROL_BASE } from '../../config/smsAdbConfig';
import SpinnerModal from '../SpinnerModal/SpinnerModal';

// This file is a cleaned-up Invoice component moved into Invoice/ folder to avoid patch chaos.

const Invoice = ({ prediction, customer, onSmsCopied, onCustomerUpdate }) => {
  const [smsString, setSmsString] = useState(null);
  const [copied, setCopied] = useState(false);
  const [error, setError] = useState(null);
  const [report, setReport] = useState(null);

  const { user } = useContext(AuthContext);
  const predictionPrice = prediction?.price || 10;

  const defaultCustomer = customer || { name: user?.displayName || (user?.email || '').split('@')[0], contact: user?.email || '' };
  const [customerInfoState, setCustomerInfoState] = useState(defaultCustomer);

  useEffect(() => {
    try {
      if (customer) {
        const profile = { displayName: customer.name, contact: customer.contact };
        const expiresAt = Date.now() + 130 * 60 * 1000;
        localStorage.setItem('user_profile', JSON.stringify({ profile, expiresAt }));
        setCustomerInfoState({ name: customer.name, contact: customer.contact });
      } else {
        const rawProfile = localStorage.getItem('user_profile');
        if (rawProfile) {
          const parsed = JSON.parse(rawProfile);
          if (parsed.expiresAt && Date.now() < parsed.expiresAt && parsed.profile) {
            setCustomerInfoState(prev => ({ ...prev, name: parsed.profile.displayName || prev.name, contact: parsed.profile.contact || prev.contact }));
            return;
          }
          localStorage.removeItem('user_profile');
        }
        (async () => {
          try {
            const token = await user?.getIdToken();
            if (!token) return;
            const res = await fetch(`${UI_BASE}/api/profile`, { headers: { Authorization: `Bearer ${token}` } });
            if (!res.ok) return;
            const json = await res.json();
            if (json.profile) setCustomerInfoState(prev => ({ ...prev, name: json.profile.displayName || prev.name, contact: json.profile.contact || prev.contact }));
          } catch (e) {
            // ignore
          }
        })();
      }
    } catch (e) {
      setCustomerInfoState(defaultCustomer);
    }
  }, [user, customer]);

  useEffect(() => {
    if (!prediction) return;
    try {
      const stored = JSON.parse(localStorage.getItem('predictionCopyReports') || '{}');
      const id = prediction.predictionId || prediction._localId;
      if (stored && id && stored[id]) setReport(stored[id]);
    } catch (e) {
      // ignore
    }
  }, [prediction]);

  useEffect(() => {
    const fetchSmsBet = async () => {
      try {
        const response = await fetch(`${UI_BASE}/api/createSmsBets`, {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({ predictions: [prediction], stake: 1 })
        });
        if (!response.ok) throw new Error(`Failed to fetch SMS bet: ${response.status}`);
        const data = await response.json();
        if (data.smsBets && data.smsBets.length > 0) setSmsString(data.smsBets[0].smsString);
        else throw new Error('No SMS bet returned from the server.');
      } catch (err) {
        console.error('Error fetching SMS bet:', err);
        setError(err.message);
      }
    };
    if (prediction) fetchSmsBet();
  }, [prediction]);

  const persistReport = (reportObj) => {
    try {
      const id = prediction && (prediction.predictionId || prediction._localId);
      if (!id) return;
      const stored = JSON.parse(localStorage.getItem('predictionCopyReports') || '{}');
      stored[id] = reportObj;
      localStorage.setItem('predictionCopyReports', JSON.stringify(stored));
    } catch (e) {
      console.warn('Failed to persist report', e);
    }
  };

  const downloadPDF = () => {
    const doc = new jsPDF({ unit: 'pt', format: 'a4' });
    const pageWidth = doc.internal.pageSize.getWidth();
    const margin = 36;
    const maxWidth = pageWidth - margin * 2;
    const parts = [];
    parts.push('Invoice');
    parts.push(`Invoice ID: ${prediction.predictionName}`);
    parts.push('');
    parts.push('Customer Details:');
    parts.push(`Name: ${customerInfoState.name}`);
    parts.push(`Contact: ${customerInfoState.contact}`);
    parts.push('');
    parts.push('Bet Slip Details:');
    parts.push(`Prediction ID: ${prediction.predictionId}`);
    parts.push(`Prediction Name: ${prediction.predictionName}`);
    parts.push(`Total Odds: ${prediction.totalOdds.toFixed(2)}`);
    prediction.selectedOdds.forEach((odd, index) => {
      parts.push(`${index + 1}. ${odd.competitors && odd.competitors[0] ? odd.competitors[0].name : 'Unknown'} vs ${odd.competitors && odd.competitors[1] ? odd.competitors[1].name : 'Unknown'}`);
    });
    if (smsString) parts.push(`SMS Bet: ${smsString}`);
    doc.text(parts.join('\n'), margin, margin + 20);
    const safeName = (prediction.predictionName || 'invoice').replace(/[^a-z0-9_-]/gi, '_');
    doc.save(`${safeName}.pdf`);
  };

  const handleCopySmsBet = () => {
    if (smsString) {
      navigator.clipboard.writeText(smsString).then(() => {
        setCopied(true); setTimeout(() => setCopied(false), 2000); if (onSmsCopied) onSmsCopied();
      }).catch(err => { console.error('Failed to copy SMS bet:', err); alert('Failed to copy SMS bet.'); });
    } else alert('No SMS bet available.');
  };

  const [smsSendStatus, setSmsSendStatus] = useState(null);
  const [autoSendResult, setAutoSendResult] = useState(null);
  const [bridgeWaiting, setBridgeWaiting] = useState(false);

  const sendSmsViaLocalController = async () => {
    if (!smsString) return alert('No SMS bet available.');
    try {
      setSmsSendStatus('sending'); setAutoSendResult(null); setBridgeWaiting(true);
      // Prepare outgoing message: use the exact smsString from createSmsBets
      let outgoing = (smsString || '').trim();
      // strip common UI prefixes like 'SMS Bet:' if present
      if (outgoing.toLowerCase().startsWith('sms bet:')) {
        outgoing = outgoing.slice(8).trim();
      }
      // include Base64-encoded message to avoid shell quoting issues on host
      let message_b64 = null;
      try {
        // encode Unicode to Base64 safely
        message_b64 = window.btoa(unescape(encodeURIComponent(outgoing)));
      } catch (err) {
        try { message_b64 = window.btoa(outgoing); } catch (e) { message_b64 = null; }
      }
      const payload = { to: '79079', message: outgoing, message_b64, meta: { predictionId: prediction.predictionId }, timeoutSecs: 40, watch: true };
      // debug log so you can verify exact outgoing payload in browser console
      console.log('Sending SMS payload to bridge:', payload);
      const res = await fetch(`${CONTROL_BASE}/send`, { method: 'POST', headers: { 'Content-Type': 'application/json' }, body: JSON.stringify(payload) });
      if (!res.ok) throw new Error('Failed to send SMS via bridge');
      const json = await res.json();
      const pr = json && json.prediction_result;
      if (pr && pr.classification === 'success') {
        setSmsSendStatus('sent');
        const rep = { status: 'success', message: pr.message || 'Confirmed', timestamp: new Date().toISOString(), meta: pr };
        setReport(rep); persistReport(rep);
      } else if (pr && pr.classification === 'failure') {
        setSmsSendStatus('failed');
        const rep = { status: 'error', message: pr.message || 'Failed', timestamp: new Date().toISOString(), meta: pr };
        setReport(rep); persistReport(rep);
      } else {
        setSmsSendStatus('failed');
        const rep = { status: 'no_response', message: 'No confirmation within timeout', timestamp: new Date().toISOString(), meta: json };
        setReport(rep); persistReport(rep);
      }
    } catch (e) {
      console.error('Send SMS enqueue error', e);
      setSmsSendStatus('failed'); alert('Failed to send SMS. Is the Gradle bridge running?');
    } finally {
      setBridgeWaiting(false);
    }
  };

  const sendAutoNow = async () => { await sendSmsViaLocalController(); };

  // Developer helper: inject a simulated reply into the device via the bridge so
  // the watch flow can complete during local testing. This calls the bridge's
  // /simulate-reply endpoint which writes a prediction_result.json into the app
  // files dir so gradle-watch or the bridge polling can pick it up.
  const simulateReply = async (opts = {}) => {
    try {
      const body = {
        phone: opts.phone || (customerInfoState && customerInfoState.contact) || '0701145798',
        classification: opts.classification || 'success',
        message: opts.message || 'Simulated confirmation',
      };
      const res = await fetch(`${CONTROL_BASE}/simulate-reply`, { method: 'POST', headers: { 'Content-Type': 'application/json' }, body: JSON.stringify(body) });
      if (!res.ok) {
        const txt = await res.text().catch(() => '');
        throw new Error(`simulate-reply failed: ${res.status} ${txt}`);
      }
      const j = await res.json();
      console.log('Simulate reply result', j);
      alert('Simulated reply injected. If a gradle watch is running it should detect the result shortly.');
    } catch (e) {
      console.error('Failed to inject simulated reply', e);
      alert('Failed to inject simulated reply: ' + (e && e.message ? e.message : String(e)));
    }
  };

  const handleConfirmSuccess = () => {
    const rep = { status: 'success', message: smsString || 'User confirmed success', timestamp: new Date().toISOString() };
    setReport(rep); persistReport(rep);
    (async () => {
      try {
        const token = await user?.getIdToken();
        if (token) {
          const body = { prediction, amount: predictionPrice, paymentMethod: 'mpesa', metadata: { smsString } };
          const res = await fetch(`${UI_BASE}/api/orders`, { method: 'POST', headers: { 'Content-Type': 'application/json', Authorization: `Bearer ${token}` }, body: JSON.stringify(body) });
          if (res.ok) {
            const json = await res.json(); setReport(prev => ({ ...prev, order: json.order }));
            const orderId = json.order && json.order.orderId;
            if (orderId) {
              try { await fetch(`${UI_BASE}/api/mpesaPayment`, { method: 'POST', headers: { 'Content-Type': 'application/json' }, body: JSON.stringify({ phoneNumber: customerInfoState.contact, amount: predictionPrice, accountReference: orderId, transactionDesc: `Payment for ${prediction.predictionName}` }) }); } catch (mpErr) { console.warn('MPesa initiation failed', mpErr); }
            }
          }
        }
      } catch (e) { console.warn('Failed to create order', e); }
    })();
    alert('Recorded success for this prediction.');
  };

  const handleReportError = () => {
    const msg = window.prompt('Paste SportPesa error message (or short description):');
    if (msg === null) return;
    const rep = { status: 'error', message: msg || 'Reported error (no message provided)', timestamp: new Date().toISOString() };
    setReport(rep); persistReport(rep); alert('Recorded error for this prediction.');
  };

  return (
    <div className="invoice-container">
      <h1>Invoice</h1>
      <p className="divider">=======</p>
      <p><strong>Invoice ID:</strong> {prediction.predictionName}</p>
      <p><strong>Date:</strong> {new Date().toLocaleDateString()}</p>

      <h3>Customer Details:</h3>
      <p className="divider">-----------------</p>
      <p><strong>Name:</strong> {customerInfoState.name}</p>
      <p><strong>Contact:</strong> {customerInfoState.contact}</p>

      <h3>Bet Slip Details:</h3>
      <p className="divider">-----------------</p>
      <p><strong>Prediction ID:</strong> {prediction.predictionId}</p>
      <p><strong>Prediction Name:</strong> {prediction.predictionName}</p>
      <p><strong>Total Odds:</strong> {prediction.totalOdds.toFixed(2)}</p>

      <h3>Selected Odds:</h3>
      <p className="divider">--------------</p>
      <div className="odds-list">
        {prediction.selectedOdds.map((odd, index) => {
          const dateString = odd.date || '';
          const dateObj = new Date(dateString);
          const isValidDate = !isNaN(dateObj);
          const formattedDate = isValidDate ? dateObj.toLocaleDateString() : 'N/A';
          const formattedTime = isValidDate ? dateObj.toLocaleTimeString() : 'N/A';
          const competitor1Name = odd.competitors && odd.competitors[0] ? odd.competitors[0].name : 'Unknown';
          const competitor2Name = odd.competitors && odd.competitors[1] ? odd.competitors[1].name : 'Unknown';
          const marketName = odd.marketName || 'N/A';
          const selectionName = odd.selectionName || 'N/A';
          const odds = odd.odds ? odd.odds.toFixed(2) : 'N/A';

          return (
            <div key={index} className="odd-card">
              <p className="match-info">{index + 1}. {competitor1Name} vs {competitor2Name}</p>
              <p><strong>Date:</strong> {formattedDate}</p>
              <p><strong>Time:</strong> {formattedTime}</p>
              <p><strong>Market:</strong> {marketName}</p>
              <p><strong>Selection:</strong> {selectionName}</p>
              <p><strong>Odds:</strong> <span className="odds-value">{odds}</span></p>
            </div>
          );
        })}
      </div>

      {smsString ? (
        <div className="sms-bet-container">
          <h3>SMS Bet:</h3>
          <p>{smsString}</p>
          <div className="button-group">
            <button onClick={handleCopySmsBet} className={`btn-invoice btn-copy ${copied ? 'copied' : ''}`}>{copied ? 'Copied!' : 'Copy SMS Bet'}</button>
            <button onClick={sendSmsViaLocalController} className="btn-invoice btn-send" title="Send this SMS via local adb controller" disabled={bridgeWaiting}>{smsSendStatus === 'sending' ? 'Sending...' : smsSendStatus === 'sent' ? 'Sent' : smsSendStatus === 'failed' ? 'Failed' : 'Send SMS'}</button>
            <button onClick={sendAutoNow} className="btn-invoice btn-send-auto" title="Attempt immediate send + verify via local controller" style={{ marginLeft: 8 }} disabled={bridgeWaiting}>Auto-Send Now</button>
            <button onClick={handleConfirmSuccess} className="btn-invoice btn-confirm">Confirm Success</button>
              <button onClick={() => simulateReply({ classification: 'success' })} className="btn-invoice btn-simulate" style={{ marginLeft: 8 }}>Simulate Reply (success)</button>
              <button onClick={() => simulateReply({ classification: 'failure' })} className="btn-invoice btn-simulate-fail" style={{ marginLeft: 8 }}>Simulate Reply (failure)</button>
            <button onClick={handleReportError} className="btn-invoice btn-error">Report Error</button>
          </div>

          {report && (
            <div className="report-container">
              <strong>Last report:</strong>
              <div style={{ fontSize: '0.8125rem', color: report.status === 'success' ? '#0a7' : '#c00' }}>{report.status.toUpperCase()}: {report.message}</div>
              <div style={{ fontSize: '0.6875rem', color: '#666' }}>{new Date(report.timestamp).toLocaleString()}</div>
            </div>
          )}

          <SpinnerModal open={bridgeWaiting} title="Sending SMS" message="Waiting for device confirmation..." />
        </div>
      ) : error ? (
        <p style={{ color: 'red' }}>Error generating SMS bet: {error}</p>
      ) : (
        <p>Loading SMS bet...</p>
      )}

      <h3>Total Amount:</h3>
      <p style={{ fontSize: '1.125rem', fontWeight: 'bold', color: '#007bff' }}>Ksh. {predictionPrice}.00</p>
      <h3>Payment Status:</h3>
      <p>Paid</p>

      <button onClick={downloadPDF} className="btn-invoice btn-download mt-5" style={{ marginTop: '1.25rem' }}>Download PDF</button>
    </div>
  );
};

export default Invoice;
