import React, { useState, useEffect, useContext } from 'react';
import jsPDF from 'jspdf';
import './Invoice.css';
import { AuthContext } from '../context/auth-context';
import { UI_BASE, CONTROL_BASE } from '../config/smsAdbConfig';

// We removed inline edit UI. Customer info now comes from backend-synced localStorage (user_profile) or AuthContext.

const Invoice = ({ prediction, customer, onSmsCopied, onCustomerUpdate }) => {
  const [smsString, setSmsString] = useState(null);
  const [copied, setCopied] = useState(false);
  const [error, setError] = useState(null);
  const [report, setReport] = useState(null);

  const { user } = useContext(AuthContext);
  const predictionPrice = prediction?.price || 10;

  // If customer not provided, use logged-in user info
  const defaultCustomer = customer || { name: user?.displayName || (user?.email || '').split('@')[0], contact: user?.email || '' };

  // Load user profile from backend-synced localStorage key 'user_profile' (with expiry)
  const [customerInfoState, setCustomerInfoState] = useState(defaultCustomer);
  useEffect(() => {
    // If a customer prop was explicitly passed (e.g., from PredictionDetails), prefer it and persist
    if (customer) {
      const profile = { displayName: customer.name, contact: customer.contact };
      const expiresAt = Date.now() + 130 * 60 * 1000; // 130 minutes
      try {
        localStorage.setItem('user_profile', JSON.stringify({ profile, expiresAt }));
      } catch (e) {
        /* ignore */
      }
      setCustomerInfoState({ name: customer.name, contact: customer.contact });
      // try saving to backend profile as well
      (async () => {
        try {
          const token = await user?.getIdToken();
          if (!token) return;
          await fetch(`${UI_BASE}/api/profile`, { method: 'POST', headers: { 'Content-Type': 'application/json', Authorization: `Bearer ${token}` }, body: JSON.stringify({ name: customer.name, contact: customer.contact }) });
        } catch (e) {
          // ignore
        }
      })();
      return;
    }
    try {
      const rawProfile = localStorage.getItem('user_profile');
      if (rawProfile) {
        const parsed = JSON.parse(rawProfile);
        if (parsed.expiresAt && Date.now() < parsed.expiresAt && parsed.profile) {
          setCustomerInfoState(prev => ({ ...prev, name: parsed.profile.displayName || prev.name, contact: parsed.profile.contact || prev.contact }));
          return;
        }
        // expired: remove it
        localStorage.removeItem('user_profile');
      }
      // fallback to backend fetch if authenticated
      (async () => {
        try {
          const token = await user?.getIdToken();
          if (!token) return;
          const res = await fetch(`${UI_BASE}/api/profile`, { headers: { Authorization: `Bearer ${token}` } });
          if (!res.ok) return;
          const json = await res.json();
          if (json.profile) setCustomerInfoState(prev => ({ ...prev, name: json.profile.displayName || prev.name, contact: json.profile.contact || prev.contact }));
        } catch (e) {
          // ignore
        }
      })();
    } catch (e) {
      setCustomerInfoState(defaultCustomer);
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [user, customer]);

  // No edit controls: profile is read from localStorage (backend-synced) or fallback to AuthContext

  useEffect(() => {
    if (!prediction) return;
    try {
      const stored = JSON.parse(localStorage.getItem('predictionCopyReports') || '{}');
      const id = prediction.predictionId || prediction._localId;
      if (stored && id && stored[id]) setReport(stored[id]);
    } catch (e) {
      // ignore parse errors
    }
  }, [prediction]);

  useEffect(() => {
    const fetchSmsBet = async () => {
      try {
        // Check if SMS string is already provided
        if (prediction.smsString) {
          setSmsString(prediction.smsString);
          return;
        }
        
        // Use the backend API to properly create SMS bets
        // This handles conversion from game IDs to SMS IDs
        // Prefer sending the original raw prediction object when available so backend can use any saved raw metadata
        // Otherwise, attempt to enhance the prediction locally so selection tokens like 'gameId#selectionId' are converted
        // into friendly competitor names and marketName '1X2' -> '3 Way'. This avoids changing the backend.
        const enhancePredictionWithFriendlyNames = (pred) => {
          if (!pred) return pred;
          // Work on a shallow clone
          const p = JSON.parse(JSON.stringify(pred));
          const games = Array.isArray(p.selectedOdds) ? p.selectedOdds : (p.games || []);
          const compsRoot = Array.isArray(p.competitors) ? p.competitors : [];
          p.selectedOdds = (games || []).map((g) => {
            const o = { ...g };
            // Normalize marketName '1X2' variants to '3 Way'
            if (o.marketName && String(o.marketName).toLowerCase().replace(/\s+/g, '') === '1x2') o.marketName = '3 Way';
            const sel = String(o.selectionName || '');
            const smsToken = sel.match(/^(\d+)#(\d+)$/);
            const comps = Array.isArray(o.competitors) && o.competitors.length ? o.competitors : (Array.isArray(p.competitors) ? p.competitors : compsRoot);
            if (smsToken) {
              const selId = smsToken[2];
              let matched = null;
              for (const c of comps || []) {
                if (!c) continue;
                if (String(c.id) === selId || String(c.selectionId) === selId || String(c.smsId) === selId || String(c.ID) === selId) { matched = c; break; }
              }
              if (matched && matched.name) o.selectionName = matched.name;
              else if (comps && comps.length >= 2) {
                if (selId === '1') o.selectionName = comps[0].name || 'Home';
                else if (selId === '2') o.selectionName = comps[1].name || 'Away';
                else o.selectionName = comps[0].name || 'Home';
              }
            } else {
              const normalizedSel = sel.trim().toUpperCase();
              if (['1','2','X'].includes(normalizedSel) && comps && comps.length >= 2) {
                if (normalizedSel === '1') o.selectionName = comps[0].name || o.selectionName;
                else if (normalizedSel === '2') o.selectionName = comps[1].name || o.selectionName;
                else o.selectionName = 'Draw';
              }
            }
            return o;
          });
          return p;
        };

        const payloadPrediction = prediction._rawPrediction || enhancePredictionWithFriendlyNames(prediction) || prediction;

        const response = await fetch(`${UI_BASE}/api/createSmsBets`, {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
          },
          body: JSON.stringify({ predictions: [payloadPrediction], stake: 1 }),
        });
        
        if (!response.ok) {
          throw new Error(`Failed to fetch SMS bet: ${response.status}`);
        }
        
        const data = await response.json();
        
        if (data.smsBets && data.smsBets.length > 0) {
          setSmsString(data.smsBets[0].smsString);
        } else {
          // Check for specific errors
          const errs = (data && data.errors) || [];
          const startedErr = errs.find(e => e && e.note === 'smsId_not_found_or_game_started');
          if (startedErr) {
            throw new Error('One or more games have already started and cannot be SMS-bet.');
          }
          throw new Error('No SMS bet returned from the server.');
        }
      } catch (err) {
        console.error('Error fetching SMS bet:', err);
        setError(err.message);
      }
    };

    if (prediction) {
      fetchSmsBet();
    }
  }, [prediction]);

  const persistReport = (reportObj) => {
    try {
      const id = prediction && (prediction.predictionId || prediction._localId);
      if (!id) return;
      const stored = JSON.parse(localStorage.getItem('predictionCopyReports') || '{}');
      stored[id] = reportObj;
      localStorage.setItem('predictionCopyReports', JSON.stringify(stored));
    } catch (e) {
      console.warn('Failed to persist report', e);
    }
  };

  const downloadPDF = () => {
    // A4 single-page output: adapt font size so content fits on one A4 page
    const doc = new jsPDF({ unit: 'pt', format: 'a4' });

    const pageWidth = doc.internal.pageSize.getWidth();
    const pageHeight = doc.internal.pageSize.getHeight();
    const margin = 36; // pts
    const maxWidth = pageWidth - margin * 2;
    const availableHeight = pageHeight - margin * 2;

    // Compose all textual content into paragraphs
    const parts = [];
    parts.push('Invoice');
    parts.push(`Invoice ID: ${prediction.predictionName}`);
    parts.push(`Date: ${new Date().toLocaleDateString()}`);
    parts.push('');
    parts.push('Customer Details:');
    parts.push('-----------------');
    parts.push(`Name: ${customerInfoState.name}`);
    parts.push(`Contact: ${customerInfoState.contact}`);
    parts.push('');
    parts.push('Bet Slip Details:');
    parts.push('-----------------');
    parts.push(`Prediction ID: ${prediction.predictionId}`);
    parts.push(`Prediction Name: ${prediction.predictionName}`);
    parts.push(`Prediction Type: ${prediction.predictiontype}`);
    parts.push(`Number of Games: ${prediction.numGames}`);
    parts.push(`Total Odds: ${prediction.totalOdds.toFixed(2)}`);
    parts.push(`Prediction Price: Ksh. ${predictionPrice}.00`);
    parts.push('');
    parts.push('Selected Odds:');
    parts.push('--------------');

    // Handle both selectedOdds and games arrays
    const oddsArray = prediction.selectedOdds || prediction.games || [];
    if (Array.isArray(oddsArray) && oddsArray.length > 0) {
      oddsArray.forEach((odd, index) => {
        const dateString = odd.date || '';
        const dateObj = new Date(dateString);
        const isValidDate = !isNaN(dateObj);
        const formattedDate = isValidDate ? dateObj.toLocaleDateString() : 'N/A';
        const formattedTime = isValidDate ? dateObj.toLocaleTimeString() : 'N/A';
        const competitor1Name = odd.competitors && odd.competitors[0] ? odd.competitors[0].name : 'Unknown';
        const competitor2Name = odd.competitors && odd.competitors[1] ? odd.competitors[1].name : 'Unknown';
        const marketName = odd.marketName || 'N/A';
        const selectionName = odd.selectionName || 'N/A';
        const odds = odd.odds ? odd.odds.toFixed(2) : 'N/A';

        const line = `${index + 1}. Date: ${formattedDate} | Time: ${formattedTime} | Match: ${competitor1Name} vs ${competitor2Name} | Market: ${marketName} | Selection: ${selectionName} | Odds: ${odds}`;
        parts.push(line);
      });
    } else {
      parts.push('No game details available.');
    }

    if (smsString) parts.push(`SMS Bet: ${smsString}`);
    parts.push('');
    parts.push(`Total Amount: Ksh. ${predictionPrice}.00`);
    parts.push('Payment Status: Paid');

    // Start with a reasonable font size and reduce it until content fits on one page
    let fontSize = 12;
    const minFontSize = 6;
    const lineHeightFactor = 1.15; // line height multiplier

    const measureLines = (fs) => {
      doc.setFont('courier', 'normal');
      doc.setFontSize(fs);
      const allLines = [];
      for (const p of parts) {
        const split = doc.splitTextToSize(String(p), maxWidth);
        // preserve blank paragraphs
        if (split.length === 0) allLines.push('');
        else allLines.push(...split);
      }
      return allLines;
    };

    let lines = measureLines(fontSize);
    let requiredHeight = lines.length * fontSize * lineHeightFactor;
    if (requiredHeight > availableHeight) {
      // compute proportional font scaling to fit
      const scale = availableHeight / requiredHeight;
      const newFont = Math.floor(fontSize * scale);
      fontSize = Math.max(minFontSize, newFont);
      lines = measureLines(fontSize);
      requiredHeight = lines.length * fontSize * lineHeightFactor;

      // if still too big, fall back to iterative decrement
      while (requiredHeight > availableHeight && fontSize > minFontSize) {
        fontSize -= 1;
        if (fontSize < minFontSize) { fontSize = minFontSize; break; }
        lines = measureLines(fontSize);
        requiredHeight = lines.length * fontSize * lineHeightFactor;
      }
    }

    // Pagination support: split lines into pages based on availableHeight
    const lineHeight = fontSize * lineHeightFactor;
    const linesPerPage = Math.floor(availableHeight / lineHeight);
    const pages = [];
    for (let i = 0; i < lines.length; i += linesPerPage) {
      pages.push(lines.slice(i, i + linesPerPage));
    }

    const totalPages = pages.length;

    const renderHeader = (pageIndex) => {
      const headerFontSize = Math.max(10, fontSize + 2);
      doc.setFontSize(headerFontSize);
      doc.setFont('courier', 'bold');
      const title = 'Invoice';
      doc.text(title, margin, margin / 1.5 + 2);
      doc.setFont('courier', 'normal');
      doc.setFontSize(9);
      const dateStr = new Date().toLocaleString();
      doc.text(dateStr, pageWidth - margin - doc.getTextWidth(dateStr), margin / 1.5 + 2);
    };

    const renderFooter = (pageIndex) => {
      const footerFontSize = 9;
      doc.setFontSize(footerFontSize);
      doc.setFont('courier', 'normal');
      const footerText = `Page ${pageIndex + 1} of ${totalPages}`;
      const textWidth = doc.getTextWidth(footerText);
      const x = (pageWidth - textWidth) / 2;
      const y = pageHeight - margin / 2;
      doc.text(footerText, x, y);
    };

    // Render pages
    for (let p = 0; p < pages.length; p++) {
      if (p > 0) doc.addPage();
      renderHeader(p);
      // start content a bit below header
      let cursorY = margin + (fontSize + 8);
      doc.setFont('courier', 'normal');
      doc.setFontSize(fontSize);
      for (const l of pages[p]) {
        doc.text(String(l), margin, cursorY);
        cursorY += lineHeight;
      }
      renderFooter(p);
    }

    const safeName = (prediction.predictionName || 'invoice').replace(/[^a-z0-9_-]/gi, '_');
    doc.save(`${safeName}.pdf`);
  };

  const handleCopySmsBet = () => {
    if (smsString) {
      navigator.clipboard.writeText(smsString).then(() => {
        setCopied(true);
        setTimeout(() => setCopied(false), 2000);
        if (onSmsCopied) onSmsCopied();
      }).catch(err => {
        console.error('Failed to copy SMS bet:', err);
        alert('Failed to copy SMS bet.');
      });
    } else {
      alert('No SMS bet available.');
    }
  };

  const [smsSendStatus, setSmsSendStatus] = useState(null); // null | queued | sending | sent | failed
  const [smsSendId] = useState(null);
  const [autoSendResult, setAutoSendResult] = useState(null);

  const sendSmsViaLocalController = async () => {
    // Prefer the Gradle bridge one-shot /send. This will run gradlew sendSms and wait for prediction_result.
    if (!smsString) return alert('No SMS bet available.');
    try {
      setSmsSendStatus('sending');
      setAutoSendResult(null);
      // include Base64-encoded message to avoid shell quoting issues on host
      let message_b64 = null;
      try {
        message_b64 = window.btoa(unescape(encodeURIComponent(smsString)));
      } catch (err) {
        try { message_b64 = window.btoa(smsString); } catch (e) { message_b64 = null; }
      }
      const payload = { to: '79079', message: smsString, message_b64, meta: { predictionId: prediction.predictionId }, timeoutSecs: 40, watch: true };
      const res = await fetch(`${CONTROL_BASE}/send`, { method: 'POST', headers: { 'Content-Type': 'application/json' }, body: JSON.stringify(payload) });
      if (!res.ok) throw new Error('Failed to send SMS via bridge');
      const json = await res.json();
      // Map bridge prediction_result into UI
      const pr = json && json.prediction_result;
      if (pr && pr.classification === 'success') {
        setSmsSendStatus('sent');
        const rep = { status: 'success', message: pr.message || 'Confirmed', timestamp: new Date().toISOString(), meta: pr };
        setReport(rep); persistReport(rep);
      } else if (pr && pr.classification === 'failure') {
        setSmsSendStatus('failed');
        const rep = { status: 'error', message: pr.message || 'Failed', timestamp: new Date().toISOString(), meta: pr };
        setReport(rep); persistReport(rep);
      } else {
        setSmsSendStatus('failed');
        const rep = { status: 'no_response', message: 'No confirmation within timeout', timestamp: new Date().toISOString(), meta: json };
        setReport(rep); persistReport(rep);
      }
    } catch (e) {
      console.error('Send SMS enqueue error', e);
      setSmsSendStatus('failed');
      alert('Failed to send SMS. Is the Gradle bridge running?');
    }
  };

  // One-shot immediate send via controller (tries to send and verify)
  const sendAutoNow = async () => {
    // Alias to sendSmsViaLocalController for Gradle bridge one-shot behavior.
    await sendSmsViaLocalController();
  };

  const handleConfirmSuccess = () => {
    const rep = {
      status: 'success',
      message: smsString || 'User confirmed success',
      timestamp: new Date().toISOString()
    };
    setReport(rep);
    persistReport(rep);
    // Try to create an order on the backend for this purchase and kick off MPesa payment
    (async () => {
      try {
        const token = await user?.getIdToken();
        if (token) {
          const body = { prediction, amount: predictionPrice, paymentMethod: 'mpesa', metadata: { smsString } };
          const res = await fetch(`${UI_BASE}/api/orders`, { method: 'POST', headers: { 'Content-Type': 'application/json', Authorization: `Bearer ${token}` }, body: JSON.stringify(body) });
          if (res.ok) {
            const json = await res.json();
            setReport(prev => ({ ...prev, order: json.order }));
            // use the orderId as AccountReference for mpesa
            const orderId = json.order && json.order.orderId;
            if (orderId) {
              // trigger server-side mpesa initiation (server uses its mpesa config)
              try {
                await fetch(`${UI_BASE}/api/mpesaPayment`, { method: 'POST', headers: { 'Content-Type': 'application/json' }, body: JSON.stringify({ phoneNumber: customerInfoState.contact, amount: predictionPrice, accountReference: orderId, transactionDesc: `Payment for ${prediction.predictionName}` }) });
              } catch (mpErr) {
                console.warn('MPesa initiation failed', mpErr);
              }
            }
          }
        }
      } catch (e) {
        console.warn('Failed to create order', e);
      }
    })();

    alert('Recorded success for this prediction.');
  };

  const handleReportError = () => {
    const msg = window.prompt('Paste SportPesa error message (or short description):');
    if (msg === null) return;
    const rep = {
      status: 'error',
      message: msg || 'Reported error (no message provided)',
      timestamp: new Date().toISOString()
    };
    setReport(rep);
    persistReport(rep);
    alert('Recorded error for this prediction.');
  };

  return (
    <div className="invoice-container">
      <h1>Invoice</h1>
      <p className="divider">=======</p>

      <p><strong>Invoice ID:</strong> {prediction.predictionName}</p>
      <p><strong>Date:</strong> {new Date().toLocaleDateString()}</p>

  <h3>Customer Details:</h3>
  <p className="divider">-----------------</p>
  <>
    <p><strong>Name:</strong> {customerInfoState.name}</p>
    <p><strong>Contact:</strong> {customerInfoState.contact}</p>
  </>

      <h3>Bet Slip Details:</h3>
      <p className="divider">-----------------</p>
      <p><strong>Prediction ID:</strong> {prediction.predictionId}</p>
      <p><strong>Prediction Name:</strong> {prediction.predictionName}</p>
      <p><strong>Prediction Type:</strong> {prediction.predictiontype}</p>
      <p><strong>Number of Games:</strong> {prediction.numGames}</p>
      <p><strong>Total Odds:</strong> {prediction.totalOdds.toFixed(2)}</p>
      <p style={{ color: '#007bff', fontWeight: 'bold' }}>
        <strong>Prediction Price:</strong> Ksh. {predictionPrice}.00
      </p>

      <h3>Selected Odds:</h3>
      <p className="divider">--------------</p>
      <div className="odds-list">
        {(() => {
          // Handle both selectedOdds and games arrays
          const oddsArray = prediction.selectedOdds || prediction.games || [];
          if (!Array.isArray(oddsArray) || oddsArray.length === 0) {
            return <p style={{ color: '#999', fontStyle: 'italic' }}>No game details available.</p>;
          }
          
          return oddsArray.map((odd, index) => {
            const dateString = odd.date || '';
            const dateObj = new Date(dateString);
            const isValidDate = !isNaN(dateObj);
            const formattedDate = isValidDate ? dateObj.toLocaleDateString() : 'N/A';
            const formattedTime = isValidDate ? dateObj.toLocaleTimeString() : 'N/A';
            const competitor1Name = odd.competitors && odd.competitors[0] ? odd.competitors[0].name : 'Unknown';
            const competitor2Name = odd.competitors && odd.competitors[1] ? odd.competitors[1].name : 'Unknown';
            const marketName = odd.marketName || 'N/A';
            const selectionName = odd.selectionName || 'N/A';
            const odds = odd.odds ? odd.odds.toFixed(2) : 'N/A';

            return (
              <div key={index} className="odd-card">
                <p className="match-info">{index + 1}. {competitor1Name} vs {competitor2Name}</p>
                <p><strong>Date:</strong> {formattedDate}</p>
                <p><strong>Time:</strong> {formattedTime}</p>
                <p><strong>Market:</strong> {marketName}</p>
                <p><strong>Selection:</strong> {selectionName}</p>
                <p><strong>Odds:</strong> <span className="odds-value">{odds}</span></p>
              </div>
            );
          });
        })()}
      </div>

      {smsString ? (
        <div className="sms-bet-container">
          <h3>SMS Bet:</h3>
          <p>{smsString}</p>
          <div className="button-group">
            <button
              onClick={handleCopySmsBet}
              className={`btn-invoice btn-copy ${copied ? 'copied' : ''}`}
            >
              {copied ? 'Copied!' : 'Copy SMS Bet'}
            </button>

            <button
              onClick={sendSmsViaLocalController}
              className="btn-invoice btn-send"
              title="Send this SMS via local adb controller"
            >
              {smsSendStatus === 'sending' ? 'Sending...' : smsSendStatus === 'queued' ? 'Queued' : smsSendStatus === 'sent' ? 'Sent' : smsSendStatus === 'failed' ? 'Failed' : 'Send SMS'}
            </button>
            <button
              onClick={sendAutoNow}
              className="btn-invoice btn-send-auto"
              title="Attempt immediate send + verify via local controller"
              style={{ marginLeft: 8 }}
            >
              Auto-Send Now
            </button>
            {smsSendId && (
              <div style={{ fontSize: '0.75rem', color: '#666', marginTop: 6 }}>
                Queue ID: {smsSendId}
              </div>
            )}

            {autoSendResult && (
              <div style={{ marginTop: 8, fontSize: '0.85rem', color: autoSendResult.confirmed ? '#0a7' : '#c00' }}>
                Auto-send: {autoSendResult.confirmed ? 'Confirmed' : autoSendResult.error ? `Error: ${autoSendResult.error}` : 'Not confirmed'}
              </div>
            )}

            <button
              onClick={handleConfirmSuccess}
              className="btn-invoice btn-confirm"
            >
              Confirm Success
            </button>

            <button
              onClick={handleReportError}
              className="btn-invoice btn-error"
            >
              Report Error
            </button>
          </div>

          {report && (
            <div className="report-container">
              <strong>Last report:</strong>
              <div style={{ fontSize: '0.8125rem', color: report.status === 'success' ? '#0a7' : '#c00' }}>
                {report.status.toUpperCase()}: {report.message}
              </div>
              <div style={{ fontSize: '0.6875rem', color: '#666' }}>
                {new Date(report.timestamp).toLocaleString()}
              </div>
            </div>
          )}
        </div>
      ) : error ? (
        <p style={{ color: 'red' }}>Error generating SMS bet: {error}</p>
      ) : (
        <p>Loading SMS bet...</p>
      )}

      <h3>Total Amount:</h3>
      <p style={{ fontSize: '1.125rem', fontWeight: 'bold', color: '#007bff' }}>
        Ksh. {predictionPrice}.00
      </p>
      <h3>Payment Status:</h3>
      <p>Paid</p>

      <button
        onClick={downloadPDF}
        className="btn-invoice btn-download mt-5"
        style={{ marginTop: '1.25rem' }}
      >
        Download PDF
      </button>
    </div>
  );
};

export default Invoice;