import React, { useState, useContext } from 'react';
import { useNavigate } from 'react-router-dom';
import { loadStripe } from '@stripe/stripe-js';
import {
    Elements,
    CardElement,
    useStripe,
    useElements
} from '@stripe/react-stripe-js';
import { AuthContext } from '../../../../context/auth-context';
import axios from 'axios';
import '../../../Navigation/Styles.css';
import './predictionItem.css';

const stripePromise = loadStripe(process.env.REACT_APP_STRIPE_PUBLISHABLE_KEY || 'pk_test_replace_me');

const CardForm = ({ amount, onPaid, setError, setLoading, userId, prediction, navigate }) => {
    const stripe = useStripe();
    const elements = useElements();

    const handleCardPay = async (e) => {
        e.preventDefault();
        if (!stripe || !elements) return;
        setError(null);
        setLoading(true);
        try {
            const resp = await axios.post('http://localhost:5000/api/payments/create-card-payment', {
                amount,
                userId: userId || 'guest'
            });
            const clientSecret = resp.data.clientSecret;
            const cardElement = elements.getElement(CardElement);
            const result = await stripe.confirmCardPayment(clientSecret, {
                payment_method: { card: cardElement }
            });
            if (result.error) {
                setError(result.error.message);
                setLoading(false);
            } else {
                if (result.paymentIntent && result.paymentIntent.status === 'succeeded') {
                    console.log('[predictionItem CardForm] Card payment succeeded, redirecting...');
                    onPaid();
                    // Redirect to prediction-details with invoice after successful card payment
                    if (navigate && prediction) {
                        setTimeout(() => {
                            navigate('/prediction-details', {
                                state: {
                                    prediction: prediction,
                                    fromPayment: true,
                                    paymentReference: result.paymentIntent.id,
                                    showInvoice: true
                                }
                            });
                        }, 1500);
                    }
                } else {
                    setError('Payment not completed.');
                    setLoading(false);
                }
            }
        } catch (err) {
            setError(err.response?.data?.message || err.message || 'Card payment failed');
            setLoading(false);
        }
    };

    return (
        <form onSubmit={handleCardPay} className="card-payment-form">
            <div className="form-group">
                <label>Card details</label>
                <div className="card-element-container">
                    <CardElement options={{
                        style: {
                            base: {
                                fontSize: '16px',
                                color: '#424770',
                                '::placeholder': {
                                    color: '#aab7c4',
                                },
                            },
                            invalid: {
                                color: '#9e2146',
                            },
                        },
                    }} />
                </div>
            </div>
            <button type="submit" className="pay-button card" disabled={!stripe}>
                Pay with Card
            </button>
        </form>
    );
};

const PredictionItem = props => {
    const navigate = useNavigate();
    const [showPayment, setShowPayment] = useState(false);
    const [isPaid, setIsPaid] = useState(false);
    const [paymentMethod, setPaymentMethod] = useState('mpesa');
    const [phoneNumber, setPhoneNumber] = useState('');
    const [loading, setLoading] = useState(false);
    const [error, setError] = useState(null);
    const [showPaybill, setShowPaybill] = useState(false);
    const [paybillInfo, setPaybillInfo] = useState(null);
    const [mpesaCode, setMpesaCode] = useState('');
    const { user } = useContext(AuthContext);
    const amount = props.price || 100; // Allow per-prediction price via props, fallback to 100
    const previewCount = props.previewCount || 3; // how many games to show in preview

    // Build prediction/product object from props for navigation
    // Handles both predictions (predictionId, predictionName) and products (productId, name)
    const prediction = {
        predictionId: props.predictionId || props.productId,
        predictionName: props.predictionName || props.name,
        predictiontype: props.predictiontype || props.category || 'Product',
        numGames: props.numGames || 1,
        totalOdds: props.totalOdds || 1,
        selectedOdds: props.selectedOdds || [],
        price: amount,
        // Product-specific fields
        productId: props.productId,
        image: props.image,
        category: props.category,
        brand: props.brand,
        date: props.date,
        description: props.description
    };

    const pollPaymentStatus = async (reference) => {
        console.log('[predictionItem] Starting poll for ref:', reference, 'prediction:', prediction);
        const pollInterval = setInterval(async () => {
            try {
                console.log('[predictionItem] Polling status...');
                const response = await axios.get(`http://localhost:5000/api/payments/status/${reference}`);
                const { payment } = response.data;
                console.log('[predictionItem] Status:', payment.status);

                if (payment.status === 'completed') {
                    clearInterval(pollInterval);
                    console.log('[predictionItem] Payment completed! Redirecting...');
                    setIsPaid(true);
                    setLoading(false);
                    
                    // Redirect to prediction-details with invoice after successful payment
                    setTimeout(() => {
                        console.log('[predictionItem] Navigate - prediction:', prediction);
                        if (prediction) {
                            navigate('/prediction-details', {
                                state: {
                                    prediction: prediction,
                                    fromPayment: true,
                                    paymentReference: reference,
                                    showInvoice: true
                                }
                            });
                        } else {
                            console.error('[predictionItem] prediction is undefined!');
                        }
                        setShowPayment(false);
                    }, 1500);
                } else if (payment.status === 'failed') {
                    clearInterval(pollInterval);
                    setError(payment.failureReason || 'Payment failed');
                    setLoading(false);
                }
            } catch (error) {
                console.error('[predictionItem] Polling error:', error);
            }
        }, 5000);

        // Clear interval after 2 minutes
        setTimeout(() => {
            clearInterval(pollInterval);
            if (!isPaid) {
                setError('Payment timeout. Please try again');
                setLoading(false);
            }
        }, 120000);
    };

    const initiatePayment = async (e) => {
        e.preventDefault();
        setLoading(true);
        setError(null);
        setShowPaybill(false);
        setPaybillInfo(null);

        try {
            // Format phone for Daraja: convert leading 0 to 254, remove leading +
            let phone = (phoneNumber || '').toString().trim();
            if (phone.startsWith('0')) phone = '254' + phone.slice(1);
            if (phone.startsWith('+')) phone = phone.slice(1);

            const payload = {
                phone: phone,
                accountNumber: user?.uid || 'guest',
                amount: amount
            };

            console.log('[predictionItem] Initiating STK push with payload:', payload);
            const response = await axios.post('http://localhost:5000/api/stkpush', payload);
            console.log('[predictionItem] STK push response:', response.data);

            // Store PayBill fallback if provided
            if (response.data.paybillFallback) {
                setPaybillInfo(response.data.paybillFallback);
            }

            // Check if backend returned a reference (new endpoint) or ResponseCode (old endpoint)
            if (response.data && response.data.reference) {
                // New endpoint: has polling reference
                console.log('[predictionItem] Got reference:', response.data.reference);
                if (response.data.success) {
                    // Use 30 second timeout: if no response, show PayBill
                    pollPaymentStatus(response.data.reference, 30000);
                } else {
                    // STK failed, show PayBill fallback
                    setLoading(false);
                    setShowPaybill(true);
                    setError(response.data.message);
                }
            } else if (response.data && (response.data.ResponseCode === '0' || response.data.ResponseCode === 0)) {
                // Old endpoint: just shows alert
                setLoading(false);
                setShowPayment(false);
                setError(null);
                alert('STK Push sent — check your phone for the M-Pesa prompt.');
            } else {
                setError((response.data && (response.data.error || response.data.message || response.data.ResponseDescription)) || 'Payment initiation failed');
                setLoading(false);
                // Show PayBill fallback even on partial failure
                if (response.data.paybillFallback) {
                    setShowPaybill(true);
                }
            }
        } catch (error) {
            const errorMsg = error.response?.data?.message || error.response?.data?.error || error.message || 'Failed to initiate payment';
            setError(errorMsg);
            
            // Store PayBill fallback from error response if available
            if (error.response?.data?.paybillFallback) {
                setPaybillInfo(error.response.data.paybillFallback);
                setShowPaybill(true);
            }
            setLoading(false);
        }
    };
    

    const handlePaymentSuccess = () => {
        setIsPaid(true);
        setLoading(false);
        setShowPayment(false);
    };

    return (
        <li key={props.predictionId} className="prediction-item">
            <div className="prediction-content">
                <div className="prediction-header">
                    <h2>{props.predictionName}</h2>
                    <span className="prediction-type">{props.predictiontype}</span>
                </div>

                <div className="prediction-details">
                    <div className="prediction-stats">
                        <span>Games: {props.numGames}</span>
                        <span>Total Odds: {props.totalOdds}</span>
                    </div>

                    {!isPaid && (
                        <div className="prediction-preview">
                            <div className="blurred-content">
                                <ul>
                                    { (props.selectedOdds || []).slice(0, previewCount).map((odd, index) => (
                                        <li key={index} className="prediction-game-item blurred">
                                            {/* Mask teams in preview to avoid exposing sales content */}
                                            <span>{`Match ${index + 1}`}</span>
                                            <span className="game-odds">{odd.odds}</span>
                                        </li>
                                    ))}
                                </ul>
                            </div>
                            {!showPayment && (
                                <button 
                                    className="unlock-button"
                                    onClick={() => setShowPayment(true)}
                                >
                                    Unlock Prediction (KES {amount})
                                </button>
                            )}
                        </div>
                    )}

                    {isPaid && (
                        <div className="prediction-full-content">
                            <ul>
                                {(props.selectedOdds || []).map((odd, index) => (
                                    <li key={index} className="prediction-game-item">
                                        {/* After purchase show selection name and odds, but still hide competitor/team names */}
                                        <span>{odd.selectionName || `Selection ${index + 1}`}</span>
                                        <span className="game-odds">{odd.odds}</span>
                                    </li>
                                ))}
                            </ul>
                        </div>
                    )}

                    {showPayment && !isPaid && (
                        <div className="payment-section">
                            <h3>Complete Payment</h3>
                            <p>Amount: KES {amount}</p>

                            <div className="payment-methods">
                                <button 
                                    className={`method-button ${paymentMethod === 'mpesa' ? 'active' : ''}`}
                                    onClick={() => setPaymentMethod('mpesa')}
                                >
                                    M-Pesa
                                </button>
                                <button 
                                    className={`method-button ${paymentMethod === 'card' ? 'active' : ''}`}
                                    onClick={() => setPaymentMethod('card')}
                                >
                                    Card
                                </button>
                            </div>

                            {error && (
                                <div className="error-message">
                                    {error}
                                </div>
                            )}

                            {paymentMethod === 'mpesa' && (
                                <form onSubmit={initiatePayment} className="mpesa-form">
                                    <div className="form-group">
                                        <label>M-Pesa Phone Number</label>
                                        <input
                                            type="tel"
                                            value={phoneNumber}
                                            onChange={(e) => setPhoneNumber(e.target.value)}
                                            placeholder="254712345678"
                                            required
                                        />
                                    </div>
                                    <button 
                                        type="submit" 
                                        className="pay-button mpesa"
                                        disabled={loading}
                                    >
                                        {loading ? 'Processing...' : 'Pay with M-Pesa'}
                                    </button>
                                </form>
                            )}

                            {paymentMethod === 'card' && (
                                <Elements stripe={stripePromise}>
                                    <CardForm
                                        amount={amount}
                                        onPaid={handlePaymentSuccess}
                                        setError={setError}
                                        setLoading={setLoading}
                                        userId={user?.uid}
                                        prediction={prediction}
                                        navigate={navigate}
                                    />
                                </Elements>
                            )}

                            <button 
                                className="cancel-button"
                                onClick={() => {
                                    setShowPayment(false);
                                    setError(null);
                                }}
                            >
                                Cancel
                            </button>
                        </div>
                    )}

                    {loading && (
                        <div className="loading-overlay">
                            <div className="spinner"></div>
                            <p>Processing payment...</p>
                            {paymentMethod === 'mpesa' && (
                                <p>Check your phone for the M-Pesa prompt</p>
                            )}
                        </div>
                    )}

                    {showPaybill && paybillInfo && (
                        <div className="payment-section paybill-fallback">
                            <h3>📱 Pay Manually via M-Pesa PayBill</h3>
                            
                            <div className="paybill-details">
                                <h4>Payment Details:</h4>
                                <p><strong>PayBill Number:</strong> {paybillInfo.paybill}</p>
                                <p><strong>Account Reference:</strong> {paybillInfo.accountReference}</p>
                                <p><strong>Amount:</strong> KES {paybillInfo.amount}</p>
                            </div>

                            <div className="paybill-instructions">
                                <h4>Step-by-Step Instructions:</h4>
                                <ol>
                                    {paybillInfo.stepByStep && paybillInfo.stepByStep.map((step, index) => (
                                        <li key={index}>{step}</li>
                                    ))}
                                </ol>
                            </div>

                            <div className="paybill-warning">
                                <p><strong>⚠️ Important:</strong> After completing the M-Pesa payment, you will receive a confirmation message with a code. Enter that code below to verify your payment.</p>
                            </div>

                            <form onSubmit={async (e) => {
                                e.preventDefault();
                                if (!mpesaCode) {
                                    setError('Please enter the M-Pesa confirmation code');
                                    return;
                                }
                                try {
                                    const response = await axios.post('http://localhost:5000/api/paybill/verify', {
                                        reference: paybillInfo.reference || 'manual-paybill-' + Date.now(),
                                        amount: paybillInfo.amount,
                                        mpesaCode: mpesaCode
                                    });
                                    if (response.data.success) {
                                        setIsPaid(true);
                                        setShowPayment(false);
                                        setShowPaybill(false);
                                        setMpesaCode('');
                                        handlePaymentSuccess();
                                    } else {
                                        setError('Code verification failed: ' + response.data.message);
                                    }
                                } catch (err) {
                                    setError('Verification failed: ' + (err.response?.data?.message || err.message));
                                }
                            }}>
                                <div className="form-group">
                                    <label>M-Pesa Confirmation Code:</label>
                                    <input 
                                        type="text" 
                                        value={mpesaCode} 
                                        onChange={(e) => setMpesaCode(e.target.value.toUpperCase())} 
                                        placeholder="e.g., SAP19BLV5E"
                                        maxLength="10"
                                        required
                                    />
                                    <p className="help-text">The code appears in your M-Pesa confirmation message</p>
                                </div>
                                
                                <button 
                                    type="submit" 
                                    className="pay-button"
                                    disabled={loading}
                                >
                                    {loading ? 'Verifying...' : 'Verify Payment'}
                                </button>
                            </form>

                            <button 
                                className="cancel-button"
                                onClick={() => {
                                    setShowPaybill(false);
                                    setPaybillInfo(null);
                                    setMpesaCode('');
                                    setError(null);
                                }}
                            >
                                Cancel
                            </button>
                        </div>
                    )}
                </div>
            </div>
        </li>
    );
};

export default PredictionItem;