import React, { useState } from 'react';
import { loadStripe } from '@stripe/stripe-js';
import { useNavigate } from 'react-router-dom';
import {
    Elements,
    CardElement,
    useStripe,
    useElements
} from '@stripe/react-stripe-js';
import axios from 'axios';
import { API_ENDPOINTS } from '../../../../config/api';
import '../../../Navigation/Styles.css';
import './predictionItem.css';

const stripePromise = loadStripe(process.env.REACT_APP_STRIPE_PUBLISHABLE_KEY || 'pk_test_replace_me');

const CardForm = ({ amount, onPaid, setError, setLoading, userId, prediction, navigate }) => {
    const stripe = useStripe();
    const elements = useElements();

    const handleCardPay = async (e) => {
        e.preventDefault();
        if (!stripe || !elements) return;
        setError(null);
        setLoading(true);
        try {
              const resp = await axios.post(API_ENDPOINTS.CREATE_CARD_PAYMENT, {
                amount,
                userId: userId || 'guest'
            });
            const clientSecret = resp.data.clientSecret;
            const cardElement = elements.getElement(CardElement);
            const result = await stripe.confirmCardPayment(clientSecret, {
                payment_method: { card: cardElement }
            });
            if (result.error) {
                setError(result.error.message);
                setLoading(false);
            } else {
                if (result.paymentIntent && result.paymentIntent.status === 'succeeded') {
                    onPaid();
                    // Redirect to prediction-details with invoice after successful card payment
                    if (navigate && prediction) {
                        setTimeout(() => {
                            navigate('/prediction-details', {
                                state: {
                                    prediction: prediction,
                                    fromPayment: true,
                                    paymentReference: result.paymentIntent.id,
                                    showInvoice: true
                                }
                            });
                        }, 1500);
                    }
                } else {
                    setError('Payment not completed.');
                    setLoading(false);
                }
            }
        } catch (err) {
            setError(err.response?.data?.message || err.message || 'Card payment failed');
            setLoading(false);
        }
    };

    return (
        <form onSubmit={handleCardPay}>
            <div className="mb-4">
                <label className="block text-sm font-medium mb-1">Card details</label>
                <div className="p-2 border rounded">
                    <CardElement />
                </div>
            </div>
            <button type="submit" className="w-full bg-blue-600 text-white py-2 rounded">
                Pay with Card
            </button>
        </form>
    );
};

const PredictionItemWithPayment = ({ 
    predictionId, 
    predictionName, 
    predictiontype, 
    numGames, 
    totalOdds, 
    selectedOdds,
    userId
}) => {
    const [showPayment, setShowPayment] = useState(false);
    const [paymentMethod, setPaymentMethod] = useState('mpesa');
    const [phoneNumber, setPhoneNumber] = useState('');
    const [loading, setLoading] = useState(false);
    const [error, setError] = useState(null);
    const [paymentStatus, setPaymentStatus] = useState(null);
    const [amount] = useState(100); // Default amount, can be made dynamic
    const navigate = useNavigate();

    // Build prediction object from props
    const prediction = {
        predictionId,
        predictionName,
        predictiontype,
        numGames,
        totalOdds,
        selectedOdds,
        price: amount
    };

    const initiatePayment = async (e) => {
        e.preventDefault();
        setLoading(true);
        setError(null);

        try {
            // Normalize phone for Daraja
            let phone = (phoneNumber || '').toString().trim();
            if (phone.startsWith('+')) phone = phone.slice(1);
            if (phone.startsWith('0')) phone = '254' + phone.slice(1);

            const response = await axios.post(API_ENDPOINTS.STK_PUSH, {
                phone: phone,
                accountNumber: userId || 'guest',
                amount: amount
            });

            if (response.data && response.data.reference) {
                pollPaymentStatus(response.data.reference);
                setLoading(true);
            } else {
                // No internal reference returned; treat as initiated and show retry
                setPaymentStatus('initiated');
                setLoading(false);
            }
        } catch (error) {
            setError(error.response?.data?.message || 'Failed to initiate payment');
            setLoading(false);
        }
    };

    const pollPaymentStatus = async (reference) => {
        console.log('[PredictionItemWithPayment] Starting poll for ref:', reference, 'prediction:', prediction);
        const pollInterval = setInterval(async () => {
            try {
                console.log('[PredictionItemWithPayment] Polling status...');
                const response = await axios.get(`${API_ENDPOINTS.PAYMENT_STATUS}/${reference}`);
                const { payment } = response.data;
                console.log('[PredictionItemWithPayment] Status:', payment.status);

                if (payment.status === 'completed') {
                    clearInterval(pollInterval);
                    console.log('[PredictionItemWithPayment] Completed! Redirecting...');
                    setPaymentStatus('completed');
                    
                    // Redirect to prediction-details with invoice after successful payment
                    setTimeout(() => {
                        console.log('[PredictionItemWithPayment] Navigate - prediction:', prediction);
                        if (prediction) {
                            navigate('/prediction-details', {
                                state: {
                                    prediction: prediction,
                                    fromPayment: true,
                                    paymentReference: reference,
                                    showInvoice: true
                                }
                            });
                        } else {
                            console.error('[PredictionItemWithPayment] prediction is undefined!');
                        }
                    }, 1500);
                } else if (payment.status === 'failed') {
                    clearInterval(pollInterval);
                    setPaymentStatus('failed');
                    setError(payment.failureReason || 'Payment failed');
                }
            } catch (error) {
                console.error('[PredictionItemWithPayment] Polling error:', error);
            }
        }, 5000);        setTimeout(() => {
            clearInterval(pollInterval);
            if (paymentStatus !== 'completed') {
                setError('Payment timeout. Please try again');
                setLoading(false);
            }
        }, 120000);
    };

    const handlePaymentSuccess = () => {
        setPaymentStatus('completed');
        setShowPayment(false);
    };

    return (
        <li key={predictionId} className="prediction-item">
            <div className="prediction-content">
                <h2>{predictionName}</h2>
                <p>Type: {predictiontype}</p>
                <p>Number of Games: {numGames}</p>
                <p>Total Odds: {totalOdds}</p>
                <div className="games-list">
                    {selectedOdds.map((odd, index) => (
                        <div key={index} className="game-item">
                            <p>Game ID: {odd.gameId}</p>
                            <p>Odds: {odd.odds}</p>
                            {odd.competitors && (
                                <p>{odd.competitors[0]?.name} vs {odd.competitors[1]?.name}</p>
                            )}
                        </div>
                    ))}
                </div>

                {!showPayment && !paymentStatus && (
                    <button 
                        onClick={() => setShowPayment(true)}
                        className="buy-button"
                    >
                        Buy Prediction (KES {amount})
                    </button>
                )}

                {showPayment && !paymentStatus && (
                    <div className="payment-section">
                        <h3>Complete Payment</h3>
                        <p>Amount to pay: KES {amount}</p>

                        <div className="payment-methods">
                            <button 
                                className={`payment-method-btn ${paymentMethod === 'mpesa' ? 'active' : ''}`}
                                onClick={() => setPaymentMethod('mpesa')}
                            >
                                M-Pesa
                            </button>
                            <button 
                                className={`payment-method-btn ${paymentMethod === 'card' ? 'active' : ''}`}
                                onClick={() => setPaymentMethod('card')}
                            >
                                Card
                            </button>
                        </div>

                        {paymentMethod === 'mpesa' && (
                            <form onSubmit={initiatePayment}>
                                <div className="form-group">
                                    <label>M-Pesa Phone Number</label>
                                    <input
                                        type="tel"
                                        value={phoneNumber}
                                        onChange={(e) => setPhoneNumber(e.target.value)}
                                        placeholder="254712345678"
                                        required
                                    />
                                </div>
                                {error && <p className="error-message">{error}</p>}
                                <button 
                                    type="submit"
                                    className="pay-button mpesa"
                                    disabled={loading}
                                >
                                    {loading ? 'Processing...' : 'Pay with M-Pesa'}
                                </button>
                            </form>
                        )}

                        {paymentMethod === 'card' && (
                            <Elements stripe={stripePromise}>
                                <CardForm
                                    amount={amount}
                                    onPaid={handlePaymentSuccess}
                                    setError={setError}
                                    setLoading={setLoading}
                                    userId={userId}
                                    prediction={prediction}
                                    navigate={navigate}
                                />
                            </Elements>
                        )}

                        <button 
                            onClick={() => setShowPayment(false)}
                            className="cancel-button"
                        >
                            Cancel
                        </button>
                    </div>
                )}

                {loading && !paymentStatus && (
                    <div className="payment-loading">
                        <p>Processing payment...</p>
                        {paymentMethod === 'mpesa' && (
                            <p>Check your phone for M-Pesa prompt</p>
                        )}
                    </div>
                )}

                {paymentStatus === 'completed' && (
                    <div className="payment-success">
                        <p>✓ Payment completed successfully!</p>
                        <p>You can now access the full prediction details.</p>
                    </div>
                )}
            </div>
        </li>
    );
};

export default PredictionItemWithPayment;